/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.realtime.events.mail;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventHandler;
import com.openexchange.push.PushEventConstants;
import com.openexchange.realtime.events.RTEventEmitterService;
import com.openexchange.realtime.events.RTListener;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;


/**
 * The {@link MailPushEventEmitter} manages subscriptions to mail push events from the RT event system and maps the OSGi events of the Push
 * system to RT Events of the RT event system.
 * 
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class MailPushEventEmitter implements RTEventEmitterService, EventHandler {

    private static final HashSet<String> EVENTS = new HashSet<String>(Arrays.asList("new"));
    private static final String NAMESPACE = "mail";

    private final ConcurrentHashMap<String, MailPushRegistration> registeredIDsPerSession = new ConcurrentHashMap<String, MailPushRegistration>();

    public MailPushEventEmitter(ServiceLookup services) {
        super();
    }

    @Override
    public String getNamespace() {
        return NAMESPACE;
    }

    @Override
    public Set<String> getSupportedEvents() {
        return EVENTS;
    }

    @Override
    public void register(String eventName, RTListener listener) {
        MailPushRegistration mailPushRegistration = registeredIDsPerSession.get(listener.getSession().getSessionID());
        if (mailPushRegistration == null) {
            mailPushRegistration = new MailPushRegistration();
            MailPushRegistration meantime = registeredIDsPerSession.putIfAbsent(listener.getSession().getSessionID(), mailPushRegistration);
            mailPushRegistration = meantime != null ? meantime : mailPushRegistration;
        }
        mailPushRegistration.addListener(listener);

    }

    @Override
    public void unregister(String eventName, RTListener listener) {
        MailPushRegistration mailPushRegistration = registeredIDsPerSession.get(listener.getSession().getSessionID());

        if (mailPushRegistration == null) {
            return;
        }

        mailPushRegistration.removeListener(listener);

        if (mailPushRegistration.hasNoMoreListeners()) {
            registeredIDsPerSession.remove(listener.getSession().getSessionID());
        }
    }

    @Override
    public void handleEvent(Event event) {
        // Dispatch a mail push event to rt event listeners
        // There is no session available for remotely received events
        Session session = (Session) event.getProperty(PushEventConstants.PROPERTY_SESSION);
        if (null != session) {
            String folder = (String) event.getProperty(PushEventConstants.PROPERTY_FOLDER);
            MailPushRegistration mailPushRegistration = registeredIDsPerSession.get(session.getSessionID());
            if (mailPushRegistration == null) {
                return;
            }
            mailPushRegistration.triggerNewMailEvent(folder);
        }
    }


}
