/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2020 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.json.cache;

import org.json.JSONValue;
import com.openexchange.exception.OXException;

/**
 * {@link JsonCacheService} - A simple persistent JSON cache.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public interface JsonCacheService {

    /**
     * Locks for specified entry.
     *
     * @param id The identifier
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @return <code>true</code> if successfully locked; otherwise <code>false</code>
     * @throws OXException If lock operation fails
     */
    boolean lock(String id, int userId, int contextId) throws OXException;

    /**
     * Unlocks for specified entry.
     *
     * @param id The identifier
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @throws OXException If unlock operation fails
     */
    public void unlock(String id, int userId, int contextId) throws OXException;

    /**
     * Gets denoted JSON value from cache.
     *
     * @param id The identifier
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @return The JSON value
     * @throws OXException If JSON value cannot be returned for any reason
     */
    JSONValue get(String id, int userId, int contextId) throws OXException;

    /**
     * Gets (optionally) denoted JSON value from cache.
     *
     * @param id The identifier
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @return The JSON value or <code>null</code> no such value exists
     * @throws OXException If JSON value cannot be returned for any reason
     */
    JSONValue opt(String id, int userId, int contextId) throws OXException;

    /**
     * Puts specified JSON value into cache.
     * <p>
     * A <code>null</code> value performs a delete.
     *
     * @param id The identifier
     * @param jsonValue The JSON value to put
     * @param duration The processing duration
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @throws OXException If JSON value cannot be put into cache for any reason
     */
    void set(String id, JSONValue jsonValue, long duration, int userId, int contextId) throws OXException;

    /**
     * Puts specified JSON value into cache if it differs from possibly existing one.
     * <p>
     * A <code>null</code> value performs a delete.
     *
     * @param id The identifier
     * @param jsonValue The JSON value to put
     * @param duration The processing duration
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @return <code>true</code> if put into cache; otherwise <code>false</code>
     * @throws OXException If JSON value cannot be put into cache for any reason
     */
    boolean setIfDifferent(String id, JSONValue jsonValue, long duration, int userId, int contextId) throws OXException;

    /**
     * Deletes denoted JSON value from cache.
     *
     * @param id The identifier
     * @param userId The user identifier
     * @param contextId The user's context identifier
     * @throws OXException If JSON value cannot be returned for any reason
     */
    void delete(String id, int userId, int contextId) throws OXException;

}
