/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.sync;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import com.openexchange.usm.api.session.SessionID;

/**
 * {@link SynchronizationLock}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public final class SynchronizationLock {

    private static final SimpleDateFormat _LOCK_TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS Z");

    private static class LockKey {

        private final SessionID _sessionID;

        private final String _id;

        public LockKey(SessionID sessionID, String id) {
            _sessionID = sessionID;
            _id = id;
        }

        @Override
        public int hashCode() {
            return _sessionID.hashCode() * 31 + _id.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (!(obj instanceof LockKey))
                return false;
            LockKey other = (LockKey) obj;
            return _id.equals(other._id) && _sessionID.equals(other._sessionID);
        }
    }

    private static class LockValue {

        private final String _acquirer;

        private final CountDownLatch _countdownLatch;

        public LockValue(String acquirer) {
            _acquirer = acquirer;
            _countdownLatch = new CountDownLatch(1);
        }

        public String getAcquirer() {
            return _acquirer;
        }

        public CountDownLatch getCountdownLatch() {
            return _countdownLatch;
        }
    }

    private static long _lockTimeout;

    private static final ConcurrentHashMap<LockKey, LockValue> _lockedKeys = new ConcurrentHashMap<LockKey, LockValue>(16, 0.75f, 1);

    /**
     * Configures the timeout to use for acquiring locks on specific session locks.
     * 
     * @param timeout timeout in milliseconds: The lock() method will wait for upto this time when trying to gain exclusive access to the specified lock
     */
    public static void setLockTimeout(long timeout) {
        _lockTimeout = timeout;
    }

    /**
     * Tries to acquire a lock for a given ID in a given session. Locks are acquired per session per ID. For different sessions and/or IDs a
     * different lock is used. If acquiring the lock is successful, returns null. If it is not successful, returns the "acquirer" text that
     * was stored when the lock was acquired by someone else.
     * 
     * @param session Session for which to acquire the lock
     * @param id ID for which to acquire the lock
     * @param acquirer Text that will be stored with the lock (anybody else that fails to acquire the lock while it is held will get this
     *            String which may contain some description who/why this lock was acquired)
     * @return null if lock was acquired successfully, String stored as acquirer by the code that currently holds the lock if the lock is
     *         currently not available
     */
    public static String lock(SessionID sessionID, String id, String acquirer) {
        LockKey key = new LockKey(sessionID, id);
        LockValue value = new LockValue(buildAcquirerWithDateTime(acquirer));
        LockValue oldValue = _lockedKeys.putIfAbsent(key, value);
        if (oldValue == null)
            return null;
        long now = System.currentTimeMillis();
        long limit = now + _lockTimeout;
        for (; now < limit; now = System.currentTimeMillis()) {
            try {
                if (oldValue.getCountdownLatch().await(limit - now, TimeUnit.MILLISECONDS)) {
                    oldValue = _lockedKeys.putIfAbsent(key, value);
                    if (oldValue == null)
                        return null;
                }
            } catch (InterruptedException e) {
                break;
            }
        }
        return oldValue.getAcquirer();
    }

    private static String buildAcquirerWithDateTime(String acquirer) {
        synchronized (_LOCK_TIME_FORMAT) {
            return acquirer + " (since " + _LOCK_TIME_FORMAT.format(new Date()) + ')';
        }
    }

    /**
     * Releases a previously acquired lock. No security check is made if the lock is actually currently locked. After the call the
     * session/ID combination is available for locking.
     * 
     * @param session
     * @param id
     */
    public static void unlock(SessionID sessionID, String id) {
        LockValue lock = _lockedKeys.remove(new LockKey(sessionID, id)); // Remove lock from table
        if (lock != null)
            lock.getCountdownLatch().countDown(); // Notify any threads that were waiting on lock
    }
}
