/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.util.Toolkit;

public class USMSessionCleanupTask implements Runnable {

	/**
	 * Initial delay after startup of SessionManager/TimerService after which the cleanup task is first performed
	 */
	public static final long INITIAL_DELAY = 1000L * 60L * 3L; // 3 minutes

	/**
	 * Interval between executions of the cleanup task
	 */
	public static final long EXECUTION_INTERVAL = 24L * 60L * 60L * 1000L; // 1 day
	
	private static class ContextSessionID {
	    private final int _contextID;
	    private final int _sessionID;
	    
        public ContextSessionID(int cid, int sessionID) {
            _contextID = cid;
            _sessionID = sessionID;
        }

        public int getContextID() {
            return _contextID;
        }

        public int getSessionID() {
            return _sessionID;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + _contextID;
            result = prime * result + _sessionID;
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            ContextSessionID other = (ContextSessionID) obj;
            if (_contextID != other._contextID)
                return false;
            if (_sessionID != other._sessionID)
                return false;
            return true;
        }
	}

	private final static String[] USM_TABLES = {
			"easUIDMapping",
			"USMDataStorage",
			"USMSessionFields",
			"usmIdMapping",
			"USMSession" };

	private final SessionManagerImpl _sessionManager;

	public USMSessionCleanupTask(SessionManagerImpl sessionManager) {
		_sessionManager = sessionManager;
	}

	public void run() {
		getLog().debug("Checking for old USM sessions");
		try {
			int count = 0;
			Set<Integer> remainingContextIDs = new HashSet<Integer>(_sessionManager.getDatabaseAccess().getAllContextIds());
            while (!remainingContextIDs.isEmpty()) {
                int context = remainingContextIDs.iterator().next();
                count += checkForOldSessions(context);
                for (int contextId : _sessionManager.getDatabaseAccess().getContextsInSameSchema(context))
                    remainingContextIDs.remove(contextId);
            }
			getLog().info("Checked for old USM sessions, " + count + " were removed");
		} catch (DatabaseAccessException e) {
			getLog().error("Couldn't access DB to check for old USM sessions", e);
		} catch (Exception e) {
			getLog().error("Uncaught error occurred while checking for old USM sessions", e);
		}
	}

	private Log getLog() {
		return _sessionManager.getJournal();
	}

	private int checkForOldSessions(int cid) throws DatabaseAccessException {
		getLog().debug("Checking for old USM sessions in same schema as context " + cid);
		int count = removeSessions(cid, getSessionsToRemove(cid));
		getLog().debug(count + " USM sessions removed in same schema as context " + cid);
		return count;
	}

	private int removeSessions(int cid, Set<ContextSessionID> sessionIDsToRemove) throws DatabaseAccessException {
		if (sessionIDsToRemove.isEmpty())
			return 0;
		int count = 0;
		EncapsulatedConnection con = null;
		PreparedStatement[] stmts = new PreparedStatement[USM_TABLES.length];
		try {
			con = _sessionManager.getDatabaseAccess().getWritable(cid);
			for (int i = 0; i < USM_TABLES.length; i++)
				stmts[i] = con.prepareStatement("DELETE FROM " + USM_TABLES[i] + " WHERE cid = ? AND usmSessionId = ?");
			for (ContextSessionID id : sessionIDsToRemove) {
				getLog().debug("Removing USM session data for context " + id.getContextID() + ", usmSessionId=" + id.getSessionID());
				// Directly operate on the DB, we can not create a Session object since the user name is missing (and it would also produce way too much overhead)
				for (PreparedStatement s : stmts) {
					s.setInt(1, id.getContextID());
					s.setInt(2, id.getSessionID());
					s.executeUpdate();
				}
				count++;
			}
		} catch (SQLException e) {
			getLog().error("SQL Exception while removing old USM sessions in same schema as context " + cid, e);
			return count;
		} finally {
			for (PreparedStatement s : stmts)
				Toolkit.close(s);
			Toolkit.close(con);
		}
		return count;
	}

	private Set<ContextSessionID> getSessionsToRemove(int cid) throws DatabaseAccessException {
		long now = System.currentTimeMillis();
		long interval = _sessionManager.getSessionStorageTimeLimit();
		long normalLimit = now - interval;
		long specialLimit = normalLimit - interval; // For those USM sessions that have the new field not set, use the newest synckey and the special limit
		EncapsulatedConnection con = null;
		PreparedStatement stmt = null;
		ResultSet rs = null;
		try {
			con = _sessionManager.getDatabaseAccess().getReadOnly(cid);
			// Step 0: Check if table USMSession exists
			if (!tableExists("USMSession", con.getMetaData())) {
				getLog().debug(
						"Table USMSession does not exist - no USM session data must be removed for context " + cid);
				return Collections.emptySet();
			}
			// Step 1: Get All USM sessions in that context
			stmt = con.prepareStatement("SELECT cid, usmSessionId FROM USMSession");
			rs = stmt.executeQuery();
			Set<ContextSessionID> oldSessionIDs = new HashSet<ContextSessionID>();
			while (rs.next())
				oldSessionIDs.add(new ContextSessionID(rs.getInt(1), rs.getInt(2)));
			rs.close();
			rs = null;
			stmt.close();
			stmt = null;
			// Step 2: Get last access field value for all sessions, remove session from "old" session list if field is present, add to deletion list if value is too old
			Set<ContextSessionID> sessionIDsToRemove = new HashSet<ContextSessionID>();
			stmt = con.prepareStatement("SELECT cid, usmSessionId, value FROM USMSessionFields WHERE field = ?");
			stmt.setString(1, PersistentSessionData.FIELD_PREFIX + SessionManagerImpl.LAST_ACCESS_FIELD_NAME);
			rs = stmt.executeQuery();
			while (rs.next()) {
			    ContextSessionID id = new ContextSessionID(rs.getInt(1), rs.getInt(2));
				long lastAccess = Long.parseLong(rs.getString(3));
				oldSessionIDs.remove(id);
				if (lastAccess < normalLimit)
					sessionIDsToRemove.add(id);
			}
			rs.close();
			rs = null;
			stmt.close();
			stmt = null;
			// Step 3: For all old sessions, determine newest sync key, add to deletion list if it is too old
			stmt = con.prepareStatement("SELECT MAX(SyncKey) FROM USMDataStorage WHERE cid = ? AND usmSessionId = ?");
			for (ContextSessionID sessionID : oldSessionIDs) {
				stmt.setInt(1, sessionID.getContextID());
				stmt.setInt(2, sessionID.getSessionID());
				rs = stmt.executeQuery();
				if (rs.next()) {
					long syncKey = rs.getLong(1);
					if (syncKey < specialLimit)
						sessionIDsToRemove.add(sessionID);
				} else {
					// No SyncState at all -> can be safely removed
					sessionIDsToRemove.add(sessionID);
				}
				rs.close();
				rs = null;
			}
			stmt.close();
			stmt = null;
			// Step 4: Check if any of the sessions to remove is currently active in the SessionManager, do not remove those sessions
			if (!sessionIDsToRemove.isEmpty()) {
				for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
					sessionIDsToRemove.remove(new ContextSessionID(session.getContextId(), session.getSessionId()));
				}
			}
			return sessionIDsToRemove;
		} catch (SQLException e) {
			getLog().error("SQL Exception while determining USM sessions to remove in same schema as context " + cid, e);
			return Collections.emptySet();
		} finally {
			Toolkit.close(rs);
			Toolkit.close(stmt);
			Toolkit.close(con);
		}
	}

	/**
	 * Check a table's existence
	 *
	 * @param tableName The table name to check
	 * @param dbmd The database's meta data
	 * @return <code>true</code> if table exists; otherwise <code>false</code>
	 * @throws SQLException If a SQL error occurs
	 */
	// this code has been copied from USMBaseUpdateTask.tableExists(String, DatabaseMetaData) to avoid unwanted dependencies
	private static boolean tableExists(final String tableName, final DatabaseMetaData dbmd) throws SQLException {
		ResultSet resultSet = null;
		try {
			resultSet = dbmd.getTables(null, null, tableName, new String[] { "TABLE" });
			return resultSet.next();
		} finally {
			Toolkit.close(resultSet);
		}
	}

}
