/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.ContentTypeTransferHandler;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.FolderContentType;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TimestampMismatchException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMResourceAccessException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.CompleteSessionID;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionChangeListener;
import com.openexchange.usm.api.session.SessionID;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.api.session.UserID;
import com.openexchange.usm.mapping.FolderIdMapping;
import com.openexchange.usm.ox_json.JSONResult;
import com.openexchange.usm.ox_json.JSONResultType;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.sync.ContentSyncerSupport;
import com.openexchange.usm.session.sync.FolderElementsStorage;
import com.openexchange.usm.session.sync.FolderHierarchyStorage;
import com.openexchange.usm.session.sync.OXDataCacheID;
import com.openexchange.usm.session.sync.SynchronizationLock;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;

public class SessionImpl implements Session, Serializable {

    private static final boolean MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION = false;

    private static final long FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION = -1L;

    private static final DataObject[] EMPTY_DATA_OBJECT_ARRAY = new DataObject[0];

    private final static String FOLDER_HIERARCHY_ID = "";

    private final static int FOLDER_ID_LENGTH_LIMIT = 39;

    private final static int DIRECT_PING_RESPONSE_LIMIT = 1;

    private static final long START_DATE_LIMIT = -100000000L * 86400L;

    private final PersistentSessionData _persistentData;

    private final SessionManagerImpl _sessionManager;

    private final List<SessionChangeListener> _changeListeners = new CopyOnWriteArrayList<SessionChangeListener>();

    private final Map<String, Long> _syncedFolderContent = new ConcurrentHashMap<String, Long>(16, 0.75f, 1);

    private final Map<String, Object> _customProperties = new ConcurrentHashMap<String, Object>(16, 0.75f, 1);

    private final Map<ContentType, Comparator<? super DataObject>> _contentTypeSorters = new ConcurrentHashMap<ContentType, Comparator<? super DataObject>>(
        16,
        0.75f,
        1);

    private final Queue<CustomPropertyLimit> _customPropertyLimits = new PriorityQueue<CustomPropertyLimit>();

    private final Object _waitObject = new Object(); // Used to wait on folder changes and to notify about folder changes

    private final SyncStateStorage _syncStateStorage;

    private final OXDataCacheID _folderHierarchyCacheID;

    private final CompleteSessionID _completeSessionID;

    private final SessionInitializer _sessionInitializer;
    
    private int _uniqueSessionID;

    private String _defaultInboxId;

    private String _defaultAddress;

    private String _password;

    private DataObjectFilter _syncServerObjectFilter;

    private TimeZone _userTimeZone;

    private String[] _rootFolders = new String[0];

    private long _startDate = 0L;

    private long _endDate = Long.MAX_VALUE;

    private int _mailLimit = 0;

    private long _lastFolderSync = 0L;

    private long _lastFolderChange = FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION;

    private FolderIdMapping _folderIdMapping = null;

    private DataObjectFilter _lastEmailFilter = null;

    private long _lastAccessCheck = 0L;

    private int _folderIdLengthLimit;

    private String _clientIP;

    private int _directPingResponses = 0;

    private final ReentrantLock _oxLock = new ReentrantLock();

    private Map<String, String> _xHeaders = new HashMap<String, String>();
    
    private final Map<String,String> _mailFolderIdSeparators = new HashMap<String, String>();

    public SessionImpl(SessionManagerImpl sessionManager, String user, String password, String protocol, String device, SessionInitializer initializer) {
        _sessionManager = sessionManager;
        _password = password;
        _completeSessionID = new CompleteSessionID(new SessionID(user, protocol, device), new UserID(-1, -1), null);
        _persistentData = new PersistentSessionData(this);
        _syncStateStorage = new SyncStateStorage(this);
        _folderHierarchyCacheID = new OXDataCacheID(this);
        _sessionInitializer = initializer;
    }

    public void initialize(JSONObject configuration) throws USMException {
        int contextID;
        int userIdentifier;
        try {
            contextID = configuration.getInt("context_id");
        } catch (JSONException e) {
            throw generateInitializationException("context_id", e);
        }
        try {
            userIdentifier = configuration.getInt("identifier");
        } catch (JSONException e) {
            throw generateInitializationException("identifier", e);
        }
        try {
            String timeZone = configuration.getString("timezone");
            _userTimeZone = TimeZone.getTimeZone(timeZone);
            if (!timeZone.equals(_userTimeZone.getID()))
                _sessionManager.getJournal().error(this + " Can't resolve TimeZone " + timeZone + ", using " + _userTimeZone.getID());
        } catch (JSONException e) {
            _sessionManager.getJournal().error(this + " No timezone found in initialization", e);
        }
        _completeSessionID.setUserID(new UserID(contextID, userIdentifier));
        _defaultInboxId = JSONToolkit.getString(configuration, "modules", "mail", "defaultFolder", "inbox");
        _defaultAddress = JSONToolkit.getString(configuration, "modules", "mail", "defaultaddress");
        _persistentData.initialize(_sessionInitializer, configuration);
        JSONObject separators = JSONToolkit.getJSONObject(configuration, "modules", "mail", "separators");
        if (separators != null) {
            for (String key : JSONToolkit.keys(separators)) {
                String separator = separators.optString(key);
                if (separator != null)
                    _mailFolderIdSeparators.put(key, separator);
            }
        }
    }

    private USMException generateInitializationException(String fieldName, Throwable cause) {
        String message = this + " No " + fieldName + " found in OX user configuration";
        _sessionManager.getJournal().error(message, cause);
        return new USMException(USMSessionManagementErrorCodes.SESSION_INITIALIZATION_FAILED, message);
    }

    public String getUser() {
        return _completeSessionID.getSessionID().getUser();
    }

    public String getPassword() {
        return _password;
    }

    public void setPassword(String password) {
        _password = password;
    }

    public String getDevice() {
        return _completeSessionID.getSessionID().getDevice();
    }

    public String getProtocol() {
        return _completeSessionID.getSessionID().getProtocol();
    }

    public int getContextId() {
        return _completeSessionID.getUserID().getContextId();
    }

    public int getSessionId() {
        return _uniqueSessionID;
    }

    public void setUniqueSessionID(int uniqueSessionID) {
        _uniqueSessionID = uniqueSessionID;
    }

    public int getUserIdentifier() {
        return _completeSessionID.getUserID().getUserId();
    }

    public String getDefaultEmailAddress() {
        return _defaultAddress;
    }

    public String getPersistentField(String name) {
        return _persistentData.getField(name);
    }

    public void setPersistentField(String name, String value) throws USMSQLException, DatabaseAccessException {
        _persistentData.setField(name, value);
    }

    public Map<String, String> getPersistentFields() {
        return _persistentData.getPersistentFields();
    }

    public ConflictResolution getConflictResolution() {
        return _persistentData.getConflictResolution();
    }

    private ConflictResolution getConflictResolution(ConflictResolution conflictResolution) {
        return (conflictResolution == null) ? getConflictResolution() : conflictResolution;
    }

    public void setConflictResolution(ConflictResolution resolution) throws DatabaseAccessException, USMSQLException {
        _persistentData.setConflictResolution(resolution);
    }

    public void setContentTypeFilter(ContentType... usedContentTypes) throws USMSQLException, DatabaseAccessException {
        _persistentData.setContentTypeFilter(usedContentTypes);
    }

    public void setFieldFilter(ContentType contentType, String... fieldsOfInterest) throws USMSQLException, DatabaseAccessException {
        setFieldFilter(contentType, generateFieldBitSet(contentType, fieldsOfInterest));
    }

    public void setFieldFilter(ContentType contentType, BitSet fieldsOfInterest) throws DatabaseAccessException, USMSQLException {
        _persistentData.setFieldFilter(contentType, fieldsOfInterest);
    }

    public void setRootFolders(String... folderIds) {
        _rootFolders = folderIds;
    }

    public SessionManagerImpl getSessionManager() {
        return _sessionManager;
    }

    public EncapsulatedConnection getWritableDBConnection() throws DatabaseAccessException {
        return _sessionManager.getDatabaseAccess().getWritable(getContextId());
    }

    public EncapsulatedConnection getReadOnlyDBConnection() throws DatabaseAccessException {
        return _sessionManager.getDatabaseAccess().getReadOnly(getContextId());
    }

    public BitSet getFieldFilter(ContentType contentType) {
        return _persistentData.getFieldFilter(contentType);
    }

    @Override
    public String toString() {
        return String.valueOf(getContextId()) + ':' + String.valueOf(getUserIdentifier()) + ':' + getUser() + ':' + getProtocol() + ':' + getDevice();
    }

    public String getDescription() {
        return "CID: " + getContextId() + ", ID: " + getUserIdentifier() + ", User: " + getUser() + ", Protocol: " + getProtocol() + ", Device: " + getDevice();
    }

    public SyncResult syncChangesWithServer(String folderId, long timestamp, int resultLimit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException {
        Folder folder = findFolder(folderId);
        ContentType contentType = folder.getElementsContentType();
        if (contentType == null)
            return SyncResult.EMPTY_RESULT;
        if (_sessionManager.isEmailPullEnabled() && contentType.getID().equals(DefaultContentTypes.MAIL_ID))
            _lastEmailFilter = filter;
        try {
            _syncedFolderContent.put(folderId, 0L);
            return _sessionManager.getIncrementalSyncer().syncChangesWithServer(
                createFolderElementsStorage(folder, filter, storeResult, timestamp, resultLimit),
                getConflictResolution(conflictResolution),
                getCachedFolderElements(folderId, contentType, timestamp),
                elements,
                timestamp,
                resultLimit,
                filter,
                _contentTypeSorters.get(contentType));
        } catch (FolderNotFoundException e) {
            markFolderHierarchyDirty();
            _syncedFolderContent.put(folderId, timestamp);
            throw e;
        } catch (USMException e) {
            _syncedFolderContent.remove(folderId);
            throw e;
        }
    }

    public SyncResult syncWithServer(String folderId, int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException {
        Folder folder = findFolder(folderId);
        ContentType contentType = folder.getElementsContentType();
        if (contentType == null)
            return SyncResult.EMPTY_RESULT;
        if (_sessionManager.isEmailPullEnabled() && DefaultContentTypes.MAIL_ID.equals(folder.getElementsContentTypeID()))
            _lastEmailFilter = filter;
        try {
            _syncedFolderContent.put(folderId, 0L);
            return _sessionManager.getSlowSyncer().syncWithServer(
                createFolderElementsStorage(folder, filter, storeResult, 0L, limit),
                getConflictResolution(conflictResolution),
                limit,
                filter,
                _contentTypeSorters.get(contentType),
                elements);
        } catch (FolderNotFoundException e) {
            markFolderHierarchyDirty();
            _syncedFolderContent.put(folderId, System.currentTimeMillis());
            throw e;
        } catch (USMException e) {
            _syncedFolderContent.remove(folderId);
            throw e;
        }
    }

    private FolderElementsStorage createFolderElementsStorage(Folder folder, DataObjectFilter filter, boolean storeResult, long oldTimestamp, int limit) {
        ContentType contentType = folder.getElementsContentType();
        return new FolderElementsStorage(this, folder, (filter != null) ? filter : _syncServerObjectFilter, storeResult, oldTimestamp, limit);
    }

    public BitSet getValidFilter(ContentType elementsContentType) {
        if (elementsContentType == null)
            throw new IllegalArgumentException("No ContentType set");
        BitSet filter = _persistentData.getFieldFilter(elementsContentType);
        if (filter == null)
            throw new IllegalArgumentException("ContentType " + elementsContentType.getID() + " excluded from synchronization");
        return filter;
    }

    public SyncResult syncFolderChangesWithServer(long timestamp, int resultLimit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... folderChanges) throws USMException {
        Folder[] cachedFolders = getCachedFolders(timestamp);
        removeUnusedFoldersFromSyncStateStorage(cachedFolders);
        SyncResult result = _sessionManager.getIncrementalSyncer().syncChangesWithServer(
            createFolderHierachyStorage(filter, storeResult, timestamp),
            getConflictResolution(conflictResolution),
            cachedFolders,
            folderChanges,
            timestamp,
            resultLimit,
            filter,
            getFolderSorter());
        return updateSynchronizationStateOfFolders(result);
    }

    public SyncResult syncFoldersWithServer(int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... clientData) throws USMException {
        SyncResult result = _sessionManager.getSlowSyncer().syncWithServer(
            createFolderHierachyStorage(filter, storeResult, 0L),
            getConflictResolution(conflictResolution),
            limit,
            filter,
            getFolderSorter(),
            clientData);
        return updateSynchronizationStateOfFolders(result);
    }

    private Comparator<? super DataObject> getFolderSorter() {
        return _contentTypeSorters.get(_sessionManager.getFolderContentType());
    }

    private SyncResult updateSynchronizationStateOfFolders(SyncResult result) {
        if (MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION) {
            for (DataObject object : result.getChanges()) {
                Folder f = (Folder) object;
                _syncedFolderContent.remove(f.getID());
            }
        }
        for (DataObject object : result.getNewState()) {
            Folder f = (Folder) object;
            if (f.getElementsContentType() == null)
                _syncedFolderContent.put(f.getID(), f.getTimestamp());
        }
        if (_lastFolderChange == FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION)
            _lastFolderChange = 0L;
        return result;
    }

    private void markFolderHierarchyDirty() {
        foldersChanged(FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION);
    }

    private FolderHierarchyStorage createFolderHierachyStorage(DataObjectFilter filter, boolean storeResult, long oldTimestamp) {
        return new FolderHierarchyStorage(this, filter == null ? _syncServerObjectFilter : filter, storeResult, oldTimestamp, _rootFolders);
    }

    public Folder[] getCachedFolders() throws DatabaseAccessException, USMSQLException {
        return getCachedFolders(_syncStateStorage.getNewestTimestamp(FOLDER_HIERARCHY_ID));
    }

    public Folder[] getCachedFolders(long timestamp) throws DatabaseAccessException, USMSQLException {
        return getCachedFolders(_syncStateStorage.get(FOLDER_HIERARCHY_ID, timestamp), timestamp);
    }

    private Folder[] getCachedFolders(Serializable[][] values, long timestamp) {
        if (values == null)
            return null;
        FolderContentType folderContentType = _sessionManager.getFolderContentType();
        Folder[] result = new Folder[values.length];
        for (int i = 0; i < result.length; i++) {
            Folder f = folderContentType.newDataObject(this);
            try {
                f.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                logDeserializationError(FOLDER_HIERARCHY_ID, timestamp, e);
                return null;
            }
            result[i] = completeStructure(f);
        }
        return result;
    }

    public Folder getCachedFolder(String folderID, long timestamp) throws DatabaseAccessException, USMSQLException {
        return getCachedFolder(_syncStateStorage.get(FOLDER_HIERARCHY_ID, timestamp), folderID, timestamp);
    }

    public Folder getCachedFolder(String folderID) throws DatabaseAccessException, USMSQLException {
        Folder f = _sessionManager.getSpecialFolder(this, folderID);
        if (f != null)
            return f;
        long newestTS = _syncStateStorage.getNewestTimestamp(FOLDER_HIERARCHY_ID);
        return getCachedFolder(_syncStateStorage.get(FOLDER_HIERARCHY_ID, newestTS), folderID, newestTS);
    }

    private Folder getCachedFolder(Serializable[][] values, String folderID, long timestamp) {
        if (values != null && folderID != null) {
            Folder result = _sessionManager.getFolderContentType().newDataObject(this);
            for (int i = 0; i < values.length; i++) {
                try {
                    result.deserialize(timestamp, values[i]);
                } catch (DeserializationFailedException e) {
                    logDeserializationError(folderID, timestamp, e);
                    return null;
                }
                if (folderID.equals(result.getID()))
                    return completeStructure(result);
            }
        }
        return null;
    }

    public void logDeserializationError(String folderID, long timestamp, DeserializationFailedException e) {
        _sessionManager.getJournal().error(
            this + " Ignoring invalid serialized cache data for " + folderID + '(' + timestamp + "): " + e.getMessage());
    }

    public DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp) throws DatabaseAccessException, USMSQLException {
        return getCachedFolderElements(folderId, contentType, timestamp, _syncStateStorage.get(getShortFolderID(folderId), timestamp));
    }

    public DataObject[] getCachedFolderElements(String folderId, ContentType contentType) throws DatabaseAccessException, USMSQLException {
        String shortID = getShortFolderID(folderId);
        long newestTS = _syncStateStorage.getNewestTimestamp(shortID);
        return getCachedFolderElements(folderId, contentType, newestTS, _syncStateStorage.get(shortID, newestTS));
    }

    private DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp, Serializable[][] values) {
        if (contentType == null)
            return EMPTY_DATA_OBJECT_ARRAY;
        if (values == null)
            return null;
        long start = System.currentTimeMillis();
        DataObject[] result = new DataObject[values.length];
        for (int i = 0; i < result.length; i++) {
            DataObject o = contentType.newDataObject(this);
            try {
                o.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                logDeserializationError(folderId, timestamp, e);
                return null;
            }
            result[i] = o;
        }
        long end = System.currentTimeMillis();
        if (_sessionManager.getJournal().isDebugEnabled())
            _sessionManager.getJournal().debug(
                this + " Deserialized " + result.length + " elements in " + folderId + " for " + timestamp + " in " + (end - start) + " ms.");
        return result;
    }

    public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID, long timestamp) throws DatabaseAccessException, USMSQLException {
        return getCachedFolderElement(
            folderId,
            contentType,
            objectID,
            _syncStateStorage.get(getShortFolderID(folderId), timestamp),
            timestamp);
    }

    public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID) throws DatabaseAccessException, USMSQLException {
        String shortID = getShortFolderID(folderId);
        long ts = _syncStateStorage.getNewestTimestamp(shortID);
        return getCachedFolderElement(folderId, contentType, objectID, _syncStateStorage.get(shortID, ts), ts);
    }

    private DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID, Serializable[][] values, long timestamp) {
        if (values != null) {
            DataObject result = contentType.newDataObject(this);
            for (int i = 0; i < values.length; i++) {
                try {
                    result.deserialize(timestamp, values[i]);
                } catch (DeserializationFailedException e) {
                    logDeserializationError(folderId, timestamp, e);
                    return null;
                }
                if (objectID.equals(result.getID()))
                    return result;
            }
        }
        return null;
    }

    public long storeCachedFolders(long oldTimestamp, long timestamp, DataObjectSet objects, boolean updatesPending, long syncStartTimestamp) throws USMException {
        checkIfObjectsAreFolders(objects.toArray());
        long result = internalStoreCachedDataObjects(FOLDER_HIERARCHY_ID, oldTimestamp, timestamp, objects);
        if (!updatesPending)
            _lastFolderSync = (syncStartTimestamp == 0L) ? result : syncStartTimestamp;
        return result;
    }

    public long storeCachedDataObjects(String folderID, long oldTimestamp, long timestamp, DataObjectSet objects, boolean updatesPending) throws USMException {
       checkValidObjectsForSyncState(folderID, objects.toArray());
        long result = internalStoreCachedDataObjects(getShortFolderID(folderID), oldTimestamp, timestamp, objects);
        if (updatesPending)
            _syncedFolderContent.remove(folderID);
        else if (_syncedFolderContent.containsKey(folderID))
            _syncedFolderContent.put(folderID, result);
        return result;
    }

    private long internalStoreCachedDataObjects(String folderID, long oldTimestamp, long timestamp, DataObjectSet objects) throws DatabaseAccessException, USMSQLException {
        long resultingTimestamp = _syncStateStorage.put(folderID, timestamp, objects, oldTimestamp, this);
        if (resultingTimestamp != timestamp) {
            for (DataObject o : objects) {
                o.setTimestamp(resultingTimestamp);
                o.commitChanges();
            }
        }
        return resultingTimestamp;
    }

    public DataObject readDataObject(String folderID, String objectID, String... fields) throws USMException {
        Folder folder = findFolder(folderID);
        return readDataObject(folder, objectID, generateFieldBitSet(folder.getElementsContentType(), fields));
    }

    public DataObject readDataObject(String folderID, String objectID, BitSet fields) throws USMException {
        return readDataObject(findFolder(folderID), objectID, fields);
    }

    private Folder completeStructure(Folder f) {
        f.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(f.getElementsContentTypeID()));
        return f;
    }

    private DataObject readDataObject(Folder f, String objectID, BitSet fields) throws USMException {
        ContentType contentType = f.getElementsContentType();
        DataObject object = contentType.newDataObject(this);
        object.setID(objectID);
        object.setParentFolder(f);
        contentType.getTransferHandler().readDataObject(object, fields);
        return object;
    }

    public Map<DataObject, USMException> updateDataObjects(DataObject... changedObjects) throws USMException {
        Map<DataObject, USMException> resultMap = new HashMap<DataObject, USMException>();

        // First, split between folder changes and element changes within folders
        List<DataObject> elementChanges = new ArrayList<DataObject>();
        List<DataObject> folderChanges = new ArrayList<DataObject>();
        for (DataObject o : changedObjects) {
            o.setFailed(true);
            if (o.getContentType().getID().equals(DefaultContentTypes.FOLDER_ID))
                folderChanges.add(o);
            else
                elementChanges.add(o);
        }
        if (!folderChanges.isEmpty()) {
            // perform all folder changes, update timestamp if possible
            long timestamp = folderChanges.get(0).getTimestamp();
            DataObjectSet newFolderHierarchy = new DataObjectSet();
            long newTimestamp = checkTimestampAndUpdate(
                newFolderHierarchy,
                folderChanges,
                timestamp,
                getCachedFolders(timestamp),
                "All Folders must have the same timestamp",
                resultMap);
            _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
            if (newTimestamp != 0) {
                long modifiedTimestamp = storeCachedFolders(timestamp, newTimestamp, newFolderHierarchy, true, 0L);
                for (DataObject folder : folderChanges)
                    folder.setTimestamp(modifiedTimestamp);
            }
        }
        // group elements by folder, operate on each folder separately
        while (!elementChanges.isEmpty()) {
            DataObject first = elementChanges.remove(elementChanges.size() - 1);
            long timestamp = first.getTimestamp();
            List<DataObject> currentFolderChanges = new ArrayList<DataObject>();
            currentFolderChanges.add(first);
            String folderId = first.getParentFolderID();
            for (int i = elementChanges.size() - 1; i >= 0; i--) {
                DataObject o = elementChanges.get(i);
                if (o.getParentFolderID().equals(folderId))
                    currentFolderChanges.add(elementChanges.remove(i));
            }
            _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, folderId));
            DataObjectSet newElements = new DataObjectSet();
            long newTimestamp = checkTimestampAndUpdate(
                newElements,
                currentFolderChanges,
                timestamp,
                getCachedFolderElements(folderId, first.getContentType(), timestamp),
                "All elements in a Folder must have the same timestamp",
                resultMap);
            if (newTimestamp != 0) {
                long modifiedTimestamp = storeCachedDataObjects(folderId, timestamp, newTimestamp, newElements, true);
                for (DataObject object : currentFolderChanges)
                    object.setTimestamp(modifiedTimestamp);
            }
        }
        return resultMap;
    }

    private long checkTimestampAndUpdate(DataObjectSet resultingServerObjects, List<DataObject> requests, long timestamp, DataObject[] oldState, String errorMessage, Map<DataObject, USMException> resultMap) throws DatabaseAccessException, USMSQLException {
        for (DataObject o : requests) {
            if (o.getTimestamp() != timestamp) {
                // If at least one element has a timestamp mismatch, fail on all elements in the folder
                for (DataObject o2 : requests) {
                    resultMap.put(o2, new TimestampMismatchException(
                        USMSessionManagementErrorCodes.TIMESTAMP_MISMATCH_ON_DATAOBJECT_UPDATE,
                        errorMessage));
                }
                return 0;
            }
        }
        ContentType type = requests.get(0).getContentType();
        ContentTypeTransferHandler handler = type.getTransferHandler();
        if (oldState == null) {
            for (DataObject o : requests) {
                callServerUpdatesAccordingToTimstamp(resultMap, handler, o);
            }
            for (DataObject o : requests)
                o.setFailed(false);
            return 0;
        }
        resultingServerObjects.clear();
        List<DataObject> creations = new ArrayList<DataObject>();
        List<DataObject> changes = new ArrayList<DataObject>();
        List<DataObject> deletions = new ArrayList<DataObject>();
        for (DataObject o : oldState)
            resultingServerObjects.add(o.createCopy(true));
        for (DataObject o : requests) {
            fillMapsAccordingToChangedState(resultMap, creations, changes, deletions, o);
        }

        ContentSyncerSupport.executeServerUpdates(_sessionManager, resultingServerObjects, creations, changes, deletions, resultMap);

        for (DataObject o : requests)
            o.setFailed(false);
        return ContentSyncerSupport.updateTimestampAndCommitChanges(resultingServerObjects);
    }

    private void fillMapsAccordingToChangedState(Map<DataObject, USMException> resultMap, List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions, DataObject o) {
        switch (o.getChangeState()) {
        case CREATED:
            creations.add(o);
            return;
        case DELETED:
            deletions.add(o);
            return;
        case MODIFIED:
            changes.add(o);
            return;
        case UNMODIFIED:
            return;
        }
        resultMap.put(
            o,
            new InternalUSMException(
                USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER1,
                "Illegal ChangeState " + o.getChangeState()));
    }

    private void callServerUpdatesAccordingToTimstamp(Map<DataObject, USMException> resultMap, ContentTypeTransferHandler handler, DataObject o) {
        switch (o.getChangeState()) {
        case CREATED:
            try {
                handler.writeNewDataObject(o);
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case DELETED:
            try {
                handler.writeDeletedDataObject(o);
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case MODIFIED:
            try {
                handler.writeUpdatedDataObject(o, o.getTimestamp());
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case UNMODIFIED:
            return;
        }
        resultMap.put(
            o,
            new InternalUSMException(
                USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER2,
                "Illegal ChangeState " + o.getChangeState()));
    }

    public BitSet generateFieldBitSet(ContentType contentType, String... fieldsOfInterest) {
        ContentTypeField[] fields = contentType.getFields();
        BitSet fieldSet = new BitSet();
        for (String fieldName : fieldsOfInterest) {
            generateFieldsOfInterestBitSet(contentType, fields, fieldSet, fieldName);
        }
        return fieldSet;
    }

    private void generateFieldsOfInterestBitSet(ContentType contentType, ContentTypeField[] fields, BitSet fieldSet, String fieldName) {
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldName().equals(fieldName)) {
                fieldSet.set(i);
                return;
            }
        }
        throw new IllegalArgumentException("Field " + fieldName + " not provided by ContentType " + contentType);
    }

    public Folder findFolder(String folderID) throws USMException {
        Folder folder = _sessionManager.getSpecialFolder(this, folderID);
        if (folder != null)
            return folder;
        folder = getCachedFolder(folderID);
        if (folder != null)
            return folder;
        FolderContentType type = _sessionManager.getFolderContentType();
        Folder result = type.newDataObject(this);
        result.setID(folderID);
        type.getTransferHandler().readDataObject(result, getFieldFilter(type));
        result.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(result.getElementsContentTypeID()));
        return result;
    }

    public void addChangeListener(SessionChangeListener listener) {
        _changeListeners.add(listener);
    }

    public void removeChangeListener(SessionChangeListener listener) {
        _changeListeners.remove(listener);
    }

    public void foldersChanged(long timestamp) {
        _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
        if (_changeListeners.size() > 0) {
            ObjectChanges changes = new ObjectChangesImpl(true);
            for (SessionChangeListener listener : _changeListeners) {
                try {
                    listener.changesOccurred(changes);
                } catch (Exception e) {
                    _sessionManager.getJournal().error(
                        this + " Exception while notifying listener " + listener + " of folder structure change",
                        e);
                }
            }
        }
        _lastFolderChange = timestamp;
        _sessionManager.getJournal().debug("lastFolderChange " + _lastFolderChange);
        synchronized (_waitObject) {
            _waitObject.notifyAll();
        }
    }

    public void folderContentChanged(long timestamp) {
        try {
            folderContentChanged(timestamp, getAllFolderIDs(getCachedFolders()));
        } catch (DatabaseAccessException e) {
            _sessionManager.getJournal().error(this + " Database access error during OX event notification", e);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.OX_EVENT_DB_ERROR_1,
                this + " Database access error during OX event notification",
                e);
        } catch (USMSQLException e) {
            _sessionManager.getJournal().error(this + " SQL error during OX event notification", e);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.OX_EVENT_DB_ERROR_2,
                this + " SQL error during OX event notification",
                e);
        }
    }

    public void folderContentChanged(long timestamp, String... folderIds) {
        Log journal = _sessionManager.getJournal();
        if (journal.isDebugEnabled())
            journal.debug("Folder Content changed, timestamp: " + timestamp);
        if (_changeListeners.size() > 0) {
            ObjectChanges changes = new ObjectChangesImpl(false, folderIds);
            for (SessionChangeListener listener : _changeListeners) {
                try {
                    listener.changesOccurred(changes);
                } catch (Exception e) {
                    _sessionManager.getJournal().error(
                        this + " Exception while notifying listener " + listener + " of content change in folders " + Arrays.toString(folderIds),
                        e);
                }
            }
        }
        for (String id : folderIds) {
            if (null != id) {
                journal.debug("Removing folder from sync folder content, " + id);
                _syncedFolderContent.remove(id);
                _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, id));
                if ("1".equals(id)) { // for Remote Events when folderID == 1, it means that the default inbox has been changed
                    if (_defaultInboxId != null) {
                        if (journal.isDebugEnabled())
                            journal.debug("Removing default inbox from sync folder content, " + _defaultInboxId);
                        _syncedFolderContent.remove(_defaultInboxId);
                        _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, _defaultInboxId));
                    }
                }
            }
        }
        synchronized (_waitObject) {
            _waitObject.notifyAll();
        }
    }
    
    public boolean needsSynchronization(String folderID) {
        return Toolkit.provided(folderID) ? !_syncedFolderContent.containsKey(folderID) : isFolderHierarchySynchronized();
    }

    private boolean isFolderHierarchySynchronized() {
        return _lastFolderSync != 0L && _lastFolderChange != FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION && _lastFolderSync >= _lastFolderChange;
    }

    public ObjectChanges waitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException {
        Log journal = _sessionManager.getJournal();
        if (journal.isDebugEnabled())
            journal.debug(this + " waitforChanges(" + watchFolderChanges + "," + Arrays.toString(contentChangeParentIDsToWatch) + "," + timeout + "), lastFolderChange=" + _lastFolderChange + ", lastFolderSync=" + _lastFolderSync);
        ObjectChanges result = waitForChanges(
            _sessionManager.isEmailPullEnabled(),
            watchFolderChanges,
            contentChangeParentIDsToWatch,
            timeout);
        if (journal.isDebugEnabled())
            journal.debug(this + " waitforChanges Result=" + result);
        return result;
    }

    private ObjectChanges waitForChanges(boolean emailPullEnabled, boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException {
        Log journal = _sessionManager.getJournal();
        ObjectChanges result;
        Folder[] folders = null;
        if (contentChangeParentIDsToWatch == null) {
            folders = getCachedFolders();
            contentChangeParentIDsToWatch = getAllFolderIDs(folders);
        }
        synchronized (_waitObject) {
            result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
            if (result != null) {
                if (!_sessionInitializer.allowsUnlimitedWaitForChangesOnModifiedFolders() && ++_directPingResponses > DIRECT_PING_RESPONSE_LIMIT && "EAS".equals(getProtocol())) {
                    // fix for bug 22741 - enable the ping response limit ONLY for protocols that require it (e.g. EAS)
                    if (journal.isDebugEnabled())
                        _sessionManager.getJournal().debug(
                            this + " Too many pings without sync, marked as synchronized: " + Arrays.toString(contentChangeParentIDsToWatch));
                    for (String id : contentChangeParentIDsToWatch) {
                        _syncedFolderContent.put(id, 0L);
                    }
                }
                return result;
            }
        }
        _directPingResponses = 0;
        emailPullEnabled &= contentChangeParentIDsToWatch.length > 0;
        long start = System.currentTimeMillis();
        long end = start + timeout;
        if (emailPullEnabled) {
            if (folders == null)
                folders = getCachedFolders();
            List<Folder> emailFolders = new ArrayList<Folder>();
            if (folders != null) {
                DataObjectSet folderSet = new DataObjectSet(folders);
                for (String folderId : contentChangeParentIDsToWatch) {
                    Folder f = (Folder) folderSet.get(folderId);
                    if (f != null && DefaultContentTypes.MAIL_ID.equals(f.getElementsContentTypeID()))
                        emailFolders.add(f);
                }
            }
            if (!emailFolders.isEmpty()) {
                long pullTime = _sessionManager.computePullTime(start, end);
                if (pullTime > 0) {
                    if (journal.isDebugEnabled())
                        journal.debug(this + " Waiting " + (pullTime - start) + "ms until pull timeout");
                    result = doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, pullTime);
                    if (result != null)
                        return result;
                    if (journal.isDebugEnabled())
                        _sessionManager.getJournal().debug(this + " Performing email pull");
                    result = performEmailPull(emailFolders);
                    if (result != null)
                        return result;
                }
                if (journal.isDebugEnabled())
                    journal.debug(this + " Waiting " + (end - System.currentTimeMillis()) + "ms after pull timeout");
            }
        }
        return doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, end);
    }

    private ObjectChanges performEmailPull(List<Folder> emailFolders) throws USMException {
        List<String> changedFolders = new ArrayList<String>();
        for (Folder f : emailFolders) {
            String folderId = f.getID();
            Long timestamp = _syncedFolderContent.get(folderId);
            if (timestamp == null || timestamp == 0L)
                changedFolders.add(folderId);
            else {
                SyncResult syncResult = syncChangesWithServer(folderId, timestamp, NO_LIMIT, _lastEmailFilter, false, null);
                if (syncResult.getChanges().length > 0)
                    changedFolders.add(folderId);
            }
        }
        return ObjectChangesImpl.create(false, changedFolders);
    }

    private ObjectChanges doWaitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long end) throws InterruptedException {
        ObjectChanges result = null;
        for (long waitTime = end - System.currentTimeMillis(); waitTime > 0; waitTime = end - System.currentTimeMillis()) {
            synchronized (_waitObject) {
                result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
                if (result != null)
                    return result;
                _waitObject.wait(waitTime);
            }
        }
        return result;
    }

    private ObjectChanges extractFolderChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch) {
        boolean folderStructureChanged = watchFolderChanges && !isFolderHierarchySynchronized();
        List<String> changedFolders = new ArrayList<String>();
        for (String folderId : contentChangeParentIDsToWatch) {
            if (!_syncedFolderContent.containsKey(folderId))
                changedFolders.add(folderId);
        }
        return ObjectChangesImpl.create(folderStructureChanged, changedFolders);
    }

    private String[] getAllFolderIDs(Folder[] folders) {
        if (folders == null)
            return new String[0];
        String[] folderIds = new String[folders.length + SessionManagerImpl.SPECIAL_FOLDERS.size()];
        for (int i = 0; i < folders.length; i++)
            folderIds[i] = folders[i].getID();
        int foldersLength = folders.length;
        for (int i = 0; i < SessionManagerImpl.SPECIAL_FOLDERS.size(); i++)
            folderIds[foldersLength + i] = SessionManagerImpl.SPECIAL_FOLDERS.get(i);
        return folderIds;
    }

    public Object getCustomProperty(String key) {
        return _customProperties.get(key);
    }

    public Object setCustomProperty(String key, Object value) {
        synchronized (_customPropertyLimits) {
            _customPropertyLimits.remove(new CustomPropertyLimit(key));
        }
        Object oldValue = (value == null) ? _customProperties.remove(key) : _customProperties.put(key, value);
        return oldValue;
    }

    public Object setCustomProperty(String key, Object value, long timeout) {
        synchronized (_customPropertyLimits) {
            Object oldValue = (value == null) ? _customProperties.remove(key) : _customProperties.put(key, value);
            if (oldValue != null)
                _customPropertyLimits.remove(new CustomPropertyLimit(key));
            _customPropertyLimits.add(new CustomPropertyLimit(key, timeout));
            return oldValue;
        }
    }

    public int removeOutdatedCustomProperties(long now) {
        int count = 0;
        synchronized (_customPropertyLimits) {
            for (CustomPropertyLimit limit = _customPropertyLimits.peek(); limit != null && limit.getStorageLimit() < now; limit = _customPropertyLimits.peek()) {
                if (_customProperties.remove(limit.getKey()) != null)
                    count++;
                _customPropertyLimits.poll();
            }
        }
        return count;
    }

    public DataObjectFilter getSyncServerObjectFilter() {
        return _syncServerObjectFilter;
    }

    public void setSyncServerObjectFilter(DataObjectFilter filter) {
        _syncServerObjectFilter = filter;
    }

//    public static void main(String[] args) {
//        Thread[] writeThreads = new Thread[100];
//        final SessionImpl mySession = new SessionImpl(null, "user", "password", "protocol", "device");
//        for (int i = 0; i < writeThreads.length; i++) {
//            writeThreads[i] = new Thread("Thread " + i) {
//
//                @Override
//                public void run() {
//                    try {
//                        for (;;) {
//                            mySession.foldersChanged(0L);
//                            Thread.sleep(100L);
//                            mySession.folderContentChanged(0L, getName());
//                            Thread.sleep(100L);
//                        }
//                    } catch (InterruptedException ignored) {
//                        // exit on interrupt
//                    }
//                }
//            };
//            writeThreads[i].start();
//        }
//        Thread readThread = new Thread("Reader") {
//
//            @Override
//            public void run() {
//                try {
//                    for (;;) {
//                        ObjectChanges result = mySession.waitForChanges(true, null, 1000L);
//                        System.out.println("Result: " + result);
//                    }
//                } catch (USMException e) {
//                    e.printStackTrace();
//                } catch (InterruptedException e) {
//                    e.printStackTrace();
//                }
//            }
//        };
//        readThread.start();
//        // try {
//        // Thread.sleep(10000L);
//        // } catch (InterruptedException e) {
//        // }
//        // for (Thread t : writeThreads)
//        // t.interrupt();
//        // readThread.interrupt();
//    }

    public TimeZone getUserTimeZone() {
        return _userTimeZone;
    }

    public long getEndDate() {
        return _endDate;
    }

    public long getStartDate() {
        return _startDate;
    }

    public void setEndDate(long date) {
        _endDate = date;
    }

    public void setStartDate(long date) {
        _startDate = (date < START_DATE_LIMIT) ? START_DATE_LIMIT : date;
    }

    public int getMailLimit() {
        return _mailLimit;
    }

    public void setMailLimit(int limit) {
        _mailLimit = limit;
    }

    public int getObjectsLimit() {
        return _sessionManager.getMaxObjectsPerFolder();
    }

    public int getPIMAttachmentsCountLimit() {
        return _sessionManager.getAttachmentCountLimitPerObject();
    }

    public long getPIMAttachmentsSizeLimit() {
        return _sessionManager.getAttachmentSizeLimitPerObject();
    }

    public String getFolderID(String shortFolderID) throws USMSQLException {
        if (shortFolderID.length() > 0) {
            switch (shortFolderID.charAt(0)) {
            case 'D':
                try {
                    return getFolderIdMapping().getLongID(Integer.parseInt(shortFolderID.substring(1)));
                } catch (NumberFormatException ignored) {
                    // Common Exception containing error message will be generated at end of method
                }
                break;
            case 'F':
                return shortFolderID.substring(1);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Illegal short folder ID " + shortFolderID);
    }

    public String getShortFolderID(String folderID) throws DatabaseAccessException, USMSQLException {
        if (folderID.length() < getFolderIdLengthLimit())
            return "F" + folderID;
        return "D" + getFolderIdMapping().getShortID(folderID);
    }

    private FolderIdMapping getFolderIdMapping() {
        if (_folderIdMapping == null)
            _folderIdMapping = _sessionManager.getMappingService().getShortIDStorage(this);
        return _folderIdMapping;
    }

    @Deprecated
    public long storeSyncState(long timestamp, String folderID, DataObject[] objects) throws USMException {
        return storeSyncState(0L, timestamp, folderID, objects);
    }

    public long storeSyncState(long timestampToKeep, long timestamp, String folderID, DataObject[] objects) throws USMException {
        DataObjectSet set = checkValidObjectsForSyncState(folderID, objects);
        _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, folderID));
        return storeCachedDataObjects(folderID, timestampToKeep, timestamp, set, false);
    }

    private DataObjectSet checkValidObjectsForSyncState(String folderID, DataObject[] objects) throws USMException {
        DataObjectSet set = new DataObjectSet();
        ContentType t = null;
        for (DataObject o : objects) {
            if (t == null)
                t = o.getContentType();
            else if (!o.getContentType().getID().equals(t.getID())) {
                throw new USMException(
                    USMSessionManagementErrorCodes.ILLEGAL_OBJETCS_TO_BE_SAVED_IN_SYNC_STATE,
                    "DataObjects of different ContentTypes");
            }
            if (o.getContentType().getCode() != DefaultContentTypes.RESOURCES_CODE && o.getContentType().getCode() != DefaultContentTypes.GROUPS_CODE)
                o.setParentFolderID(folderID);
            set.add(o);
        }
        return set;
    }

    public long storeSyncState(long timestampToKeep, long timestamp, DataObject[] objects) throws USMException {
        checkIfObjectsAreFolders(objects);
        _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
        DataObjectSet set = new DataObjectSet(objects);
        return storeCachedFolders(timestampToKeep, timestamp, set, false, 0L);
    }

    private void checkIfObjectsAreFolders(DataObject[] objects) throws USMException {
        for (DataObject o : objects) {
            if (o.getContentType().getCode() != DefaultContentTypes.FOLDER_CODE)
                throw new USMException(USMSessionManagementErrorCodes.ILLEGAL_OBJETCS_TO_BE_SAVED_IN_SYNC_STATE, "DataObjects are not folders");
        }
    }

    public void endSyncronization() throws USMSQLException, DatabaseAccessException {
        _syncStateStorage.removeAllObjectsForSession();
        _persistentData.removeAllDBFieldsForSession();
        _sessionManager.getSessionStorage().removeSession(this);
    }

    public void storeUUID(ContentType contentType, int objectId, UUID uuid, int folderOwner) throws USMException {
        _sessionManager.storeUUID(getContextId(), contentType, objectId, uuid, (folderOwner == getUserIdentifier()) ? 0 : folderOwner);
    }

    public PersistentSessionData getPersistentSessionData() {
        return _persistentData;
    }

    public UUID getContextUUID() throws DatabaseAccessException, USMSQLException {
        return _sessionManager.getContextUUID(getContextId());
    }

    public void insertStoredUUID(DataObject o) throws DatabaseAccessException, USMSQLException {
        _sessionManager.insertStoredUUID(o, null, null);
    }

    public int getMappedObjectId(ContentType contentType, UUID uuid) throws USMException {
        return _sessionManager.getMappedObject(this, contentType.getCode(), uuid);
    }

    public UUID getUUID(ContentType contentType, int objectId) throws USMException {
        return _sessionManager.getUUID(this, contentType.getCode(), objectId);
    }

    public Folder getDummyContentTypeFolder(ContentType type) {
        return _sessionManager.getSpecialFolder(this, type.getID());
    }

    public ContentType[] getContentTypes() throws DatabaseAccessException, USMSQLException {
        return _persistentData.getContentTypes();
    }

    public void remapCachedData(String oldObjectID, String newObjectID) throws DatabaseAccessException, USMSQLException {
        _syncStateStorage.remapStates(oldObjectID, newObjectID);
    }

    public void setLastAccessCheck(long lastAccessCheck) {
        _lastAccessCheck = lastAccessCheck;
    }

    public long getLastAccessCheck() {
        return _lastAccessCheck;
    }

    public JSONObject getOXUserConfiguration(String... path) throws AuthenticationFailedException, OXCommunicationException {
        return _sessionManager.getOXAjaxAccess().getConfiguration(this, path);
    }

    public boolean checkProperDBStorage() throws DatabaseAccessException, USMSQLException {
        return _persistentData.checkProperDBStorage();
    }

    public void setContentTypeSorter(ContentType contentType, Comparator<? super DataObject> comparator) {
        if (comparator == null)
            _contentTypeSorters.remove(contentType);
        else
            _contentTypeSorters.put(contentType, comparator);
    }

    public OXConnectionInformation getOXConnectionData() {
        return _completeSessionID.getOXConnectionInformation();
    }

    public void setOXConnectionData(OXConnectionInformation data) {
        _completeSessionID.setOXConnectionInformation(data);
    }

    public OXDataCacheID getFolderHierarchyCacheID() {
        return _folderHierarchyCacheID;
    }

    public int getFolderIdLengthLimit() {
        return _folderIdLengthLimit > 0 ? _folderIdLengthLimit : FOLDER_ID_LENGTH_LIMIT;
    }

    public void setFolderIdLengthLimit(int limit) {
        _folderIdLengthLimit = limit;
    }

    public String getClientIp() {
        return _clientIP;
    }

    public void setClientIp(String ip) {
        _clientIP = ip;
    }

    public void updateClientIp(String ip) {
        if (_clientIP.equals(ip))
            return;
        _clientIP = ip;
        Log journal = _sessionManager.getJournal();
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("clientIP", _clientIP);
        try {
            JSONResult result = _sessionManager.getOXAjaxAccess().doPost("login", "changeip", this, parameters);
            if (result.getResultType() == JSONResultType.Error && journal.isDebugEnabled())
                journal.debug(this + " Changing ip failed: " + result.toString());
        } catch (AuthenticationFailedException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " Authentication failed on changing client ip.");
        } catch (OXCommunicationException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " OX Communication Exception on changing client ip.");
        }
    }

    public Map<String, String> getXRequestHeaders() {
        return _xHeaders;
    }

    public void setXRequestHeaders(Map<String, String> headers) {
        _xHeaders = headers;
    }

    public long getNewestTimestamp(String folderID) {
        Log journal = _sessionManager.getJournal();
        try {
            String shortID = getShortFolderID(folderID);
            return _syncStateStorage.getNewestTimestamp(shortID);
        } catch (DatabaseAccessException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " DatabaseAccessException on getting the newest timestamp for folder: " + folderID);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.SESSION_TIMESTAMP_DB_ERROR_1,
                this + " DatabaseAccessException on getting the newest timestamp for folder: " + folderID,
                e);
        } catch (USMSQLException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " USMSQLException on getting the newest timestamp for folder: " + folderID);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.SESSION_TIMESTAMP_DB_ERROR_2,
                this + " USMSQLException on getting the newest timestamp for folder: " + folderID,
                e);
        }
    }

    public void invalidateCachedData(String folderID) {
        _sessionManager.getOXDataCache().removeCachedData(
            Toolkit.provided(folderID) ? new OXDataCacheID(this, folderID) : _folderHierarchyCacheID);
        if (Toolkit.provided(folderID))
            _syncedFolderContent.remove(folderID);
        else
            markFolderHierarchyDirty();
    }

    public SyncStateStorage getDataObjectCache() {
        return _syncStateStorage;
    }

    public Lock getOXConnectionLock() {
        return _oxLock;
    }

    public CompleteSessionID getCompleteSessionID() {
        return _completeSessionID;
    }

    public String tryLock(String id, String acquirer) {
        return SynchronizationLock.lock(_completeSessionID.getSessionID(), id, acquirer);
    }

    public void unlock(String id) {
        SynchronizationLock.unlock(_completeSessionID.getSessionID(), id);
    }
    
    public SessionInitializer getSessionInitializer() {
        return _sessionInitializer;
    }
    
    public Map<String,String> getMailFolderIdSeparators() {
        return _mailFolderIdSeparators;
    }

    private void removeUnusedFoldersFromSyncStateStorage(Folder[] cachedFolders) throws DatabaseAccessException, USMSQLException {
        if (cachedFolders == null)
            return;
        List<String> shortIDs = new ArrayList<String>();
        List<String> folderIDs = new ArrayList<String>();
        for (DataObject o : cachedFolders) {
            String id = o.getID();
            folderIDs.add(id);
            shortIDs.add(getShortFolderID(id));
        }
        shortIDs.add(FOLDER_HIERARCHY_ID);
        for (String specialFolder : _sessionManager.getSpecialFolders())
            shortIDs.add(getShortFolderID(specialFolder));
        _syncStateStorage.retain(shortIDs.toArray(new String[shortIDs.size()]));
        if (_folderIdMapping != null)
            _folderIdMapping.retain(folderIDs.toArray(new String[folderIDs.size()]));
    }
}
