/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.connector.exceptions.InvalidUUIDException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.JSONToolkit;

/**
 * Handler for the USM-Ping Command.
 * @author ldo
 */
public class PingHandler extends NormalCommandHandler {

    private static final String _DEFAULT_INBOX_FOLDER_KEY = "json.DefaultInboxFolder";

    private static final String INTERVAL = "interval";
    private static final String[] REQUIRED_PARAMETERS = { SESSIONID };
    private static final String[] OPTIONAL_PARAMETERS = { OBJECTIDS };

    private DataObjectSet _allCachedFolders;

    private boolean _watchFolderHierarchyChanges = true;

    public PingHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
        _servlet.newPingRequestArrived();
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {

        JSONArray objectUUIDs = _parameters.has(OBJECTIDS) ? getJSONArray(_parameters, OBJECTIDS) : null;
        try {
            String[] oxIDs = extractOXIdsFromUUIDArray(objectUUIDs);
            ObjectChanges changes = getSession().waitForChanges(_watchFolderHierarchyChanges, oxIDs, 0);
            JSONArray resultArray = null;
            if (changes != null) {
                resultArray = toUUIDsJSONArray(changes.getContentChangeParentIDs());
                if (changes.hasFolderStructureChanged())
                    resultArray.put(String.valueOf(getSession().getContextUUID()));
            }
            String inboxId = getDefaultInboxId();
            if (inboxId != null && !hasDefaultInboxChanges(changes, inboxId)) {
                if (resultArray == null)
                    resultArray = new JSONArray();
                long newestTimestamp = getSession().getNewestTimestamp(inboxId);
                if (newestTimestamp > 0 && System.currentTimeMillis() - newestTimestamp >= _servlet.getInboxPollingInterval()) {
                    // clear cached value so that it will be updated from OX configuration
                    getSession().setCustomProperty(_DEFAULT_INBOX_FOLDER_KEY, null);
                    Folder defaultInbox = (Folder) getAllCachedFolders().get(inboxId);
                    if (defaultInbox != null) {
                        SyncResult syncResult = getSession().syncChangesWithServer(
                            inboxId,
                            newestTimestamp,
                            Session.NO_LIMIT,
                            null,
                            false,
                            null);
                        if (syncResult.getChanges().length > 0 && defaultInbox.getUUID() != null)
                            resultArray.put(defaultInbox.getUUID());
                    }
                }
            }
            JSONObject data = new JSONObject();
            if (resultArray != null && resultArray.length() > 0)
                data.put(OBJECTIDS, resultArray);
            data.put(INTERVAL, _servlet.getPingInterval());
            return new ResponseObject(ResponseStatusCode.SUCCESS, data);
        } catch (InvalidUUIDException e) {
            throw USMJSONAPIException.createInvalidUUIDException(e);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_INTERNAL_ERROR, e);
        } catch (InterruptedException e) {
            throw new USMJSONAPIException(ConnectorBundleErrorCodes.PING_WAIT_FOR_CHANGES_INTERRUPTED_ERROR,
                    ResponseStatusCode.INTERNAL_ERROR, "waiting for changes interrupted", e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.PING_JSON_ERROR, e);
        }
    }

    private DataObjectSet getAllCachedFolders() throws DatabaseAccessException, USMSQLException {
        if (_allCachedFolders == null) {
            _allCachedFolders = new DataObjectSet(getSession().getCachedFolders());
        }
        return _allCachedFolders;
    }

    private String getDefaultInboxId() throws USMException {
        Object o = getSession().getCustomProperty(_DEFAULT_INBOX_FOLDER_KEY);
        if (o instanceof String)
            return (String) o;
        String defaultInboxID = JSONToolkit.getString(getSession().getOXUserConfiguration("modules", "mail", "defaultFolder"), "inbox");
        getSession().setCustomProperty(_DEFAULT_INBOX_FOLDER_KEY, defaultInboxID);
        return defaultInboxID;
    }

    private static boolean hasDefaultInboxChanges(ObjectChanges changes, String inboxId) {
        if (changes == null)
            return false;

        for (String folderID : changes.getContentChangeParentIDs()) {
            if (folderID.equals(inboxId))
                return true;
        }
        return false;
    }

    private UUID getContextUUID() {
        try {
            return getSession().getContextUUID();
        } catch (DatabaseAccessException e) {
            USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_NO_DB_ACCESS, e);
        } catch (USMSQLException e) {
            USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_SQL_ERROR, e);
        }
        return null; // not reached
    }

    private String[] extractOXIdsFromUUIDArray(JSONArray jsonArray) throws USMJSONAPIException, InvalidUUIDException, DatabaseAccessException, USMSQLException {
        if (jsonArray == null)
            return null;
        int length = jsonArray.length();
        List<String> oxIDs = new ArrayList<String>(length);
        UUID contextUUID = getContextUUID();
        for (int i = 0; i < length; i++) {
            String id = getString(jsonArray, i);
            if (isSpecialFolderType(id)) {
                Folder f = (Folder) getAllCachedFolders().get(id);
                if (f != null)
                    oxIDs.add(f.getID());
            } else {
                UUID uuid = extractUUIDFromString(id);
                if (uuid.equals(contextUUID)) {
                    _watchFolderHierarchyChanges = true;
                } else {
                    Folder f = (Folder) getAllCachedFolders().get(uuid);
                    if (f != null)
                        oxIDs.add(f.getID());
                }
            }
        }
        return oxIDs.toArray(new String[oxIDs.size()]);
    }

    private JSONArray toUUIDsJSONArray(String[] idsArray) throws DatabaseAccessException, USMSQLException {
        DataObjectSet allFolders = getAllCachedFolders();
        JSONArray result = new JSONArray();
        for (int i = 0; i < idsArray.length; i++) {
            if (isSpecialFolderType(idsArray[i])) {
                result.put(idsArray[i]);
            } else {
                Folder cachedFolder = (Folder) allFolders.get(idsArray[i]);
                if (cachedFolder != null && cachedFolder.getUUID() != null && isOfDefaultContentType(cachedFolder.getElementsContentTypeID()))
                    result.put(cachedFolder.getUUID().toString());
            }
        }
        return result;
    }

    private static boolean isSpecialFolderType(String id) {
        return DefaultContentTypes.GROUPS_ID.equals(id) || DefaultContentTypes.RESOURCES_ID.equals(id);
    }

    private static boolean isOfDefaultContentType(String contentTypeID) {
        return DefaultContentTypes.FOLDER_ID.equals(contentTypeID) || DefaultContentTypes.CONTACTS_ID.equals(contentTypeID) || DefaultContentTypes.CALENDAR_ID.equals(contentTypeID) || DefaultContentTypes.TASK_ID.equals(contentTypeID) || DefaultContentTypes.MAIL_ID.equals(contentTypeID);
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }
}
