/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.realtime.handle.impl;

import java.util.concurrent.BlockingQueue;
import com.openexchange.exception.OXException;
import com.openexchange.realtime.directory.ResourceDirectory;
import com.openexchange.realtime.dispatch.MessageDispatcher;
import com.openexchange.realtime.exception.RealtimeExceptionCodes;
import com.openexchange.realtime.packet.Stanza;


/**
 * {@link AbstractStrategyHandler}
 *
 * @author <a href="mailto:steffen.templin@open-xchange.com">Steffen Templin</a>
 */
public abstract class AbstractStrategyHandler<T extends Stanza> implements StrategyHandler<T>, Runnable {
    
    protected static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(AbstractStrategyHandler.class);
    
    protected final BlockingQueue<T> queue;
    
    protected final HandlerStrategy<T> strategy;
    
    public AbstractStrategyHandler(BlockingQueue<T> queue, HandlerStrategy<T> strategy) {
        super();
        this.queue = queue;
        this.strategy = strategy;
    }
    
    @Override
    public void run() {
        while (true) {
            try {
                T stanza = queue.take();
                strategy.handleStanza(stanza, this);
            } catch (InterruptedException e) {
                return;
            } catch (Throwable t) {
                LOG.error("Error while handling stanza.", t);
            }
        }
    }
    
    protected ResourceDirectory getResourceDirectory() throws OXException {
        ResourceDirectory resourceDirectory = Services.optService(ResourceDirectory.class);
        if (resourceDirectory == null) {
            throw RealtimeExceptionCodes.NEEDED_SERVICE_MISSING.create(ResourceDirectory.class.getName());
        }

        return resourceDirectory;
    }

    protected MessageDispatcher getMessageDispatcher() throws OXException {
        MessageDispatcher messageDispatcher = Services.optService(MessageDispatcher.class);
        if (messageDispatcher == null) {
            throw RealtimeExceptionCodes.NEEDED_SERVICE_MISSING.create(MessageDispatcher.class.getName());
        }

        return messageDispatcher;
    }
}
