/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2020 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.eventsystem;

import com.openexchange.exception.OXException;

/**
 * {@link EventSystemService} - The event system service.
 * <p>
 * Broadcasts events in cluster either following Point-to-Point Messaging Domain or Publish/Subscribe Messaging Domain.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since 7.4.2
 */
public interface EventSystemService {

    /**
     * Publishes given event to all listeners which subscribed to the topic of the event.
     * <p>
     * <img src="http://docs.oracle.com/javaee/1.3/jms/tutorial/1_3_1-fcs/doc/images/Fig2.3.gif" alt="pub-sub" width="480">
     *
     * @param event The event
     * @throws OXException If event publication fails
     */
    void publish(Event event) throws OXException;

    /**
     * Delivers given event, thus available for exactly one listener that subscribed to the topic of the event.
     * <p>
     * <img src="http://docs.oracle.com/javaee/1.3/jms/tutorial/1_3_1-fcs/doc/images/Fig2.2.gif" alt="p2p" width="480">
     *
     * @param event The event
     * @throws OXException If enqueue operations fails
     */
    void deliver(Event event) throws OXException;

    /**
     * Creates a "synchronized" event handler for specified event handler. <br>
     * Synchronized in terms of only one listener of the same type will be allowed to handle the event.
     *
     * <pre>
     *         EventSystemService eventSystemService = ...;
     *
     *         final Dictionary<String, String> dict = new Hashtable<String, String>(2);
     *         dict.put(EventConstants.EVENT_TOPIC, "some/interesting/topics/*");
     *
     *         final EventHandler synchronizedEventHandler = eventSystemService.synchronizedEventHandler(new MyEventHandler());
     *
     *         registerService(EventHandler.class, synchronizedEventHandler, dict);
     * </pre>
     *
     * @param eventHandler The delegate event handler
     * @return The "synchronized" event handler for specified event handler
     * @throws OXException If generating a "synchronized" event handler fails
     */
    EventHandler synchronizedEventHandler(EventHandler eventHandler) throws OXException;

}
