/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.json.streaming;

import java.text.ParseException;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * {@link AttachmentTempId}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class AttachmentTempId {

    private final static boolean _ENCODE_UNCOMMON_CHARACTERS = false;

    private final static String _PIM_IDENTIFIER = "PIM_";

    private final static String _MAIL_IDENTIFIER = "MAIL_";

    private final static char _DELIMITER = '_';

    private final static String _HEX_CHARS = "0123456789abcdef";

    /**
     * Parses a tempid and extracts the data that was stored by the server. If the tempid is not of the correct format, throws an
     * USMJSONAPIException.
     * 
     * @param s String as generated from AttachmentTempId.toString()
     * @return AttachmentTempId with parsed information
     * @throws USMJSONAPIException if format is not correct or s is null
     */
    public static AttachmentTempId fromString(String s) throws USMJSONAPIException {
        if (s == null)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.TEMPID_NOT_SET,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "tempid was not set");
        String id = null;
        boolean isPIM = false;
        if (s.startsWith(_PIM_IDENTIFIER)) {
            isPIM = true;
            id = s.substring(_PIM_IDENTIFIER.length());
        } else if (s.startsWith(_MAIL_IDENTIFIER)) {
            id = s.substring(_MAIL_IDENTIFIER.length());
        }
        AttachmentTempId result = null;
        if (id != null) {
            try {
                if (id.indexOf("__") < 0) {
                    result = parseSimpleId(id, isPIM);
                } else if (!isPIM) {
                    result = parseComplexId(id);
                }
                if (result != null)
                    return result;
            } catch (ParseException e) {
                // fall through to error
            } catch (NumberFormatException e) {
                // fall through to error
            }
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.INVALID_TEMPID,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Illegal server tempid '" + s + '\'');
    }

    private static AttachmentTempId parseComplexId(String id) throws ParseException {
        StringBuilder folderId = new StringBuilder();
        int pos = parseObjectId(id, 0, folderId);
        StringBuilder objectId = new StringBuilder();
        pos = parseObjectId(id, pos, folderId);
        StringBuilder attachmentId = new StringBuilder();
        pos = parseObjectId(id, pos, attachmentId);
        if (pos >= id.length())
            return null;
        long size = Long.parseLong(id.substring(pos));
        return new AttachmentTempId(false, folderId.toString(), objectId.toString(), attachmentId.toString(), size);
    }

    private static AttachmentTempId parseSimpleId(String id, boolean isPIM) {
        int p1 = id.indexOf(_DELIMITER);
        if (p1 <= 0)
            return null;
        int p2 = id.indexOf(_DELIMITER, p1 + 1);
        if (p2 <= p1)
            return null;
        int p3 = id.indexOf(_DELIMITER, p2 + 1);
        if (p3 <= p2)
            return null;
        String attId = id.substring(p2 + 1, p3);
        if (isPIM) // Verify that attachmentId is an integer value
            Integer.parseInt(attId);
        long size = Long.parseLong(id.substring(p3 + 1));
        return new AttachmentTempId(isPIM, id.substring(0, p1), id.substring(p1 + 1, p2), attId, size);
    }

    private static int parseObjectId(String id, int start, StringBuilder part) throws ParseException {
        for (int i = start; i < id.length(); i++) {
            char c = id.charAt(i);
            if (c == _DELIMITER) {
                int end = i + 1;
                if (end >= id.length() || id.charAt(end) != _DELIMITER)
                    return end;
                if (_ENCODE_UNCOMMON_CHARACTERS) {
                    if (end >= id.length() - 6)
                        throw new ParseException("Can not parse part of '" + id + '\'', start);
                    part.append((char) Integer.parseInt(id.substring(end + 1, end + 5), 16));
                    i += 5;
                } else {
                    part.append(_DELIMITER);
                    i++;
                }
            } else {
                part.append(c);
            }
        }
        return id.length();
    }

    private static void addObjectIdToTempId(StringBuilder sb, String id) {
        for (int i = 0; i < id.length(); i++) {
            char c = id.charAt(i);
            if (_ENCODE_UNCOMMON_CHARACTERS) {
                if (isSimpleChar(c)) {
                    sb.append(c);
                } else {
                    // escape complex characters with __uuuu where uuuu is the unicode in hex
                    sb.append(_DELIMITER).append(_DELIMITER);
                    sb.append(_HEX_CHARS.charAt(c >> 12));
                    sb.append(_HEX_CHARS.charAt((c >> 8) & 0xF));
                    sb.append(_HEX_CHARS.charAt((c >> 4) & 0xF));
                    sb.append(_HEX_CHARS.charAt(c & 0xF));
                }
            } else {
                if (c == _DELIMITER)
                    sb.append(_DELIMITER);
                sb.append(c);
            }
        }
    }

    private static boolean isSimpleChar(char c) {
        return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z');
    }

    private final boolean _pimAttachment;

    private final String _folderId;

    private final String _objectId;

    private final String _attachmentId;

    private final long _size;

    /**
     * Initializes a new {@link AttachmentTempId} from a PIM Object and one of its PIM-Attachments.
     * 
     * @param dataObject
     * @param att
     */
    public AttachmentTempId(DataObject dataObject, PIMAttachment att) {
        this(true, dataObject.getParentFolderID(), dataObject.getID(), Integer.toString(att.getOxId()), att.getFileSize());
    }

    /**
     * Initializes a new {@link AttachmentTempId} from an Email object and the attachment ID of one of its mail attachments TODO bug27000
     * Also store exact length of attachment in tempid, also add matching code to extraction from String representation
     * 
     * @param mailObject
     * @param attachmentId
     * @param attachmentSize
     */
    public AttachmentTempId(DataObject mailObject, String attachmentId, long attachmentSize) {
        this(false, mailObject.getParentFolderID(), mailObject.getID(), attachmentId, attachmentSize);
    }

    private AttachmentTempId(boolean isPIM, String folderId, String objectId, String attachmentId, long size) {
        _pimAttachment = isPIM;
        _folderId = folderId;
        _objectId = objectId;
        _attachmentId = attachmentId;
        _size = size;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (isPIMAttachment())
            sb.append(_PIM_IDENTIFIER);
        else
            sb.append(_MAIL_IDENTIFIER);
        addObjectIdToTempId(sb, _folderId);
        sb.append(_DELIMITER);
        addObjectIdToTempId(sb, _objectId);
        sb.append(_DELIMITER);
        addObjectIdToTempId(sb, _attachmentId);
        sb.append(_DELIMITER);
        sb.append(_size);
        return sb.toString();
    }

    public boolean isPIMAttachment() {
        return _pimAttachment;
    }

    public String getFolderId() {
        return _folderId;
    }

    public String getObjectId() {
        return _objectId;
    }

    public String getAttachmentId() {
        return _attachmentId;
    }

    public long getSize() {
        return _size;
    }
}
