/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.clt;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * {@link JMXUSMSessionInfo} Helper class that parses and stores USM session information as reported by the USM JMX session interface
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class JMXUSMSessionInfo {

    // "CID: " + _cid + ", ID: " + _id + ", Protocol: " + _protocol + ", Device: " + _device + ", usmSessionId: " + _usmSessionId;
    private static final Pattern _SESSION_PATTERN = Pattern.compile("^CID:\\s*(\\d+),\\s*ID:\\s*(\\d+),\\s*Protocol:\\s*(\\w+),\\s*Device:\\s*(\\w+),\\s*usmSessionId:\\s*(\\d+)$");

    // d.addFieldData("    " + f.substring(FIELD_PREFIX.length()) + " = " + result.getString(3));
    private static final Pattern _FIELD_PATTERN = Pattern.compile("^\\s+(\\S+)\\s+=\\s(.*)$");

    private static final SimpleDateFormat _TIMESTAMP_FORMAT = new SimpleDateFormat("EEEE, dd.MM.yyyy HH:mm:ss.SSS z '('Z')'");

    public static List<JMXUSMSessionInfo> fromJMXResult(String[] jmxResult) {
        List<JMXUSMSessionInfo> result = new ArrayList<JMXUSMSessionInfo>();
        JMXUSMSessionInfo currentInfo = null;
        for (String line : jmxResult) {
            Matcher m = _SESSION_PATTERN.matcher(line);
            if (m.matches()) {
                int cid = Integer.parseInt(m.group(1));
                int id = Integer.parseInt(m.group(2));
                int usmSessionId = Integer.parseInt(m.group(5));
                currentInfo = new JMXUSMSessionInfo(cid, id, m.group(3), m.group(4), usmSessionId);
                result.add(currentInfo);
            } else if (currentInfo != null) {
                m = _FIELD_PATTERN.matcher(line);
                if (m.matches())
                    currentInfo.addField(m.group(1), m.group(2));
            }
        }
        return result;
    }

    private final int _cid;

    private final int _id;

    private final String _protocol;

    private final String _device;

    private final int _usmSessionId;

    private final SortedMap<String, String> _fields = new TreeMap<String, String>();

    private JMXUSMSessionInfo(int cid, int id, String protocol, String device, int usmSessionId) {
        _cid = cid;
        _id = id;
        _protocol = protocol;
        _device = device;
        _usmSessionId = usmSessionId;
    }

    private void addField(String key, String value) {
        if (isTimestampField(key)) {
            try {
                _fields.put(key, value + " (" + _TIMESTAMP_FORMAT.format(new Date(Long.parseLong(value.trim()))) + ')');
                return;
            } catch (Exception ignored) {
                // fall through
            }
        }
        _fields.put(key, value);
    }

    private static boolean isTimestampField(String key) {
        return key.equals("USM_ACCESS") || key.equals("eas.FirstAccess") || key.equals("eas.LastAccess");
    }

    public int getCID() {
        return _cid;
    }

    public int getID() {
        return _id;
    }

    public String getProtocol() {
        return _protocol;
    }

    public String getDevice() {
        return _device;
    }

    public int getUsmSessionId() {
        return _usmSessionId;
    }

    public SortedMap<String, String> getFields() {
        return Collections.unmodifiableSortedMap(_fields);
    }

    @Override
    public String toString() {
        return "CID: " + _cid + ", ID: " + _id + ", Protocol: " + _protocol + ", Device: " + _device + ", usmSessionId: " + _usmSessionId;
    }
}
