/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.DESTINATIONFOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.DESTINATIONSYNCID;
import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SOURCEFOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.SOURCESYNCID;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_DETAILS;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_MESSAGE;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_STATUS;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.api.session.SynchronizationConflictException;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.InvalidUUIDException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Handler for the MoveItems USM-JSON-Command.
 * @author ldo
 *
 */
public class MoveItemsHandler extends SyncCommandHandler {

	private static final String ERRORS = "errors";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SOURCEFOLDERID, SOURCESYNCID, DESTINATIONFOLDERID,
			DESTINATIONSYNCID, OBJECTIDS };
	private static final String[] OPTIONAL_PARAMETERS = {};

	private ErrorStatusCode _errorStatus = ErrorStatusCode.OTHER;

	private long _originalSyncID;

    public MoveItemsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        String sourceFolderUUID = getStringParameter(SOURCEFOLDERID);
        String destinationFolderUUID = getStringParameter(DESTINATIONFOLDERID);

        String acquirer = "moveItems:" + sourceFolderUUID + " -> " + destinationFolderUUID;
        String previousAcquirer = _session.tryLock(sourceFolderUUID, acquirer);
        if (previousAcquirer != null)
            return createCannotLockFolderResponse("moveItems", "source ", sourceFolderUUID, previousAcquirer);
        try {
            previousAcquirer = _session.tryLock(destinationFolderUUID, acquirer);
            if (previousAcquirer != null)
                return createCannotLockFolderResponse("moveItems", "destination ", destinationFolderUUID, previousAcquirer);
            try {
                Folder sourceFolder = getFolderByUUID(sourceFolderUUID);
                Folder destinationFolder = getFolderByUUID(destinationFolderUUID);
                long sourceFolderSyncID = getLong(_parameters, SOURCESYNCID);
                long destinationSyncID = getLong(_parameters, DESTINATIONSYNCID);
                if (sourceFolderUUID.equals(destinationFolderUUID))
                    throw new USMJSONAPIException(
                        ConnectorBundleErrorCodes.MOVE_ITEMS_SAME_SOURCE_DEST,
                        ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                        generateErrorMessage(
                            "Source and Destination are the same.",
                            sourceFolder,
                            sourceFolderSyncID,
                            destinationFolder,
                            destinationSyncID));
                JSONArray objectsToMove = getJSONArray(_parameters, OBJECTIDS);
                // move items
                JSONObject response = moveItems(sourceFolder, sourceFolderSyncID, destinationFolder, destinationSyncID, objectsToMove);
                return new ResponseObject(ResponseStatusCode.SUCCESS, response);
            } finally {
                _session.unlock(destinationFolderUUID);
            }
        } finally {
            _session.unlock(sourceFolderUUID);
        }
    }

    private boolean checkExisting(Folder folder) {
		BitSet fields = new BitSet();
		try {
			folder.getContentType().getTransferHandler().readFolder(folder, fields);
		} catch (USMException e) {
			return false;
		}
		return true;
	}

	private String generateErrorMessage(String message, Folder source, long sourceSyncID, Folder dest, long destSyncID) {
		return message + '(' + source.getUUID() + '/' + source.getID() + ':' + sourceSyncID + "->" + dest.getUUID()
				+ '/' + dest.getID() + ':' + destSyncID + ')';
	}

	private JSONObject moveItems(Folder sourceFolder, long sourceFolderSyncID, Folder destinationFolder,
			long destinationSyncID, JSONArray objectsToMove) throws USMJSONAPIException {
		JSONObject responseData = new JSONObject();
		DataObject[] changes = null;
		try {
			if (!sourceFolder.getElementsContentTypeID().equals(destinationFolder.getElementsContentTypeID()))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_NOT_MATCHING_FOLERS,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, generateErrorMessage(
								"Source and Destination are not of same type.", sourceFolder, sourceFolderSyncID,
								destinationFolder, destinationSyncID));
			DataObject[] cachedSourceElements = _session.getCachedFolderElements(sourceFolder.getID(),
					sourceFolder.getElementsContentType(), sourceFolderSyncID);
            if (cachedSourceElements == null)
                throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_INVALID_SYNC_ID,
                        ResponseStatusCode.UNKNOWN_SYNCID, generateErrorMessage("Unknown source SyncID", sourceFolder,
                                sourceFolderSyncID, destinationFolder, destinationSyncID));
            DataObjectSet sourceElements = new DataObjectSet(cachedSourceElements);
			DataObject[] destinationElements = _session.getCachedFolderElements(destinationFolder.getID(),
					destinationFolder.getElementsContentType(), destinationSyncID);
			if (destinationElements == null)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_INVALID_SYNC_ID_2,
						ResponseStatusCode.UNKNOWN_SYNCID, generateErrorMessage("Unknown destination SyncID",
								sourceFolder, sourceFolderSyncID, destinationFolder, destinationSyncID));
			_originalSyncID = sourceFolderSyncID;
			List<DataObject> changesList = new ArrayList<DataObject>();
			int length = objectsToMove.length();
            for (int i = 0; i < length; i++) {
                String uuidString = getString(objectsToMove, i);
                try {
                    DataObject object = getDataObjectByUUID(sourceElements, extractUUIDFromString(uuidString));
                    object.setParentFolder(destinationFolder);
                    changesList.add(object);
                } catch (DataObjectNotFoundException e) {
                    addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                } catch (MultipleOperationsOnDataObjectException e) {
                    addError(uuidString, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
                    removeDuplicatesAndAddToError(changesList, uuidString, e);
                } catch (InvalidUUIDException e) {
                    addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                }
            }
			changes = new DataObject[changesList.size()];
			changes = changesList.toArray(changes);
			SyncResult syncResult = _session.syncChangesWithServer(sourceFolder.getID(), sourceFolderSyncID, -1, null,
					false, ConflictResolution.ERROR, changes);
			initializeErrorMap(syncResult);
			DataObject[] newSourceState = getNewSourceStateToSave(sourceFolder, sourceFolderSyncID, changes);
			long newSourceSyncID = _session.storeSyncState(sourceFolderSyncID, syncResult.getTimestamp(),
					sourceFolder.getID(), newSourceState);
			responseData.put(SOURCESYNCID, newSourceSyncID);

			logSyncData(" moveItems", sourceFolder.getUUID().toString(), sourceFolderSyncID, newSourceSyncID,
					syncResult.getTimestamp());

			if (!_errorMap.isEmpty()) {
				JSONObject notmoved = new JSONObject();
				//reset to other
				_errorStatus = checkFolderExistance(sourceFolder, destinationFolder);
				int destinationRigths = getOwnRights(destinationFolder);
				int sourceRights = getOwnRights(sourceFolder);
				addErrorsToErrorResponse(notmoved, destinationRigths, sourceRights);
				responseData.put(ERRORS, notmoved);
			}
			_originalSyncID = destinationSyncID;
			if (_errorStatus != ErrorStatusCode.DESTINATION_NOT_EXIST) {
				DataObject[] newDestinationState = getNewDestinationStateToSave(destinationFolder, destinationSyncID,
						changes);
				long newDestinationSyncID = _session.storeSyncState(destinationSyncID, destinationSyncID,
						destinationFolder.getID(), newDestinationState);
				responseData.put(DESTINATIONSYNCID, newDestinationSyncID);

				logSyncData(" moveItems", destinationFolder.getUUID().toString(), destinationSyncID,
						newDestinationSyncID, 0L);
			} else {
				responseData.put(DESTINATIONSYNCID, destinationSyncID);
			}
		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMSQLException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR, e);
		} catch (DatabaseAccessException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_2, e);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (FolderNotFoundException e) {
			return createErrorResponse(sourceFolderSyncID, destinationSyncID, e, sourceFolder, destinationFolder,
					changes);
		} catch (OXCommunicationException e) {
			return createErrorResponse(sourceFolderSyncID, destinationSyncID, e, sourceFolder, destinationFolder,
					changes);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_3, e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.MOVE_ITEMS_JSON_ERROR, e);
		}
		return responseData;
	}

	private void removeDuplicatesAndAddToError(List<DataObject> changesList, String uuidString, MultipleOperationsOnDataObjectException e) {
	    for(Iterator<DataObject> i = changesList.iterator(); i.hasNext();) {
	        DataObject o = i.next();
	        if(o.getUUID().toString().equals(uuidString)) {
	            i.remove();
                addError(uuidString, o, e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
	        }
	    }
    }

    private void addErrorsToErrorResponse(JSONObject notmoved, int destinationRigths, int sourceRights) throws JSONException {
        for (Map.Entry<String, SyncErrorData> entry : _errorMap.entrySet()) {
            String uuid = entry.getKey();
            SyncErrorData data = entry.getValue();
            addErrorToNotMovedArray(sourceRights, destinationRigths, notmoved, uuid, data.getObject(), data._error);
        }
    }

	private void addErrorToNotMovedArray(int sourceRights, int destinationRights, JSONObject notmoved, String uuid,
			DataObject dataObject, USMException exception) throws JSONException {
		JSONObject errorObject = new JSONObject();
		JSONObject details = exception.getErrorDetailsForJSONResponse();
		if (details == null)
			details = exception.getOxErrorForJSONResponse();
		if (details == null) {
			details = new JSONObject();
			details.put(ERROR_MESSAGE, exception.getMessage());
		}
		errorObject.put(ERROR_DETAILS, details);
		errorObject.put(ERROR_STATUS, getErrorStatusCode(destinationRights, sourceRights, dataObject, exception)
				.getStatusCode());
		notmoved.put(uuid, errorObject);
	}

	private JSONObject createErrorResponse(long sourceFolderSyncID, long destinationSyncID, USMException e,
			Folder sourceFolder, Folder destinationFolder, DataObject[] changes) {
		JSONObject errorResponse = new JSONObject();
		try {
			errorResponse.put(SOURCESYNCID, sourceFolderSyncID);
			JSONObject notmoved = new JSONObject();
			//reset to other
			_errorStatus = checkFolderExistance(sourceFolder, destinationFolder);
			int destinationRigths = getOwnRights(destinationFolder);
			int sourceRights = getOwnRights(sourceFolder);
			for (int i = 0; changes != null && i < changes.length; i++) {
				addErrorToNotMovedArray(sourceRights, destinationRigths, notmoved, changes[i].getUUID().toString(),
						changes[i], e);
			}
			addErrorsToErrorResponse(notmoved, destinationRigths, sourceRights);
			errorResponse.put(ERRORS, notmoved);
			errorResponse.put(DESTINATIONSYNCID, destinationSyncID);
		} catch (JSONException ignored) {
			//do nothing
		}
		return errorResponse;
	}

	private ErrorStatusCode getErrorStatusCode(int destinationRigths, int sourceRights, DataObject errorDataObject,
			USMException exception) {
	    if(_errorStatus != ErrorStatusCode.OTHER)
	        return _errorStatus;
		if (errorDataObject != null) {
			if (!checkCreatePermissionsAllObjects(destinationRigths))
				return ErrorStatusCode.DESTINATION_NO_PERMISSION;
			if (isOwnObject(errorDataObject) && !checkDeletePermissionsOwnObjects(sourceRights))
				return ErrorStatusCode.SOURCE_NO_PERMISSION_OWN_OBJECTS;
			if (!checkDeletePermissionsAllObjects(sourceRights))
				return ErrorStatusCode.SOURCE_NO_PERMISSION_ALL_OBJECTS;
		}
		if (isUnknownUUIDError(exception))
			return ErrorStatusCode.UNKNOWN_UUID;
		return ErrorStatusCode.OTHER;
	}

	private ErrorStatusCode checkFolderExistance(Folder sourceFolder, Folder destinationFolder) {
		//check special error cases
		if (!checkExisting(destinationFolder))
			return ErrorStatusCode.DESTINATION_NOT_EXIST;
		if (!checkExisting(sourceFolder))
			return ErrorStatusCode.SOURCE_NOT_EXIST;
		return ErrorStatusCode.OTHER;
	}

	private DataObject[] getNewDestinationStateToSave(Folder folder, long syncID, DataObject[] requestedChanges)
			throws USMException {
		DataObject[] oldState = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(),
				syncID);
		List<DataObject> resultList = new ArrayList<DataObject>();
		resultList.addAll(Arrays.asList(oldState));
		for (DataObject o : requestedChanges) {
			if (hasNoError(o)) {
				if (o.getContentType().getCode() != DefaultContentTypes.MAIL_CODE)
					o.setFieldContent(UtilConstants.MODIFIED_BY, _session.getUserIdentifier());
				o.commitChanges();
				resultList.add(o);
			}
		}
		return resultList.toArray(new DataObject[resultList.size()]);
	}

	private DataObject[] getNewSourceStateToSave(Folder folder, long syncID, DataObject[] requestedChanges)
			throws USMException {
		DataObject[] oldState = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(),
				syncID);
		List<DataObject> resultList = new ArrayList<DataObject>();
		resultList.addAll(Arrays.asList(oldState));
		for (DataObject dataObject : requestedChanges) {
			String uuidString = dataObject.getUUID().toString();
			if (!_errorMap.containsKey(uuidString)) {
				for (DataObject oldObject : oldState) {
					if (oldObject.getID().equals(dataObject.getOriginalID()))
						resultList.remove(oldObject);
				}
			}
		}
		return resultList.toArray(new DataObject[resultList.size()]);
	}

    private boolean isUnknownUUIDError(USMException exception) {
        return (exception instanceof DataObjectNotFoundException) || (exception instanceof InvalidUUIDException);
    }

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

	@Override
	protected long getOriginalSyncID() {
		return _originalSyncID;
	}
}
