/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.realtime.payload.converter.impl;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.conversion.simple.SimpleConverter;
import com.openexchange.exception.OXException;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.payload.converter.PayloadTreeConverter;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.server.ServiceLookup;

/**
 * {@link DefaultPayloadTreeConverter}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class DefaultPayloadTreeConverter implements PayloadTreeConverter {

    private final Map<ElementPath, String> preferredFormats;
    private final ServiceLookup services;

    public DefaultPayloadTreeConverter(ServiceLookup services) {
        super();
        preferredFormats = new ConcurrentHashMap<ElementPath, String>();
        this.services = services;
    }

    @Override
    public PayloadTree incoming(PayloadTree payloadTree) throws OXException {

        return new PayloadTree(incoming(payloadTree.getRoot()));

    }

    @Override
    public PayloadTreeNode incoming(PayloadTreeNode node) throws OXException {
        ElementPath elementPath = node.getElementPath();
        String format = preferredFormats.get(elementPath);
        if (format == null) {
            format = "json";
        }

        PayloadElement payloadElement = node.getPayloadElement();

        SimpleConverter service = services.getService(SimpleConverter.class);
        if (null == service) {
            throw ServiceExceptionCode.serviceUnavailable(SimpleConverter.class);
        }

        Object transformed = service.convert(
            payloadElement.getFormat(),
            format,
            payloadElement.getData(),
            null);

        PayloadElement transformedPayload = new PayloadElement(
            transformed,
            format,
            payloadElement.getNamespace(),
            payloadElement.getElementName());

        PayloadTreeNode resultNode = new PayloadTreeNode(transformedPayload);

        if (node.hasChildren()) {
            for (PayloadTreeNode child : node.getChildren()) {
                PayloadTreeNode transformedChild = incoming(child);
                resultNode.addChild(transformedChild);
            }
        }
        return resultNode;

    }

    @Override
    public PayloadTree outgoing(PayloadTree payloadTree, String format) throws OXException {
        return new PayloadTree(outgoing(payloadTree.getRoot(), format));
    }

    @Override
    public PayloadTreeNode outgoing(PayloadTreeNode node, String format) throws OXException {

        PayloadElement payloadElement = node.getPayloadElement();

        SimpleConverter service = services.getService(SimpleConverter.class);
        if (null == service) {
            throw ServiceExceptionCode.serviceUnavailable(SimpleConverter.class);
        }

        Object transformed = service.convert(
            payloadElement.getFormat(),
            format,
            payloadElement.getData(),
            null);

        PayloadElement transformedPayload = new PayloadElement(
            transformed,
            format,
            payloadElement.getNamespace(),
            payloadElement.getElementName());

        PayloadTreeNode resultNode = new PayloadTreeNode(transformedPayload);

        if (node.hasChildren()) {
            for (PayloadTreeNode child : node.getChildren()) {
                PayloadTreeNode transformedChild = outgoing(child, format);
                resultNode.addChild(transformedChild);
            }
        }
        return resultNode;
    }

    @Override
    public void declarePreferredFormat(ElementPath path, String format) {
        preferredFormats.put(path, format);
    }

}
