/*
 * Copyright 2007 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.oauth;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Properties of an OAuth Consumer. Properties may be added freely, e.g. to
 * support extensions.
 * 
 * @author John Kristian
 */
public class OAuthConsumer implements Serializable {

    private static final long serialVersionUID = -2258581186977818580L;

    public final String callbackURL;
    public final String consumerKey;
    public final String consumerSecret;
    public final OAuthServiceProvider serviceProvider;
    private final Map<String, Object> properties;

    public OAuthConsumer(final String callbackURL, final String consumerKey,
            final String consumerSecret, final OAuthServiceProvider serviceProvider) {
        super();
        properties = new ConcurrentHashMap<String, Object>();
        this.callbackURL = callbackURL;
        this.consumerKey = consumerKey;
        this.consumerSecret = consumerSecret;
        this.serviceProvider = serviceProvider;
    }


    /**
     * Gets the property associated with given name.
     * 
     * @param name The name
     * @return The value or <code>null</code> if absent
     */
    @SuppressWarnings("unchecked")
    public <V> V getProperty(final String name) {
        try {
            return (V) properties.get(name);
        } catch (final ClassCastException e) {
            return null;
        }
    }

    /**
     * Sets specified property.
     * <p>
     * If value is <code>null</code>, a remove is performed.
     * 
     * @param name The name
     * @param value The value (possibly <code>null</code>)
     */
    public void setProperty(final String name, final Object value) {
        if (null == value) {
            properties.remove(name);
        } else {
            properties.put(name, value);
        }
    }

    /**
     * Gets the properties associated with this consumer.
     * 
     * @return The properties' iterator
     */
    public Iterator<Map.Entry<String, Object>> getProperties() {
        return OAuth.unmodifiableIterator(properties.entrySet().iterator());
    }

    /**
     * The name of the property whose value is the Accept-Encoding header in
     * HTTP requests.
     */
    public static final String ACCEPT_ENCODING = "HTTP.header.Accept-Encoding";

    /**
     * The name of the property whose value is the <a
     * href="http://oauth.pbwiki.com/AccessorSecret">Accessor Secret</a>.
     */
    public static final String ACCESSOR_SECRET = "oauth_accessor_secret";

}
