/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.dataobject;

import java.io.Serializable;
import java.util.BitSet;
import java.util.Map;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.USMDebugConstants;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.session.ChangeState;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.ProtocolInformation;
import com.openexchange.usm.api.session.Session;

/**
 * Base implementation of DataObject. It stores ID and ParentFolderID directly in
 * fields provided by the ContentType, by default "id" for ID and "folder_id" for
 * ParentFolderID. If a ContentType uses other field names for that information,
 * it can  use the constructor with additional parameters to specify the correct
 * field names.
 *
 * @author afe
 *
 */
public class SimpleDataObject implements DataObject {
	private final static int ID_INDEX = 0;
	private final static int PARENT_FOLDER_ID_INDEX = 1;

	static boolean _formatToString = false;
	protected final static boolean _skipUnsetValuesForToString = false;

	public static void setToStringFormatted(boolean formatToString) {
		_formatToString = formatToString;
	}

	protected final Session _session;

	protected final ContentType _contentType;

	protected final Map<String, Integer> _fieldNameToIndexMap;

	protected final int _lastModifiedIndex;
	
	protected ChangeState _changeState = ChangeState.CREATED;

	protected final Object[] _originalFieldContent;

	protected final Object[] _fieldContent;

	protected long _timestamp, _originalTimestamp;

	private boolean _failed;

	private UUID _uuid;

	private DataObject _source;

	private Folder _parentFolder;

	private ProtocolInformation _protocolInfo;

	private int _parentFolderOwnerID;

	public SimpleDataObject(Session session, ContentType contentType) {
		if (session == null || contentType == null)
			throw new IllegalArgumentException("Session and ContentType must be specified");
		_source = this;
		_session = session;
		_contentType = contentType;
		ContentTypeField[] fields = _contentType.getFields();
		int fieldCount = fields.length;
		_originalFieldContent = new Object[fieldCount];
		_fieldContent = new Object[fieldCount];
		for (int i = 0; i < fieldCount; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			_originalFieldContent[i] = fieldType.createNewObject();
			_fieldContent[i] = fieldType.createNewObject();
		}
		_fieldNameToIndexMap = FieldMapStorageImpl.getInstance().getFieldNameToIndexMap(contentType);
		Integer lastModifiedIndex = _fieldNameToIndexMap.get("last_modified");
		_lastModifiedIndex = (lastModifiedIndex != null) ? lastModifiedIndex.intValue() : -1;
	}

	protected SimpleDataObject(SimpleDataObject source, boolean linkUUIDs) {
		_source = linkUUIDs ? source._source : this;
		_uuid = source._uuid;
		_protocolInfo = source._protocolInfo;
		_session = source._session;
		_contentType = source._contentType;
		_fieldNameToIndexMap = source._fieldNameToIndexMap;
		_lastModifiedIndex = source._lastModifiedIndex;
		_changeState = source._changeState;
		_timestamp = source._timestamp;
		_originalTimestamp = source._originalTimestamp;
		_failed = source._failed;
		_fieldContent = new Object[source._fieldContent.length];
		_originalFieldContent = new Object[source._originalFieldContent.length];
		_parentFolderOwnerID = source._parentFolderOwnerID;
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < _fieldContent.length; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			_fieldContent[i] = fieldType.createCopy(source._fieldContent[i]);
			_originalFieldContent[i] = fieldType.createCopy(source._originalFieldContent[i]);
		}
	}

	public Session getSession() {
		return _session;
	}

	public ContentType getContentType() {
		return _contentType;
	}

	public ChangeState getChangeState() {
		return _changeState;
	}

	public void setChangeState(ChangeState newState) {
		if (newState != ChangeState.CREATED && newState != ChangeState.DELETED)
			throw new IllegalArgumentException("Changing from " + _changeState + " to " + newState + " not permitted");
		_changeState = newState;
	}

	public String getID() {
		return convertObjectToString((getFieldContent(ID_INDEX)));
	}

	public String getOriginalID() {
		return convertObjectToString((getOriginalFieldContent(ID_INDEX)));
	}

	public void setID(String id) {
		setFieldContent(ID_INDEX, id);
	}

	public long getTimestamp() {
		return _timestamp;
	}

	public long getOriginalTimestamp() {
		return _originalTimestamp;
	}

	public void setTimestamp(long timestamp) {
		_timestamp = timestamp;
		updateChangeState(_timestamp != _originalTimestamp);
	}

	public String getParentFolderID() {
		ensureCorrectParentFolderID();
		return convertObjectToString((getFieldContent(PARENT_FOLDER_ID_INDEX)));
	}

	public String getOriginalParentFolderID() {
		return convertObjectToString(getOriginalFieldContent(PARENT_FOLDER_ID_INDEX));
	}

	public void setParentFolderID(String folderID) {
		setFieldContent(PARENT_FOLDER_ID_INDEX, folderID);
	}

	public Object getFieldContent(int fieldIndex) {
		return _fieldContent[fieldIndex];
	}

	public Object getOriginalFieldContent(int fieldIndex) {
		return _originalFieldContent[fieldIndex];
	}

	public void setFieldContent(int fieldIndex, Object data) {
		_fieldContent[fieldIndex] = _contentType.getFields()[fieldIndex].getFieldType().checkValue(data);
		if (fieldIndex == PARENT_FOLDER_ID_INDEX)
			_parentFolder = null;
		updateChangeState(fieldIndex);
	}

	public Object getFieldContent(String fieldName) {
		return getFieldContent(getFieldIndex(fieldName));
	}

	public Object getOriginalFieldContent(String fieldName) {
		return getOriginalFieldContent(getFieldIndex(fieldName));
	}

	public void setFieldContent(String fieldName, Object data) {
		setFieldContent(getFieldIndex(fieldName), data);
	}

	public void setParentFolder(Folder folder) {
		setParentFolderID(folder.getID());
		_parentFolder = folder;
	}

	public Folder getParentFolder() {
		return _parentFolder;
	}

	public boolean isFieldModified(int fieldIndex) {
		return !_contentType.getFields()[fieldIndex].getFieldType().isEqual(_fieldContent[fieldIndex],
				_originalFieldContent[fieldIndex]);
	}

	public boolean isFieldModified(String fieldName) {
		return isFieldModified(getFieldIndex(fieldName));
	}

	public boolean isModified() {
		return _changeState != ChangeState.UNMODIFIED;
	}

	public void rollbackChanges() {
		_parentFolder = null;
		_timestamp = _originalTimestamp;
		System.arraycopy(_originalFieldContent, 0, _fieldContent, 0, _fieldContent.length);
		_changeState = ChangeState.UNMODIFIED;
		_failed = false;
	}

	public void commitChanges() {
		ensureCorrectParentFolderID();
		_originalTimestamp = _timestamp;
		System.arraycopy(_fieldContent, 0, _originalFieldContent, 0, _fieldContent.length);
		_changeState = ChangeState.UNMODIFIED;
		_failed = false;
	}

	protected void updateChangeState(boolean newFieldContentIsNotEqual) {
		if (newFieldContentIsNotEqual) {
			if (_changeState == ChangeState.UNMODIFIED)
				_changeState = ChangeState.MODIFIED;
		} else {
			if (_changeState == ChangeState.MODIFIED && _timestamp == _originalTimestamp) {
				for (int i = 0; i < _fieldContent.length; i++) {
					if (isFieldModified(i))
						return;
				}
				_changeState = ChangeState.UNMODIFIED;
			}
		}
	}

	protected void updateChangeState(int fieldIndex) {
		updateChangeState(isFieldModified(fieldIndex));
	}

	public int getFieldIndex(String fieldName) {
		Integer index = _fieldNameToIndexMap.get(fieldName);
		if (index != null)
			return index;
		throw new IllegalArgumentException("Unknown field " + fieldName + " in " + _contentType.getID());
	}

	@Override
	public String toString() {
		String extended = "";
		if (USMDebugConstants.SIMPLE_DATA_OBJECT_EXTENDED_TOSTRING) {
			try {
				extended += "'" + getFieldContent("title") + "', ";
				extended += "'" + getUUID() + "'|";
			} catch (RuntimeException ignored) {
				// ignore - no extended toString available
			}
		}
		return extended + toString(null, _skipUnsetValuesForToString);
	}

	public String toString(BitSet requestedFields) {
		return toString(requestedFields, _skipUnsetValuesForToString);
	}

	public String toString(BitSet requestedFields, boolean skipUnsetFields) {
		StringBuilder sb = new StringBuilder(1000);
		sb.append(getDataObjectName()).append('(').append(_session).append(',').append(_contentType.getID())
				.append(',').append(_changeState)/*.append(',').append(_uuid)*/.append("){");
		if (_formatToString)
			sb.append("\n   ");
		sb.append("Timestamp:").append(_originalTimestamp);
		if (_timestamp != _originalTimestamp)
			sb.append("=>").append(_timestamp);
		addSpecialFieldsToStringBuilder(sb);
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < fields.length; i++) {
			if (requestedFields == null || requestedFields.get(i)) {
				DataType<?> fieldType = fields[i].getFieldType();
				addElementToStringBuilder(fieldType, sb, fields[i].getFieldName(), _fieldContent[i],
						_originalFieldContent[i], skipUnsetFields);
			}
		}
		if (_formatToString)
			sb.append('\n');
		return sb.append(",data:").append(getProtocolInformation()).append('}').toString();
	}

	protected void addSpecialFieldsToStringBuilder(StringBuilder sb) {
		// This method can be overwritten by sub-classes to add other special fields in the toString()-method
	}

	/**
	 * This can be overwritten by sub classes to provide information about implementing class in toString()-method
	 * @return
	 */
	protected String getDataObjectName() {
		return getClass().getSimpleName();
	}

	protected void addElementToStringBuilder(DataType<?> type, StringBuilder sb, String name, Object current,
			Object old, boolean skipUnsetFields) {
		boolean changed = !type.isEqual(current, old);
		if (skipUnsetFields && !changed && type.isEqual(type.createNewObject(), current))
			return;
		sb.append(',');
		if (_formatToString)
			sb.append("\n   ");
		sb.append(name).append(':');
		appendValueToStringBuilder(type, sb, old);
		if (changed) {
			sb.append("=>");
			appendValueToStringBuilder(type, sb, current);
		}
	}

	protected void appendValueToStringBuilder(DataType<?> type, StringBuilder sb, Object v) {
		JSONArray a = new JSONArray();
		try {
			type.addToJSONArray(_session, a, v);
			Object val = a.opt(0);
			if (val instanceof JSONArray)
				sb.append(((JSONArray) val).toString(0).replaceAll("\n", ""));
			else if (val instanceof JSONObject)
				sb.append(((JSONObject) val).toString(0).replaceAll("\n", ""));
			else {
				String s = String.valueOf(val);
				if (s.length() > 0 && s.charAt(0) == '"' && s.endsWith("\""))
					sb.append(s.substring(1, s.length() - 1));
				else
					sb.append(s);
			}
		} catch (JSONException e) {
			sb.append("<JSON_ERROR>");
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		String id = getID();
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result + ((_session == null) ? 0 : _session.hashCode());
		return result;
	}

	/**
	 * Checks for equality. This test does *not* check if timestamps or attached ProtocolInformation is the same, those are ignored for the comparison !
	 * 
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SimpleDataObject other = (SimpleDataObject) obj;
		return internalEquals(other, false, true);
	}

	/**
	 * Checks for equality without checking the ID. This test does *not* check if timestamps or attached ProtocolInformation is the same, those are ignored for the comparison !
	 * 
	 */
	public boolean equalsWithoutID(DataObject other) {
		if (!(other instanceof SimpleDataObject))
			return false;
		return internalEquals((SimpleDataObject) other, true, true);
	}
	
	public boolean equalsWithLastModified(DataObject other) {
        if (!(other instanceof SimpleDataObject))
            return false;
        return internalEquals((SimpleDataObject) other, false, false);
	}

    private boolean internalEquals(SimpleDataObject other, boolean ignoreID, boolean ignoreLastModified) {
        if (!_changeState.equals(other._changeState))
            return false;
        if (!_contentType.equals(other._contentType))
            return false;
        if (!_session.equals(other._session))
            return false;
        ContentTypeField[] fields = _contentType.getFields();
        for (int i = 0; i < _fieldContent.length; i++) {
            if (ignoreLastModified && i == _lastModifiedIndex)
                continue;
            if (ignoreID && i == ID_INDEX)
                continue;
            DataType<?> type = fields[i].getFieldType();
            if (!type.isEqual(_fieldContent[i], other._fieldContent[i]))
                return false;
            if (!type.isEqual(_originalFieldContent[i], other._originalFieldContent[i]))
                return false;
        }
        return true;
    }

	public DataObject createCopy(boolean linkUUIDs) {
		return new SimpleDataObject(this, linkUUIDs);
	}

	protected String convertObjectToString(Object o) {
		return (o == null) ? null : o.toString();
	}

	public boolean isFailed() {
		return _failed;
	}

	public void setFailed(boolean failed) {
		_failed = failed;
	}

	private void ensureCorrectParentFolderID() {
		if (_parentFolder != null) // If parent folder is set via object, make sure that the field contains the current value of the id of the linked folder
			setParentFolder(_parentFolder);
	}

	public Serializable[] serialize() {
		ensureCorrectParentFolderID();
		Serializable[] result = new Serializable[_fieldContent.length + 2];
		ContentTypeField[] fields = _contentType.getFields();
		result[0] = _source.getProtocolInformation();
		result[1] = _source.getUUID();
		for (int i = 0; i < _fieldContent.length; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			result[i + 2] = fieldType.serialize(_fieldContent[i]);
		}
		return result;
	}

	public void deserialize(long timestamp, Serializable[] data) throws DeserializationFailedException {
		_changeState = ChangeState.UNMODIFIED;
		_timestamp = timestamp;
		_originalTimestamp = timestamp;
		_failed = false;
		_parentFolder = null;
		if (data.length == 0)
			throw new DeserializationFailedException(DataObjectErrorCodes.DESERIALIZATION_FIELDS_MISSING,
					"Invalid size of serialized data for " + _contentType.getID());
		int indexForDes = 0;
		if (data[0] instanceof UUID) {
			_protocolInfo = null;
			_uuid = (UUID) data[0];
			indexForDes = 1;
		} else if ((data[0] instanceof ProtocolInformation) || (data.length > 1 && data[1] instanceof UUID)) {
			_protocolInfo = data[0] instanceof ProtocolInformation ? (ProtocolInformation) data[0] : null;
			_uuid = data[1] instanceof UUID ? (UUID) data[1] : null;
			indexForDes = 2;
		} else { // For compatibility with older USM versions where no UUID was stored with each object
			_uuid = UUID.randomUUID();
		}
		_contentType.deserializeData(data, indexForDes, _fieldContent);
		System.arraycopy(_fieldContent, 0, _originalFieldContent, 0, _fieldContent.length);
	}

	public UUID getUUID() {
		return (_source == this) ? _uuid : _source.getUUID();
	}

	public void setUUID(UUID uuid) {
		if (_source == this)
			_uuid = uuid;
		else
			_source.setUUID(uuid);
	}

	public void linkUUIDTo(DataObject source) {
		if (source == null)
			source = this;
		if (source == this)
			_uuid = _source.getUUID();
		_source = source;
	}

	public ProtocolInformation getProtocolInformation() {
		return (_source == this) ? _protocolInfo : _source.getProtocolInformation();
	}

	public void setProtocolInformation(ProtocolInformation info) {
		if (_source == this)
			_protocolInfo = info;
		else
			_source.setProtocolInformation(info);
	}

	public int getParentFolderOwnerID() {
		return _parentFolderOwnerID;
	}

	public void setParentFolderOwnerID(int id) {
		_parentFolderOwnerID = id;
	}
}
