/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Handler for the USM-invalidateObjects command.
 * @author ldo
 *
 */

public class InvalidateObjectsHandler extends NormalCommandHandler {

	private static final String NOT_FOUND = "not found";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SYNCID };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDERID, OBJECTIDS };

    public InvalidateObjectsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		long syncID = getSyncID();
		String folderUUID = getStringParameter(FOLDERID, null);
		JSONArray objectUUIDs = _parameters.has(OBJECTIDS) ? getJSONArray(_parameters, OBJECTIDS) : new JSONArray();
		JSONObject resultData = new JSONObject();

		try {
			DataObject[] objects = null;
			String folderID = null;
			if (folderUUID == null) {
				//client has sent folders in the objects list
				objects = _session.getCachedFolders(syncID);
			} else {
				try {
					Folder folder = getFolderByUUID(folderUUID);
					folderID = folder.getID();
					objects = _session.getCachedFolderElements(folderID, folder.getElementsContentType(), syncID);
				} catch (USMJSONAPIException e) {
					resultData.put(folderUUID, NOT_FOUND);
					JSONObject response = new JSONObject();
					response.put("syncid", syncID);
					response.put("objects", resultData);
					return new ResponseObject(ResponseStatusCode.BAD_REQUEST, response);
				}
			}
			DataObjectSet objectsSet = new DataObjectSet(objects);
			int length = objectUUIDs.length();
			for (int i = 0; i < length; i++) {
				String uuid = objectUUIDs.getString(i);
				DataObject removedObj = objectsSet.remove(UUID.fromString(uuid));
				resultData.put(uuid, (removedObj == null) ? NOT_FOUND : "ok");
			}
			long newSyncID = (folderUUID == null) ? _session.storeSyncState(syncID, syncID, objectsSet.toArray())
					: _session.storeSyncState(syncID, syncID, folderID, objectsSet.toArray());
			JSONObject response = new JSONObject();
			response.put("syncid", newSyncID);
			response.put("objects", resultData);
			logSyncData(" invalidateObjects", folderUUID, syncID, newSyncID, 0L);
			return new ResponseObject(ResponseStatusCode.SUCCESS, response);
		} catch (DatabaseAccessException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_DB_ERROR, e);
		} catch (USMSQLException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_SQL_ERROR, e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_JSON_ERROR, e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.INVALIDATE_OBJECTS_INTERNAL_ERROR,
					e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

}
