/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.datatypes.folder;

import java.io.Serializable;

import org.json.JSONException;
import org.json.JSONObject;

public class FolderPermission implements Serializable {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	// For non-mail folders, a number as described:
	/*
	Bits 	 Value
	0-6 	Folder permissions:
	    0 	No permissions.
	    1 	See the folder.
	    2 	Create objects in the folder.
	    4 	Create subfolders.
	    64 	Admin.
	7-13 	Read permissions for objects in the folder:
	    0 	No permissions.
	    1 	Read only own objects.
	    2 	Read all objects.
	    64 	Admin.
	14-20 	Write permissions for objects in the folder:
	    0 	No permissions.
	    1 	Modify only own objects.
	    2 	Modify all objects.
	    64 	Admin.
	21-27 	Delete permissions for objects in the folder:
	    0 	No permissions.
	    1 	Delete only own objects.
	    2 	Delete all objects.
	    64 	Admin.
	28 	Admin flag:
	    0 	No permissions.
	    1 	Allowed to modify permission
	 * 
	 */
	private final int _bits;
	public static final String BITS = "bits";
	// For mail folders, the rights string as defined in RFC 2086. 
	private final String _rights;
	public static final String RIGHTS = "rights";
	// User ID of the user or group to which this permission applies. 
	private final int _entity;
	public static final String ENTITY = "entity";
	// true if entity refers to a group, false if it refers to a user. 
	private final boolean _group;
	public static final String GROUP = "group";

	/**
	 * Constructor
	 * 
	 * @param bits
	 * @param rights
	 * @param entity
	 * @param group
	 */
	public FolderPermission(int bits, String rights, int entity, boolean group) {
		_bits = bits;
		_rights = rights;
		_entity = entity;
		_group = group;
	}

	public FolderPermission() {
		_bits = 0;
		_rights = "";
		_entity = 0;
		_group = false;
	}

	public FolderPermission(JSONObject jSONObject) throws JSONException {
		_bits = jSONObject.has(BITS) ? jSONObject.getInt(BITS) : 0;
		_rights = jSONObject.has(RIGHTS) ? jSONObject.getString(RIGHTS) : "";
		_entity = jSONObject.has(ENTITY) ? jSONObject.getInt(ENTITY) : 0;
		_group = jSONObject.has(GROUP) ? jSONObject.getBoolean(GROUP) : true;
	}

	/**
	 * 
	 * @return
	 */
	public int getBits() {
		return _bits;
	}

	/**
	 * 
	 * @return
	 */
	public String getRights() {
		return _rights;
	}

	public int getEntity() {
		return _entity;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isGroup() {
		return _group;
	}

	@Override
	public int hashCode() {
		final int prime = 37;
		int result = 1;
		result = prime * result + (_group ? 1 : 0);
		result = prime * result + _entity;
		result = prime * result + _rights.hashCode();
		result = prime * result + _bits;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		FolderPermission other = (FolderPermission) obj;
		if (_group != other._group)
			return false;
		if (!_rights.equals(other._rights))
			return false;
		if (_entity != other._entity)
			return false;
		if (_bits != other._bits)
			return false;
		return true;
	}

	@Override
	public String toString() {
		return '{' + GROUP + ':' + _group + ';' + RIGHTS + ':' + _rights + ';' + ENTITY + ':' + _entity + ';' + BITS
				+ ':' + _bits + '}';
	}

	public JSONObject toJSONObject() throws JSONException {
		JSONObject js = new JSONObject();
		js.put(BITS, _bits); // XXX Not always present (only non-mail folders), do not throw Exception if not present
		js.put(RIGHTS, _rights); // XXX Not always present (only mail folders), do not throw Exception if not present
		js.put(ENTITY, _entity);
		js.put(GROUP, _group);
		return js;
	}
}
