/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 * 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * 
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com 
 * 
 * @author Viktor Pracht <viktor.pracht@open-xchange.com>
 * 
 */

/**
 * Creates the prototype object of a subclass.
 * @param {Function} parent The constructor function of the superclass.
 * @param {Object} prototype The prototype object of the subclass, containing
 * only new and overridden members.
 * @type Object
 * @return The prototype object with all inherited members added to it.
 */
function extend(parent, prototype) {
    for (var i in parent.prototype)
        if (!(i in prototype)) prototype[i] = parent.prototype[i];
    return prototype;
}

/**
 * Empty function. Does nothing.
 */
function emptyFunction() {}

/**
 * Identity function. Returns its first parameter.
 */
function identity(x) { return x; }

/**
 * Constant function.
 * Returns a new function which returns the first parameter of this function.
 * @param x
 * @type Function
 * @return A function which returns x.
 */
function constant(x) { return function() { return x; }; }

/**
 * Returns whether an object is empty.
 * @param {Object} x The tested object
 * @type Boolean
 * @return True if the object does not have enumerable properties,
 * false otherwise.
 */
function isEmpty(x) {
    for (var i in x) return false;
    return true;
}

/** @ignore */
function assert(value) { if (!value) alert("Assertion failed"); }

/**
 * Internet Explorer version number, or undefined if using a real browser.
 * Since comparisons with undefined always return false, checks involving this
 * variable should use the "<" or "<=" operators, e. g. "if (IE < 8)".
 */
var IE = (navigator.appName != "Microsoft Internet Explorer") ? undefined
    : Number(navigator.appVersion.match(/MSIE (\d+\.\d+)/)[1]);

/**
 * Boolean indicating the MacOS platform. If true, selections use the Meta key
 * instead of the Ctrl key.
 */
var Mac = navigator.platform.substring(0, 3) == "Mac";

/**
 * Creates a function which can add or remove a CSS class to a DOM node.
 * When adding, a class name is not duplicated if already present.
 * When removing, all instances of a class name are removed if present multiple
 * times.
 * @param {String} name The class name to add or remove. It must not contain any
 * characters which have special meaning in regular expressions.
 * @type Function
 * @return A function which accepts two parameters:<ul> 
 * <li>{DOM node} node The DOM node to which the class name is applied.</li>
 * <li>{Boolean} set Whether the class name should be added (true) or
 * removed (false).</li>
 * @ignore
 */
function classNameSetter(name) {
    var regex = new RegExp("(\\s)\\s*" + name + "\\s+|\\s*" + name + "\\s*",
                           "g");
    return function (node, set) {
        var c = String(node.className);
        regex.lastIndex = 0;
        if (set) {
            if (!regex.test(c)) c = c ? c + " " + name : name;
        } else {
            c = c.replace(regex, "$1");
        }
        node.className = c;
        return c;
    };
}

/**
 * Escapes a string to be included in an HTML file.
 * The characters '<', '>', '&', '"' and "'" are replaced by their HTML entity
 * references.
 * @param {String} s The string to escape.
 * @type String
 * @return The escaped string.
 */
function escapeHTML(s) {
    return s.replace(/[<>&"']/g, function(c) { return htmlEscapes[c]; });
}
var htmlEscapes = { "<": "&lt;", ">": "&gt;", "&": "&amp;", "\"": "&quot;",
                    "'": "&apos;" };

/**
 * Converts HTML entities to Unicode characters.
 * @param {String} html The HTML to unescape.
 * @type String
 * @return The unescaped string with resolved entities.
 */
function unescapeHTML(html) {
    return html.replace(/&(?:(\w+)|#x([0-9A-Fa-f]+)|#(\d+));/g,
        function(original, entity, hex, dec) {
            return entity ? unescapeHTML.entities[entity] || original :
                   hex    ? String.fromCharCode(parseInt(hex, 16)) :
                            String.fromCharCode(parseInt(dec, 10));
        });
}

unescapeHTML.entities = (function(es) {
    for (var i in es) es[i] = String.fromCharCode(es[i]);
    return es;
}({
    nbsp: 160, iexcl: 161, cent: 162, pound: 163, curren: 164, yen: 165,
    brvbar: 166, sect: 167, uml: 168, copy: 169, ordf: 170, laquo: 171,
    not: 172, shy: 173, reg: 174, macr: 175, deg: 176, plusmn: 177, sup2: 178,
    sup3: 179, acute: 180, micro: 181, para: 182, middot: 183, cedil: 184,
    sup1: 185, ordm: 186, raquo: 187, frac14: 188, frac12: 189, frac34: 190,
    iquest: 191, Agrave: 192, Aacute: 193, Acirc: 194, Atilde: 195, Auml: 196,
    Aring: 197, AElig: 198, Ccedil: 199, Egrave: 200, Eacute: 201, Ecirc: 202,
    Euml: 203, Igrave: 204, Iacute: 205, Icirc: 206, Iuml: 207, ETH: 208,
    Ntilde: 209, Ograve: 210, Oacute: 211, Ocirc: 212, Otilde: 213, Ouml: 214,
    times: 215, Oslash: 216, Ugrave: 217, Uacute: 218, Ucirc: 219, Uuml: 220,
    Yacute: 221, THORN: 222, szlig: 223, agrave: 224, aacute: 225, acirc: 226,
    atilde: 227, auml: 228, aring: 229, aelig: 230, ccedil: 231, egrave: 232,
    eacute: 233, ecirc: 234, euml: 235, igrave: 236, iacute: 237, icirc: 238,
    iuml: 239, eth: 240, ntilde: 241, ograve: 242, oacute: 243, ocirc: 244,
    otilde: 245, ouml: 246, divide: 247, oslash: 248, ugrave: 249, uacute: 250,
    ucirc: 251, uuml: 252, yacute: 253, thorn: 254, yuml: 255, fnof: 402,
    Alpha: 913, Beta: 914, Gamma: 915, Delta: 916, Epsilon: 917, Zeta: 918,
    Eta: 919, Theta: 920, Iota: 921, Kappa: 922, Lambda: 923, Mu: 924, Nu: 925,
    Xi: 926, Omicron: 927, Pi: 928, Rho: 929, Sigma: 931, Tau: 932,
    Upsilon: 933, Phi: 934, Chi: 935, Psi: 936, Omega: 937, alpha: 945,
    beta: 946, gamma: 947, delta: 948, epsilon: 949, zeta: 950, eta: 951,
    theta: 952, iota: 953, kappa: 954, lambda: 955, mu: 956, nu: 957, xi: 958,
    omicron: 959, pi: 960, rho: 961, sigmaf: 962, sigma: 963, tau: 964,
    upsilon: 965, phi: 966, chi: 967, psi: 968, omega: 969, thetasym: 977,
    upsih: 978, piv: 982, bull: 8226, hellip: 8230, prime: 8242, Prime: 8243,
    oline: 8254, frasl: 8260, weierp: 8472, image: 8465, real: 8476,
    trade: 8482, alefsym: 8501, larr: 8592, uarr: 8593, rarr: 8594, darr: 8595,
    harr: 8596, crarr: 8629, lArr: 8656, uArr: 8657, rArr: 8658, dArr: 8659,
    hArr: 8660, forall: 8704, part: 8706, exist: 8707, empty: 8709, nabla: 8711,
    isin: 8712, notin: 8713, ni: 8715, prod: 8719, sum: 8721, minus: 8722,
    lowast: 8727, radic: 8730, prop: 8733, infin: 8734, ang: 8736, and: 8743,
    or: 8744, cap: 8745, cup: 8746, "int": 8747, there4: 8756, sim: 8764,
    cong: 8773, asymp: 8776, ne: 8800, equiv: 8801, le: 8804, ge: 8805,
    sub: 8834, sup: 8835, nsub: 8836, sube: 8838, supe: 8839, oplus: 8853,
    otimes: 8855, perp: 8869, sdot: 8901, lceil: 8968, rceil: 8969,
    lfloor: 8970, rfloor: 8971, lang: 9001, rang: 9002, loz: 9674, spades: 9824,
    clubs: 9827, hearts: 9829, diams: 9830, quot: 34, amp: 38, lt: 60, gt: 62,
    OElig: 338, oelig: 339, Scaron: 352, scaron: 353, Yuml: 376, circ: 710,
    tilde: 732, ensp: 8194, emsp: 8195, thinsp: 8201, zwnj: 8204, zwj: 8205,
    lrm: 8206, rlm: 8207, ndash: 8211, mdash: 8212, lsquo: 8216, rsquo: 8217,
    sbquo: 8218, ldquo: 8220, rdquo: 8221, bdquo: 8222, dagger: 8224,
    Dagger: 8225, permil: 8240, lsaquo: 8249, rsaquo: 8250, euro: 8364
}));

/**
 * Replaces tabs, spaces and newlines with non-breaking spaces and <br>s to
 * preserve plain text formatting.
 * @param {Boolean} raw Specifies the kind of newlines:
 * "<br>" if false, "\n" if true.
 * @type String
 * @return The converted text.
 */
function replaceSpaces(text, raw) {
    var lines = text.split(raw ? /\r\n?|\n/g : /<br\s*\/?>/g);
    for (var i = 0; i < lines.length; i++) {
        var offset = 0;
        lines[i] = lines[i]
            // replace tabs with spaces
            .replace(/\t/g, function(tab, index) {
                // number of inserted spaces
                var count = 8 - (index + offset) % 8;
                offset += count - 1;
                return "        ".slice(-count);
            // Replace every other space with &nbsp;
            // A space at the start of the line is always replaced to
            // keep indentation.
            }).replace(/^ +| {2,}/g, function(space) {
                return new Array((space.length >> 1) + 1).join("\xa0 ")
                    + (space.length & 1 ? "\xa0" : "");
            });
    }
    return lines.join('<br>');
}

/**
 * Escapes a string to be included in a regular expression.
 * @param {String} s The string to escape.
 * @type String
 * @return The escaped string.
 */
function escapeRegExp(s) { return s.replace(/([|^$\\.*+?()[\]{}])/g, '\\$1'); }

/**
 * Quotes the personal name part of a mail address, if necessary.
 * @param {String} s The personal name to quote.
 * @type String
 * @return Either the quoted or the original string.
 */
function quotePersonalAddr(s) {
    return /[.,:;<>"]/.test(s) ? '"' + s.replace(/(["\\])/g, "\\$1") + '"' : s;
}

/**
 * Converts an Object into Array
 */
function ObjectToArray(object) {
    var tmp = [];
    for (var i in object) {
        tmp.push(object[i]);
    }
    return tmp;
}

/**
 * check if the given participant object contains
 * a group.
 */
function isGroupAppointment(participants) {
    if (participants) {
        if (participants.length > 1) {
            return true;
        } else {
            // looking for a group then
            for (var i=0; i < participants.length; i++) {
                if (participants[i].type == 2) {
                    // found a group
                    return true;
                }
            }
        }
    }
    return false;
}