/*
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 * 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * 
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 */

// helper for password strength
(function ($) {
    
    ox.UI.Custom = function (text) {
        this.text = text;
        ox.UI.Widget.apply(this);
        this.div = $("<div/>");
    };

    ox.UI.Custom.prototype = extend(ox.UI.Widget, {
        addContent: function (node) {
            this.node = this.parent.addCells(this.text || "", this.div.get(0));
            ox.UI.Widget.prototype.addContent.apply(this, arguments);
        }
    });
    
    ox.UI.LimitedPassword = function (text, maxLength) {
        ox.UI.Password.call(this, text);
        this.maxLength = maxLength;
    };

    ox.UI.LimitedPassword.prototype = extend(ox.UI.Password, {
        addContent: function () {
            ox.UI.Password.prototype.addContent.apply(this, arguments);
            // apply limit?
            if (this.maxLength) {
                jQuery(this.formnode).attr("maxlength", this.maxLength);
            }
        }
    });

}(jQuery));

register("Ready", function () {
    
    new ox.Configuration.InnerNode("configuration/user", _("User"));
      
    var node = new ox.Configuration.LeafNode("configuration/user/password", _("Password"));
    var page = new ox.Configuration.Page(node, _("Change Password"));
     
    var intro = new ox.UI.Text(_("Choose a new password for your account."));
    page.addWidget(intro);
    
    // old password
    var pass_old = new ox.UI.Password(_("Your old password"));
    page.addWidget(pass_old, "old_password");
    
    /**
     * Password strength
     */
    
    var enabled = ox.api.config.get(["modules", NAME, "showStrength"], false),
        minLength = ox.api.config.get(["modules", NAME, "minLength"], 4),
        maxLength = ox.api.config.get(["modules", NAME, "maxLength"], 0),
        regexp = ox.api.config.get(["modules", NAME, "regexp"], "[^a-z0-9]"),
        special = ox.api.config.get(["modules", NAME, "special"], "$, _, %");

    var pwTooShort = false;
    
    // new password
    var pass_new = new ox.UI.LimitedPassword(_("The new password"), maxLength);
    page.addWidget(pass_new, "new_password");
     
    var pass_new2 = new ox.UI.LimitedPassword(_("Confirm new password"), maxLength);
    page.addWidget(pass_new2, "new_password2");

    if (enabled) {
        
        var strength = new ox.UI.Custom(_("Password strength"));
        page.addWidget(strength);
        
        page.addWidget(
            new ox.UI.Text(
                maxLength === 0 ?
                    format(_("Minimum password length is %1$s."), minLength) :
                    format(_("Password length must be between %1$s and %2$s characters."), minLength, maxLength)
            )
        );
        
        page.addWidget(
            new ox.UI.Text(
                format(_('Your password is more secure if it also contains capital letters, numbers, and special characters like %1$s.'), special)
            )
        );
        
        var node = strength.div;
        node.css({
            fontFamily: "Arial, Helvetica, sans-serif",
            fontSize: "12px",
            lineHeight: "30px",
            backgroundColor: "#aaa",
            color: "white",
            fontWeight: "bold",
            width: "150px",
            margin: "1em 0 1em 0",
            textAlign: "center",
            MozBorderRadius: "15px",
            webkitBorderRadius: "15px",
            borderRadius: "15px 15px 15px 15px",
            textShadow: "0px 0px 3px black",
            MozBoxShadow: "0px 0px 10px 5px white",
            webkitBoxShadow: "0px 0px 10px 5px white",
            boxShadow: "0px 0px 10px 5px white",
            cursor: "default"
        });
        
        // password strength description
        var levels = {
	    deny: addTranslated(_("Too short")),
            veryweak: addTranslated(_("Very weak")),
            weak: addTranslated(_("Weak")),
            good: addTranslated(_("Good")),
            strong: addTranslated(_("Strong")),
            verystrong: addTranslated(_("Very strong"))
        };
        
        // analyze strength
        var fnChange = function () {
            var val = jQuery.trim(jQuery(this).val()),
                points = 0,
                len = val.length;
            // true becomes 1
            // lower case; +1 upper case +1; numbers +1; special chars +1
            points = 0 + /[a-z]/.test(val) + /[A-Z]/.test(val) + /[0-9]/.test(val) + (new RegExp(regexp, "i")).test(val);
            // show strength?
            node.empty();
            pwTooShort = false;
            if (len < minLength) {
                node.append(levels.deny).css({ backgroundColor: "#800" });
                pwTooShort = true;
            } else if (points >= 4 && len >= 10) {
                node.append(levels.verystrong).css({ backgroundColor: "#808" });
            } else if (points >= 3 && len >= 8) {
                node.append(levels.strong).css({ backgroundColor: "#08c" });
            } else if (points >= 2 && len >= 6) {
                node.append(levels.good).css({ backgroundColor: "#0a0" });
            } else if (points >= 1 && len >= 6) {
                node.append(levels.weak).css({ backgroundColor: "#c00" });
            } else {
                node.append(levels.veryweak).css({ backgroundColor: "#800" });
            }
            node.show();
        };
        // bind
        pass_new.onChange = fnChange;
        // initialize
        page.load = function (cont) {
            cont(this.default_value);
        };
        // clean-up
        page.leave = function() {
    		if (node) node.empty().hide();
        };
    }
    
    // ------------------
    
    page.save = function (data, cont) {
    	function emptyPass() {
    		ox.Configuration.error(_("Please enter your old password and twice the new password."));		
    	}
    	if (pass_old.get() == "") {
    		pass_old.formnode.focus();
    		return emptyPass();
    	} else if (pass_new.get() == "") {
    		pass_new.formnode.focus();
    		return emptyPass();
    	} else if (pass_new2.get() == "") {
    		pass_new2.formnode.focus();
    		return emptyPass();
    	} else if (pass_new.get() != pass_new2.get()) {
    		pass_new.formnode.focus();
    		ox.Configuration.error(_("The two newly entered passwords do not match."));
    		return;
    	} else if (pwTooShort) {
       		pass_new.formnode.focus();
       		ox.Configuration.error(format(_("Minimum password length is %1$s."), minLength));
       		return;
        }

    	ox.JSON.put(AjaxRoot+"/passwordchange?action=update&session="+session,data,
            function() {
                ox.Configuration.info(
                    _("Your new password has been saved."));
                cont();
            });
    };
});
