/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.data;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.apache.tools.ant.BuildException;

/**
 * Abstract super class for all supported types of presentation of a bundle. Contains the general methods for handling imports and exports
 * and for generating the dependencies and the required class path.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public abstract class AbstractModule {

    protected File file;
    protected String name;
    protected OSGIManifest osgiManifest;
    private String fragmentHostName;
    protected AbstractModule fragmentHost;
    protected final Set<AbstractModule> dependencies = new HashSet<AbstractModule>();
    private Set<String> deepClasspath;

    protected AbstractModule(File file) {
        super();
        this.file = file;
        name = file.getName();
    }

    public Set<String> getExportedPackages() {
        final Set<String> retval;
        if (null == osgiManifest) {
            retval = Collections.emptySet();
        } else {
            retval = osgiManifest.getExports();
        }
        return retval;
    }

    public void computeDependencies(Map<String, AbstractModule> modulesByName, Map<String, Set<AbstractModule>> modulesByPackage, boolean strict) {
        if (osgiManifest != null) {
            for (final BundleImport imported : osgiManifest.getImports()) {
                final Set<AbstractModule> exportingModules = modulesByPackage.get(imported.getPackageName());
                if (exportingModules != null) {
                    for (final AbstractModule module : exportingModules) {
                        if (module != this) {
                            dependencies.add(module);
                        }
                    }
                } else if (!JDK.exports(imported.getPackageName()) && !imported.isOptional() && strict) {
                    throw new BuildException("Can not find bundle that exports \"" + imported.getPackageName() + "\" to resolve import of bundle \"" + name + "\".");
                }
            }
            for (final RequiredBundle requiredBundle : osgiManifest.getRequiredBundles()) {
                final AbstractModule requiredModule = modulesByName.get(requiredBundle.getPackageName());
                if (null == requiredModule) {
                    if (strict) {
                        throw new BuildException("Can not find bundle to resolve require bundle \"" + requiredBundle.getPackageName() + "\".");
                    }
                } else {
                    if (!this.equals(requiredModule)) {
                        dependencies.add(requiredModule);
                    }
                }
            }
            fragmentHostName = osgiManifest.getEntry(OSGIManifest.FRAGMENT_HOST);
            if (fragmentHostName != null && !"system.bundle".equals(fragmentHostName)) {
                fragmentHost = modulesByName.get(fragmentHostName);
                if (null == fragmentHost) {
                    if (strict) {
                        throw new BuildException("Can not find bundle for fragment host \"" + fragmentHostName + "\".");
                    }
                } else {
                    dependencies.add(fragmentHost);
                }
            }
        }
    }

    public void computeDependenciesForFragments() {
        if (fragmentHost != null) {
            dependencies.addAll(fragmentHost.getDependencies());
            dependencies.remove(this); // just in case the fragment host "requires" the fragment
        }
    }

    public Set<AbstractModule> getDependencies() {
        return dependencies;
    }

    public Set<String> getRequiredClasspath() {
        final Set<String> retval = new HashSet<String>();
        for (final AbstractModule dependency : dependencies) {
            retval.addAll(dependency.getExportedClasspath());
        }
        return Collections.unmodifiableSet(retval);
    }

    public Set<String> getDeepRequiredClasspath() {
        if (null == deepClasspath) {
            final Stack<AbstractModule> seenModules = new Stack<AbstractModule>();
            deepClasspath = getDeepRequiredClasspath(seenModules, this);
        }
        return deepClasspath;
    }

    public boolean isFragment() {
        if (osgiManifest != null) {
            fragmentHostName = osgiManifest.getEntry(OSGIManifest.FRAGMENT_HOST);
        }
        return null != fragmentHostName;
    }

    private Set<String> getDeepRequiredClasspathUnseen(Stack<AbstractModule> seenModules) {
        if (null == deepClasspath) {
            deepClasspath = getDeepRequiredClasspath(seenModules, this);
        }
        return deepClasspath;
    }

    private static Set<String> getDeepRequiredClasspath(final Stack<AbstractModule> seenModules, final AbstractModule module) {
        seenModules.push(module);
        Set<String> retval = new HashSet<String>();
        for (final AbstractModule dependency : module.getDependencies()) {
            if (!seenModules.contains(dependency)) {
                retval.addAll(dependency.getDeepRequiredClasspathUnseen(seenModules));
            }
            retval.addAll(dependency.getExportedClasspath());
        }
        seenModules.pop();
        return retval;
    }

    protected final boolean isExported(final File classpathEntry) {
        if (!classpathEntry.exists()) {
            return false;
        }
        try {
            final JarFile jarFile = new JarFile(classpathEntry);
            for (final String exportedPackage : getExportedPackages()) {
                String expected = exportedPackage.replace('.', '/') + '/';
                Enumeration<JarEntry> entries = jarFile.entries();
                while (entries.hasMoreElements()) {
                    JarEntry entry = entries.nextElement();
                    if (entry.getName().startsWith(expected)) {
                        return true;
                    }
                }
            }
        } catch (IOException e) {
            throw new BuildException(e);
        }
        return false;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == null) {
            return false;
        }
        if (o == this) {
            return true;
        }
        if (o.getClass() != this.getClass()) {
            return false;
        }
        final AbstractModule other = (AbstractModule) o;
        return this.name.equals(other.name);
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public String toString() {
        return name;
    }

    public String getName() {
        return name;
    }

    protected abstract Set<String> getExportedClasspath();

    public Set<String> getSourceDirs() {
        return Collections.emptySet();
    }

    public Set<String> getTestSourceDirs() {
        return Collections.emptySet();
    }

    public String getFileOrDirName() {
        return file.getName();
    }
}
