/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.tasks;

import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;

/**
 * {@link USMSessionCleanupTask}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class USMSessionCleanupTask implements Runnable {

    private static final Logger log = LoggerFactory.getLogger(USMSessionCleanupTask.class);

    /**
     * Initial delay after startup of SessionManager/TimerService after which the cleanup task is first performed
     */
    public static final long INITIAL_DELAY = 1000L * 10L * 1L; // 3 minutes

    /**
     * Interval between executions of the cleanup task
     */
    public static final long EXECUTION_INTERVAL = 1000L * 60L * 1L; // 1 day

    private final SessionManagerImpl _sessionManager;

    /**
     * Initializes a new {@link USMSessionCleanupTask}.
     * 
     * @param sessionManager
     */
    public USMSessionCleanupTask(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        log.debug("Checking for old USM sessions");
        try {
            // TODO: Implement something more efficient (e.g. LRU mechanism)
            int count = 0;
            Set<Integer> remainingContextIDs = new HashSet<Integer>(_sessionManager.getSessionDataStorage().getAllContextIDs());
            while (!remainingContextIDs.isEmpty()) {
                int context = remainingContextIDs.iterator().next();
                count += checkForOldSessions(context, remainingContextIDs);
                for (int contextId : _sessionManager.getSessionDataStorage().getContextsInSameSchema(context)) {
                    remainingContextIDs.remove(contextId);
                }
            }
            log.info("Checked for old USM sessions, " + count + " were removed");
        } catch (StorageAccessException e) {
            log.error("Couldn't access DB to check for old USM sessions", e);
        } catch (Exception e) {
            log.error("Uncaught error occurred while checking for old USM sessions", e);
        }
    }

    /**
     * Check db for old USM sessions for the specified context ID
     * 
     * @param cid
     * @return
     * @throws StorageAccessException
     */
    private int checkForOldSessions(int cid, Set<Integer> validContextIDs) throws StorageAccessException {
        log.debug("Checking for old USM sessions in same schema as context " + cid);
        Set<ContextSessionID> sessionIDsToRemove = _sessionManager.getSessionDataStorage().getSessionsToRemove(cid, _sessionManager.getSessionStorageTimeLimit(), validContextIDs);

        // Check if any of the sessions to remove is currently active in the SessionManager, do not remove those sessions
        if (!sessionIDsToRemove.isEmpty()) {
            for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
                sessionIDsToRemove.remove(new ContextSessionID(session.getContextId(), session.getSessionId()));
            }
        }

        int count = _sessionManager.getSessionDataStorage().purgeSessions(cid, sessionIDsToRemove);
        log.debug(count + " USM sessions removed in same schema as context " + cid);
        return count;
    }
}
