package com.openexchange.usm.mapping.storage.sql.impl;

import gnu.trove.ConcurrentTIntObjectHashMap;
import gnu.trove.map.TIntObjectMap;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.mapping.storage.ContextUUIDMappingStorage;
import com.openexchange.usm.util.Toolkit;

/**
 * This class saves a uuid for the contexts in a table UUIDHeader. This is needed for USM/JSON clients. 
 * @author ldo
 *
 */ 
public class ContextUUIDMappingSQL implements ContextUUIDMappingStorage {
    
    public static final int UUID_HEADER_SQL_ERROR = 0x330004;
    
    private final TIntObjectMap<UUID> _contextUUIDs = new ConcurrentTIntObjectHashMap<UUID>();
    
    private static final  Log _journal = LogFactory.getLog(ContextUUIDMappingSQL.class);
    private DatabaseAccess _dbAccess;
    
    public ContextUUIDMappingSQL(DatabaseAccess _dbAccess) {
        super();
        this._dbAccess = _dbAccess;
    }
    
    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.mapping.storage.ContextUUIDMappingStorage#getContextUUID(int)
     */
    @Override
    public UUID getContextUUID(int cid) throws StorageAccessException, USMStorageException {
        UUID uuid = _contextUUIDs.get(cid);
        if (uuid != null)
            return uuid;

        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        ResultSet result = null;

        try {
            con = _dbAccess.getReadOnly(cid);
            statement = con.prepareStatement("SELECT header FROM UUIDHeader where cid = ?");
            statement.setInt(1, cid);

            result = statement.executeQuery();
            if (result.next()) {
                uuid = UUID.fromString(result.getString(1));
            } else {
                Toolkit.close(result);
                Toolkit.close(statement);
                Toolkit.close(con);
                //generate the header
                uuid = UUID.randomUUID();
                con = _dbAccess.getWritable(cid);
                statement = con.prepareStatement("INSERT INTO UUIDHeader (cid, header) VALUES(?, ?)");
                statement.setInt(1, cid);
                statement.setString(2, uuid.toString());
                if (statement.executeUpdate() != 1)
                    _journal.error("Insert of UUID header was not performed for cid " + cid);
            }
        } catch (SQLException e) {
            String errorMessage = " SQL error getting UUID header ";
            _journal.error(errorMessage, e);
            throw new USMStorageException(UUID_HEADER_SQL_ERROR, errorMessage, e);
        } finally {
            Toolkit.close(result);
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        _contextUUIDs.put(cid, uuid);
        return uuid;
    }
    
    

}
