/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.CLIENT_FIELD_PREFIX;
import static com.openexchange.usm.connector.commands.CommandConstants.CONFLICT_RESOLUTION;
import static com.openexchange.usm.connector.commands.CommandConstants.CUSTOM_PROPERTIES;
import static com.openexchange.usm.connector.commands.CommandConstants.END_DATE;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDER_TREE;
import static com.openexchange.usm.connector.commands.CommandConstants.KEY_SIZE;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.START_DATE;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNC_CONTENT_TYPES;
import java.util.BitSet;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.JSONSessionInitializer;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.util.JSONToolkit;

/**
 * Handler for the SetConfiguration USM-JSON-Command.
 * @author ldo
 *
 */
public class SetConfigurationHandler extends NormalCommandHandler {
	private static final String ALL = "all";
	private static final String ALL_EXCEPT = "all except";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID };
	private static final String[] OPTIONAL_PARAMETERS = {
			SYNC_CONTENT_TYPES,
			START_DATE,
			END_DATE,
			FOLDER_TREE,
			CONFLICT_RESOLUTION,
			CUSTOM_PROPERTIES };

    public SetConfigurationHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		try {
			if (_parameters.has(START_DATE))
				_session.setStartDate(getLong(_parameters, START_DATE));
			if (_parameters.has(END_DATE))
				_session.setEndDate(getLong(_parameters, END_DATE));
			if (_parameters.has(FOLDER_TREE))
				_session.setPersistentField(FolderContentType.FOLDER_TREE, String.valueOf(getInt(_parameters,
						FOLDER_TREE)));
			if (_parameters.has(CONFLICT_RESOLUTION))
				_session.setConflictResolution(getOptionalConflictResolution());
			if (_parameters.has(CUSTOM_PROPERTIES))
				setCustomProperties();
			if (_parameters.has(SYNC_CONTENT_TYPES))
				setSyncContentTypes();
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIG_INTERNAL_ERROR, e);
		}
		return new ResponseObject(ResponseStatusCode.SUCCESS);
	}

	private void setCustomProperties() throws USMJSONAPIException, StorageAccessException, USMStorageException {
		JSONObject customProperties = getJSONObject(_parameters, CUSTOM_PROPERTIES);
		for (String propertyName : JSONToolkit.keys(customProperties)) {
			if (propertyName.length() > KEY_SIZE)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_KEY_TOO_LONG,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "key too long: " + propertyName);
			_session.setPersistentField(CLIENT_FIELD_PREFIX + propertyName, getString(customProperties, propertyName));
		}
	}

	private void setSyncContentTypes() throws USMException {
		JSONObject contentTypes = getContentTypeMap();
		ContentTypeManager contentTypeManager = _servlet.getContentTypeManager();
		for (String name : JSONToolkit.keys(contentTypes)) {
			ContentType type = contentTypeManager.getContentType(name);
			if (type == null)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_UNKNOWN_CONTENTTYPE,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown content type " + name);
			JSONArray fieldNames = getContentTypeFieldNames(contentTypes, name);
			BitSet syncFields = new BitSet();
			BitSet extraFields = new BitSet();
			ContentTypeField[] fields = type.getFields();
			int length = fieldNames.length();
			for (int i = 0; i < length; i++) {
				int index = getContentTypeFieldIndex(fields, getString(fieldNames, i));
				if (index >= 0) {
					if (fields[index].getFieldID() >= 0)
						syncFields.set(index);
					else
						extraFields.set(index);
				} else
					extraFields.set(-index - 1); // special case for fields that are not part of ContentType, e.g. mail
			}
			if (!extraFields.isEmpty())
				syncFields.set(0, 2);
			BitSet originalSyncFields = new BitSet();
			originalSyncFields.or(syncFields);
			if (JSONSessionInitializer.excludeInvalidFieldsFromFilter(type, syncFields)) {
				originalSyncFields.andNot(syncFields);
				StringBuilder sb = new StringBuilder("Fields not supported for sync in ContentType ");
				sb.append(type.getID()).append(':');
				char delimiter = ' ';
				for (int i = originalSyncFields.nextSetBit(0); i >= 0; i = originalSyncFields.nextSetBit(i + 1)) {
					sb.append(delimiter).append(fields[i].getFieldName());
					delimiter = ',';
				}
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_UNSUPPORTED_FIELDS_FOR_CONTENT_TYPE,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, sb.toString());
			}
			_session.setFieldFilter(type, syncFields);
			storeExtraFields(type, extraFields);
		}
	}

	private int getContentTypeFieldIndex(ContentTypeField[] fields, String field) throws USMJSONAPIException {
		int externalMailFieldIndex = ExternalMailContentTypeFields.getFieldIndex(field);
		if (externalMailFieldIndex >= 0)
			return -1 - externalMailFieldIndex;
		if(field.equals(CommandConstants.ATTACHMENTS)) {
			field = CommandConstants.ATTACHMENTS_LAST_MODIFIED;
		}
		for (int i = 0; i < fields.length; i++) {
			if (fields[i].getFieldName().equals(field))
				return i;
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_ILLEGAL_CONTENT_TYPE_FIELD,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Illegal field name " + field);
	}

	private JSONArray getContentTypeFieldNames(JSONObject contentTypes, String name) throws USMJSONAPIException {
		try {
			Object fields = contentTypes.get(name);
			if (ALL.equals(fields)) {
				return getAvailableContentTypes(_servlet.getContentTypeManager()).getJSONArray(name);
			} else {
				JSONArray fieldsArray = contentTypes.getJSONArray(name);
				if (fieldsArray.length() > 0 && ALL_EXCEPT.equals(fieldsArray.get(0))) {
					JSONArray allFieldsArray = getAvailableContentTypes(_servlet.getContentTypeManager()).getJSONArray(
							name);
					JSONArray resultFieldsArray = new JSONArray();
					for (int i = 1; i < allFieldsArray.length(); i++) {
						if (!JSONToolkit.contains(fieldsArray, allFieldsArray.get(i)))
							resultFieldsArray.put(allFieldsArray.get(i));
					}
					return resultFieldsArray;
				} else {
					return fieldsArray;
				}
			}
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_ILLEGAL_CONTENT_TYPE_FIELD_LIST,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "ContentType field list not a valid JSONArray");
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.SET_CONFIG_CAN_NOT_GET_AVALIABLE_TYPES_2, e);
		}
	}

	private JSONObject getContentTypeMap() throws USMJSONAPIException {
		try {
			Object contentTypes = _parameters.get(SYNC_CONTENT_TYPES);
			if (ALL.equals(contentTypes)) {
				return getAvailableContentTypes(_servlet.getContentTypeManager());
			} else
				return _parameters.getJSONObject(SYNC_CONTENT_TYPES);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_CONFIG_ILLEGAL_CONTENT_TYPE_LIST,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "ContentType list not a valid JSONObject");
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.SET_CONFIG_CAN_NOT_GET_AVALIABLE_TYPES, e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}
}
