
package org.scribe.builder.api;

import org.scribe.exceptions.OAuthException;
import org.scribe.extractors.AccessTokenExtractor;
import org.scribe.model.*;
import org.scribe.oauth.OAuth20ServiceImpl;
import org.scribe.oauth.OAuthService;
import org.scribe.utils.OAuthEncoder;
import org.scribe.utils.Preconditions;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class LinkedInApi20 extends DefaultApi20 {

    private static final String AUTHORIZE_URL = "https://www.linkedin.com/oauth/v2/authorization?response_type=code&client_id=%s&redirect_uri=%s";
    private static final String SCOPED_AUTHORIZE_URL = AUTHORIZE_URL + "&scope=%s";
    private static final String SUFFIX_OFFLINE = "&" + OAuthConstants.ACCESS_TYPE + "=" + OAuthConstants.ACCESS_TYPE_OFFLINE
            + "&" + OAuthConstants.APPROVAL_PROMPT + "=" + OAuthConstants.APPROVAL_PROMPT_FORCE;

    /** The <code>"access_token": &lt;token&gt;</code> pattern */
    static final Pattern PATTERN_ACCESS_TOKEN = Pattern.compile("\"access_token\" *: *\"([^&\"]+)\"");

    /** The <code>"refresh_token": &lt;token&gt;</code> pattern */
    static final Pattern PATTERN_REFRESH_TOKEN = Pattern.compile("\"refresh_token\" *: *\"([^&\"]+)\"");

    /** The <code>"expires_in": &lt;number&gt;</code> pattern */
    static final Pattern PATTERN_EXPIRES = Pattern.compile("\"expires_in\" *: *([0-9]+)");

    public LinkedInApi20() {
        super();
    }

    @Override
    public String getAccessTokenEndpoint() {
        return "https://www.linkedin.com/oauth/v2/accessToken";
    }

    @Override
    public AccessTokenExtractor getAccessTokenExtractor() {
        return new AccessTokenExtractor() {

            @Override
            public Token extract(String response) {
                Preconditions.checkEmptyString(response, "Response body is incorrect. Can't extract a token from an empty string");

                Matcher matcher = PATTERN_ACCESS_TOKEN.matcher(response);
                if (!matcher.find()) {
                    throw new OAuthException("Response body is incorrect. Can't extract a token from this: '" + response + "'", null);
                }

                String token = OAuthEncoder.decode(matcher.group(1));
                String refreshToken = "";
                Matcher refreshMatcher = PATTERN_REFRESH_TOKEN.matcher(response);
                if (refreshMatcher.find()) {
                    refreshToken = OAuthEncoder.decode(refreshMatcher.group(1));
                }
                Date expiry = null;
                Matcher expiryMatcher = PATTERN_EXPIRES.matcher(response);
                if (expiryMatcher.find()) {
                    int lifeTime = Integer.parseInt(OAuthEncoder.decode(expiryMatcher.group(1)));
                    expiry = new Date(System.currentTimeMillis() + lifeTime * 1000);
                }
                Token result = new Token(token, refreshToken, expiry, response);
                return result;
            }
        };
    }

    @Override
    public String getAuthorizationUrl(OAuthConfig config) {
        // Append scope if present
        if (config.hasScope()) {
            String format = config.isOffline() ? SCOPED_AUTHORIZE_URL + SUFFIX_OFFLINE : SCOPED_AUTHORIZE_URL;
            return String.format(format, config.getApiKey(),
                    OAuthEncoder.encode(config.getCallback()),
                    OAuthEncoder.encode(config.getScope()));
        } else {
            String format = config.isOffline() ? AUTHORIZE_URL + SUFFIX_OFFLINE : AUTHORIZE_URL;
            return String.format(format, config.getApiKey(),
                    OAuthEncoder.encode(config.getCallback()));
        }
    }

    @Override
    public Verb getAccessTokenVerb() {
        return Verb.POST;
    }

    @Override
    public OAuthService createService(OAuthConfig config) {
        return new LinkedInOAuth2Service(this, config);
    }

    public static class LinkedInOAuth2Service extends OAuth20ServiceImpl {

        private final DefaultApi20 api;
        private final OAuthConfig config;

        LinkedInOAuth2Service(DefaultApi20 api, OAuthConfig config) {
            super(api, config);
            this.api = api;
            this.config = config;
        }

        /**
         * Requests the token info for specified access token and returns the lifetime of the token in seconds.
         *
         * @param accessToken The access token to validate
         * @param url The URL to invoke; e.g. "https:///www.linkedin.com/oauth/v2/tokeninfo"
         * @return The expiry in seconds or <code>-1</code> if access token is already expired
         */
        public int getExpiry(String accessToken, String url) {
            OAuthRequest request = new OAuthRequest(Verb.POST, url);
            request.addBodyParameter(OAuthConstants.ACCESS_TOKEN, accessToken);
            Response response = request.send();
            if (response.getCode() == 400) {
                // Expired
                return -1;
            }

            int expiry = -1;
            Matcher expiryMatcher = PATTERN_EXPIRES.matcher(response.getBody());
            if (expiryMatcher.find())
            {
                int lifeTime = Integer.parseInt(OAuthEncoder.decode(expiryMatcher.group(1)));
                expiry = lifeTime;
            }
            return expiry;
        }

        @Override
        public Token getAccessToken(Token requestToken, Verifier verifier) {
            OAuthRequest request = new OAuthRequest(api.getAccessTokenVerb(), api.getAccessTokenEndpoint());
            switch (api.getAccessTokenVerb()) {
                case POST:
                    request.addBodyParameter(OAuthConstants.CLIENT_ID, config.getApiKey());
                    // API Secret is optional
                    if (config.getApiSecret() != null && config.getApiSecret().length() > 0) {
                        request.addBodyParameter(OAuthConstants.CLIENT_SECRET, config.getApiSecret());
                    }
                    if (requestToken == null) {
                        request.addBodyParameter(OAuthConstants.CODE, verifier.getValue());
                        request.addBodyParameter(OAuthConstants.REDIRECT_URI, config.getCallback());
                        request.addBodyParameter(OAuthConstants.GRANT_TYPE, OAuthConstants.GRANT_TYPE_AUTHORIZATION_CODE);
                    } else {
                        request.addBodyParameter(OAuthConstants.REFRESH_TOKEN, requestToken.getSecret());
                        request.addBodyParameter(OAuthConstants.GRANT_TYPE, OAuthConstants.GRANT_TYPE_REFRESH_TOKEN);
                    }
                    break;
                case GET:
                default:
                    request.addQuerystringParameter(OAuthConstants.CLIENT_ID, config.getApiKey());
                    // API Secret is optional
                    if (config.getApiSecret() != null && config.getApiSecret().length() > 0) {
                        request.addQuerystringParameter(OAuthConstants.CLIENT_SECRET, config.getApiSecret());
                    }
                    request.addQuerystringParameter(OAuthConstants.CODE, verifier.getValue());
                    request.addQuerystringParameter(OAuthConstants.REDIRECT_URI, config.getCallback());
                    if(config.hasScope()) {
                        request.addQuerystringParameter(OAuthConstants.SCOPE, config.getScope());
                    }
            }
            Response response = request.send();
            return api.getAccessTokenExtractor().extract(response.getBody());
        }
    }

}
