/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.contenttypes.transferhandlers.ContentTypeTransferHandler;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TimestampMismatchException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMResourceAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.exceptions.UUIDNotFoundException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionChangeListener;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.api.session.assets.CompleteSessionID;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SessionID;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.assets.UserID;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.storage.SyncStateStorage;
import com.openexchange.usm.session.sync.ContentSyncerSupport;
import com.openexchange.usm.session.sync.FolderElementsStorage;
import com.openexchange.usm.session.sync.FolderHierarchyStorage;
import com.openexchange.usm.session.sync.OXDataCacheID;
import com.openexchange.usm.session.sync.SynchronizationLock;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;
import com.openexchange.usm.util.UUIDToolkit;

public class SessionImpl implements Session {

    private static final boolean MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION = false;

    private static final long FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION = -1L;

    private static final DataObject[] EMPTY_DATA_OBJECT_ARRAY = new DataObject[0];

    private final static String FOLDER_HIERARCHY_ID = "";

    private final static int FOLDER_ID_LENGTH_LIMIT = 39;

    private final static int DIRECT_PING_RESPONSE_LIMIT = 1;

    private static final long START_DATE_LIMIT = -100000000L * 86400L;

    private final PersistentSessionData _persistentData;

    private final SessionManagerImpl _sessionManager;

    private final List<SessionChangeListener> _changeListeners = new CopyOnWriteArrayList<SessionChangeListener>();

    private final Map<String, Long> _syncedFolderContent = new ConcurrentHashMap<String, Long>(16, 0.75f, 1);

    private final Map<String, Object> _customProperties = new ConcurrentHashMap<String, Object>(16, 0.75f, 1);

    private final Map<ContentType, Comparator<? super DataObject>> _contentTypeSorters = new ConcurrentHashMap<ContentType, Comparator<? super DataObject>>(
        16,
        0.75f,
        1);

    private final Queue<CustomPropertyLimit> _customPropertyLimits = new PriorityQueue<CustomPropertyLimit>();

    private final Object _waitObject = new Object(); // Used to wait on folder changes and to notify about folder changes

    private final SyncStateStorage _syncStateStorage;

    private final OXDataCacheID _folderHierarchyCacheID;

    private final CompleteSessionID _completeSessionID;

    private final SessionInitializer _sessionInitializer;

    private int _uniqueSessionID;

    private String _defaultInboxId;

    private String _defaultAddress;

    private String _password;

    private DataObjectFilter _syncServerObjectFilter;

    private TimeZone _userTimeZone;

    private String[] _rootFolders = new String[0];

    private long _startDate = 0L;

    private long _endDate = Long.MAX_VALUE;
    
    private long _startDateCalendar = 0L;

    // Similar to BaseXMLDelegate.getTimeFrame set on now + 3 years
    private long _endDateCalendar = System.currentTimeMillis() + 3L * 365L * 86400L * 1000L;

    private int _mailLimit = 0;

    private long _lastFolderSync = 0L;

    private long _lastFolderChange = FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION;

    private DataObjectFilter _lastEmailFilter = null;

    private long _lastAccessCheck = 0L;

    private int _folderIdLengthLimit;

    private String _clientIP;

    private int _directPingResponses = 0;

    private final ReentrantLock _oxLock = new ReentrantLock();

    private Map<String, String> _xHeaders = new HashMap<String, String>();

    private final Map<String, String> _mailFolderIdSeparators = new HashMap<String, String>();

    public SessionImpl(SessionManagerImpl sessionManager, String user, String password, String protocol, String device, SessionInitializer initializer) {
        _sessionManager = sessionManager;
        _password = password;
        _completeSessionID = new CompleteSessionID(new SessionID(user, protocol, device), new UserID(-1, -1), null);
        _persistentData = new PersistentSessionData(this);
        _syncStateStorage = new SyncStateStorage(this);
        _folderHierarchyCacheID = new OXDataCacheID(this);
        _sessionInitializer = initializer;
    }

    public void initialize(JSONObject configuration) throws USMException {
        int contextID;
        int userIdentifier;
        try {
            contextID = configuration.getInt("context_id");
        } catch (JSONException e) {
            throw generateInitializationException("context_id", e);
        }
        try {
            userIdentifier = configuration.getInt("identifier");
        } catch (JSONException e) {
            throw generateInitializationException("identifier", e);
        }
        try {
            String timeZone = configuration.getString("timezone");
            _userTimeZone = TimeZone.getTimeZone(timeZone);
            if (!timeZone.equals(_userTimeZone.getID()))
                _sessionManager.getJournal().error(this + " Can't resolve TimeZone " + timeZone + ", using " + _userTimeZone.getID());
        } catch (JSONException e) {
            _sessionManager.getJournal().error(this + " No timezone found in initialization", e);
        }
        _completeSessionID.setUserID(new UserID(contextID, userIdentifier));
        _defaultInboxId = JSONToolkit.getString(configuration, "modules", "mail", "defaultFolder", "inbox");
        _defaultAddress = JSONToolkit.getString(configuration, "modules", "mail", "defaultaddress");
        _persistentData.initialize(_sessionInitializer, configuration);
        JSONObject separators = JSONToolkit.getJSONObject(configuration, "modules", "mail", "separators");
        if (separators != null) {
            for (String key : JSONToolkit.keys(separators)) {
                String separator = separators.optString(key);
                if (separator != null)
                    _mailFolderIdSeparators.put(key, separator);
            }
        }
    }

    private USMException generateInitializationException(String fieldName, Throwable cause) {
        String message = this + " No " + fieldName + " found in OX user configuration";
        _sessionManager.getJournal().error(message, cause);
        return new USMException(USMSessionManagementErrorCodes.SESSION_INITIALIZATION_FAILED, message);
    }

    @Override
    public String getUser() {
        return _completeSessionID.getSessionID().getUser();
    }

    @Override
    public String getPassword() {
        return _password;
    }

    public void setPassword(String password) {
        _password = password;
    }

    @Override
    public String getDevice() {
        return _completeSessionID.getSessionID().getDevice();
    }

    @Override
    public String getProtocol() {
        return _completeSessionID.getSessionID().getProtocol();
    }

    @Override
    public int getContextId() {
        return _completeSessionID.getUserID().getContextId();
    }

    @Override
    public int getSessionId() {
        return _uniqueSessionID;
    }

    public void setUniqueSessionID(int uniqueSessionID) {
        _uniqueSessionID = uniqueSessionID;
    }

    @Override
    public int getUserIdentifier() {
        return _completeSessionID.getUserID().getUserId();
    }

    @Override
    public String getDefaultEmailAddress() {
        return _defaultAddress;
    }

    @Override
    public String getPersistentField(String name) {
        return _persistentData.getField(name);
    }

    @Override
    public void setPersistentField(String name, String value) throws USMStorageException, StorageAccessException {
        _persistentData.setField(name, value);
    }

    @Override
    public Map<String, String> getPersistentFields() {
        return _persistentData.getPersistentFields();
    }

    @Override
    public ConflictResolution getConflictResolution() {
        return _persistentData.getConflictResolution();
    }

    private ConflictResolution getConflictResolution(ConflictResolution conflictResolution) {
        return (conflictResolution == null) ? getConflictResolution() : conflictResolution;
    }

    @Override
    public void setConflictResolution(ConflictResolution resolution) throws StorageAccessException, USMStorageException {
        _persistentData.setConflictResolution(resolution);
    }

    @Override
    public void setContentTypeFilter(ContentType... usedContentTypes) throws USMStorageException, StorageAccessException {
        _persistentData.setContentTypeFilter(usedContentTypes);
    }

    @Override
    public void setFieldFilter(ContentType contentType, String... fieldsOfInterest) throws USMStorageException, StorageAccessException {
        setFieldFilter(contentType, generateFieldBitSet(contentType, fieldsOfInterest));
    }

    @Override
    public void setFieldFilter(ContentType contentType, BitSet fieldsOfInterest) throws StorageAccessException, USMStorageException {
        _persistentData.setFieldFilter(contentType, fieldsOfInterest);
    }

    @Override
    public void setRootFolders(String... folderIds) {
        _rootFolders = folderIds;
    }

    public SessionManagerImpl getSessionManager() {
        return _sessionManager;
    }

    @Override
    public BitSet getFieldFilter(ContentType contentType) {
        return _persistentData.getFieldFilter(contentType);
    }

    @Override
    public String toString() {
        return String.valueOf(getContextId()) + ':' + String.valueOf(getUserIdentifier()) + ':' + getUser() + ':' + getProtocol() + ':' + getDevice();
    }

    public String getDescription() {
        return "CID: " + getContextId() + ", ID: " + getUserIdentifier() + ", User: " + getUser() + ", Protocol: " + getProtocol() + ", Device: " + getDevice();
    }

    @Override
    public SyncResult syncChangesWithServer(String folderId, long timestamp, int resultLimit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException {
        Folder folder = findFolder(folderId);
        ContentType contentType = folder.getElementsContentType();
        if (contentType == null)
            return SyncResult.EMPTY_RESULT;
        if (_sessionManager.isEmailPullEnabled() && contentType.getID().equals(DefaultContentTypes.MAIL_ID))
            _lastEmailFilter = filter;
        try {
            if (storeResult) {
                // 0 is used as a flag that this folder is currently synced (to inform parallel running synchronizations)
                // Example: parallel running ping and sync on different nodes
                // (may happen if a client ignores cookies which are used by load balancers to delegate sessions onto one node) 
                _syncedFolderContent.put(folderId, 0L);
            }
            return _sessionManager.getIncrementalSyncer().syncChangesWithServer(
                createFolderElementsStorage(folder, filter, storeResult, timestamp, resultLimit),
                getConflictResolution(conflictResolution),
                getCachedFolderElements(folderId, contentType, timestamp),
                elements,
                timestamp,
                resultLimit,
                filter,
                _contentTypeSorters.get(contentType));
        } catch (FolderNotFoundException e) {
            markFolderHierarchyDirty();
            _syncedFolderContent.put(folderId, timestamp);
            throw e;
        } catch (USMException e) {
            _syncedFolderContent.remove(folderId);
            throw e;
        }
    }

    @Override
    public SyncResult syncWithServer(String folderId, int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException {
        Folder folder = findFolder(folderId);
        ContentType contentType = folder.getElementsContentType();
        if (contentType == null)
            return SyncResult.EMPTY_RESULT;
        if (_sessionManager.isEmailPullEnabled() && DefaultContentTypes.MAIL_ID.equals(folder.getElementsContentTypeID()))
            _lastEmailFilter = filter;
        try {
            if (storeResult)
                _syncedFolderContent.put(folderId, 0L);
            return _sessionManager.getSlowSyncer().syncWithServer(
                createFolderElementsStorage(folder, filter, storeResult, 0L, limit),
                getConflictResolution(conflictResolution),
                limit,
                filter,
                _contentTypeSorters.get(contentType),
                elements);
        } catch (FolderNotFoundException e) {
            markFolderHierarchyDirty();
            _syncedFolderContent.put(folderId, System.currentTimeMillis());
            throw e;
        } catch (USMException e) {
            _syncedFolderContent.remove(folderId);
            throw e;
        }
    }

    private FolderElementsStorage createFolderElementsStorage(Folder folder, DataObjectFilter filter, boolean storeResult, long oldTimestamp, int limit) {
        return new FolderElementsStorage(this, folder, (filter != null) ? filter : _syncServerObjectFilter, storeResult, oldTimestamp, limit);
    }

    public BitSet getValidFilter(ContentType elementsContentType) {
        if (elementsContentType == null)
            throw new IllegalArgumentException("No ContentType set");
        BitSet filter = _persistentData.getFieldFilter(elementsContentType);
        if (filter == null)
            throw new IllegalArgumentException("ContentType " + elementsContentType.getID() + " excluded from synchronization");
        return filter;
    }

    @Override
    public SyncResult syncFolderChangesWithServer(long timestamp, int resultLimit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... folderChanges) throws USMException {
        Folder[] cachedFolders = getCachedFolders(timestamp);
        removeUnusedFoldersFromSyncStateStorage(cachedFolders);
        SyncResult result = _sessionManager.getIncrementalSyncer().syncChangesWithServer(
            createFolderHierachyStorage(filter, storeResult, timestamp),
            getConflictResolution(conflictResolution),
            cachedFolders,
            folderChanges,
            timestamp,
            resultLimit,
            filter,
            getFolderSorter());
        return updateSynchronizationStateOfFolders(result);
    }

    @Override
    public SyncResult syncFoldersWithServer(int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... clientData) throws USMException {
        SyncResult result = _sessionManager.getSlowSyncer().syncWithServer(
            createFolderHierachyStorage(filter, storeResult, 0L),
            getConflictResolution(conflictResolution),
            limit,
            filter,
            getFolderSorter(),
            clientData);
        return updateSynchronizationStateOfFolders(result);
    }

    private Comparator<? super DataObject> getFolderSorter() {
        return _contentTypeSorters.get(_sessionManager.getFolderContentType());
    }

    private SyncResult updateSynchronizationStateOfFolders(SyncResult result) {
        if (MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION) {
            for (DataObject object : result.getChanges()) {
                Folder f = (Folder) object;
                _syncedFolderContent.remove(f.getID());
            }
        }
        for (DataObject object : result.getNewState()) {
            Folder f = (Folder) object;
            if (f.getElementsContentType() == null)
                _syncedFolderContent.put(f.getID(), f.getTimestamp());
        }
        if (_lastFolderChange == FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION)
            _lastFolderChange = 0L;
        return result;
    }

    public void markFolderHierarchyDirty() {
        foldersChanged(FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION);
    }

    private FolderHierarchyStorage createFolderHierachyStorage(DataObjectFilter filter, boolean storeResult, long oldTimestamp) {
        return new FolderHierarchyStorage(this, filter == null ? _syncServerObjectFilter : filter, storeResult, oldTimestamp, _rootFolders);
    }

    @Override
    public Folder[] getCachedFolders() throws StorageAccessException, USMStorageException {
        return getCachedFolders(_syncStateStorage.getNewestTimestamp(FOLDER_HIERARCHY_ID));
    }

    @Override
    public Folder[] getCachedFolders(long timestamp) throws StorageAccessException, USMStorageException {
        return getCachedFolders(_syncStateStorage.get(FOLDER_HIERARCHY_ID, timestamp), timestamp);
    }

    private Folder[] getCachedFolders(Serializable[][] values, long timestamp) {
        if (values == null)
            return null;
        FolderContentType folderContentType = _sessionManager.getFolderContentType();
        Folder[] result = new Folder[values.length];
        for (int i = 0; i < result.length; i++) {
            Folder f = folderContentType.newDataObject(this);
            try {
                f.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                logDeserializationError(FOLDER_HIERARCHY_ID, timestamp, e);
                return null;
            }
            result[i] = completeStructure(f);
        }
        return result;
    }

    @Override
    public Folder getCachedFolder(String folderID, long timestamp) throws StorageAccessException, USMStorageException {
        return getCachedFolder(_syncStateStorage.get(FOLDER_HIERARCHY_ID, timestamp), folderID, timestamp);
    }

    @Override
    public Folder getCachedFolder(String folderID) throws StorageAccessException, USMStorageException {
        Folder f = _sessionManager.getSpecialFolder(this, folderID);
        if (f != null)
            return f;
        long newestTS = _syncStateStorage.getNewestTimestamp(FOLDER_HIERARCHY_ID);
        return getCachedFolder(_syncStateStorage.get(FOLDER_HIERARCHY_ID, newestTS), folderID, newestTS);
    }

    private Folder getCachedFolder(Serializable[][] values, String folderID, long timestamp) {
        if (values != null && folderID != null) {
            Folder result = _sessionManager.getFolderContentType().newDataObject(this);
            for (int i = 0; i < values.length; i++) {
                try {
                    result.deserialize(timestamp, values[i]);
                } catch (DeserializationFailedException e) {
                    logDeserializationError(folderID, timestamp, e);
                    return null;
                }
                if (folderID.equals(result.getID()))
                    return completeStructure(result);
            }
        }
        return null;
    }

    public void logDeserializationError(String folderID, long timestamp, DeserializationFailedException e) {
        _sessionManager.getJournal().error(
            this + " Ignoring invalid serialized cache data for " + folderID + '(' + timestamp + "): " + e.getMessage());
    }

    @Override
    public DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp) throws StorageAccessException, USMStorageException {
        return getCachedFolderElements(folderId, contentType, timestamp, _syncStateStorage.get(getShortFolderID(folderId), timestamp));
    }

    @Override
    public DataObject[] getCachedFolderElements(String folderId, ContentType contentType) throws StorageAccessException, USMStorageException {
        if (folderId == null)
            return EMPTY_DATA_OBJECT_ARRAY;
        String shortID = getShortFolderID(folderId);
        long newestTS = _syncStateStorage.getNewestTimestamp(shortID);
        return getCachedFolderElements(folderId, contentType, newestTS, _syncStateStorage.get(shortID, newestTS));
    }

    private DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp, Serializable[][] values) {
        if (contentType == null)
            return EMPTY_DATA_OBJECT_ARRAY;
        if (values == null)
            return null;
        long start = System.currentTimeMillis();
        DataObject[] result = new DataObject[values.length];
        for (int i = 0; i < result.length; i++) {
            DataObject o = contentType.newDataObject(this);
            try {
                o.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                logDeserializationError(folderId, timestamp, e);
                return null;
            }
            result[i] = o;
        }
        long end = System.currentTimeMillis();
        if (_sessionManager.getJournal().isDebugEnabled())
            _sessionManager.getJournal().debug(
                this + " Deserialized " + result.length + " elements in " + folderId + " for " + timestamp + " in " + (end - start) + " ms.");
        return result;
    }

    @Override
    public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID, long timestamp) throws StorageAccessException, USMStorageException {
        return getCachedFolderElement(
            folderId,
            contentType,
            objectID,
            _syncStateStorage.get(getShortFolderID(folderId), timestamp),
            timestamp);
    }

    @Override
    public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID) throws StorageAccessException, USMStorageException {
        String shortID = getShortFolderID(folderId);
        long ts = _syncStateStorage.getNewestTimestamp(shortID);
        return getCachedFolderElement(folderId, contentType, objectID, _syncStateStorage.get(shortID, ts), ts);
    }

    private DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID, Serializable[][] values, long timestamp) {
        if (values != null) {
            DataObject result = contentType.newDataObject(this);
            for (int i = 0; i < values.length; i++) {
                try {
                    result.deserialize(timestamp, values[i]);
                } catch (DeserializationFailedException e) {
                    logDeserializationError(folderId, timestamp, e);
                    return null;
                }
                if (objectID.equals(result.getID()))
                    return result;
            }
        }
        return null;
    }

    public long storeCachedFolders(long oldTimestamp, long timestamp, DataObjectSet objects, boolean updatesPending, long syncStartTimestamp) throws USMException {
        checkIfObjectsAreFolders(objects.toArray());
        long result = internalStoreCachedDataObjects(FOLDER_HIERARCHY_ID, oldTimestamp, timestamp, objects);
        if (!updatesPending)
            _lastFolderSync = (syncStartTimestamp == 0L) ? result : syncStartTimestamp;
        return result;
    }

    public long storeCachedDataObjects(String folderID, long oldTimestamp, long timestamp, DataObjectSet objects, boolean updatesPending) throws USMException {
        // TODO: Why do we ignore the returned set?
        checkValidObjectsForSyncState(folderID, objects.toArray());
        long result = internalStoreCachedDataObjects(getShortFolderID(folderID), oldTimestamp, timestamp, objects);
        if (updatesPending)
            _syncedFolderContent.remove(folderID);
        else if (_syncedFolderContent.containsKey(folderID))
            _syncedFolderContent.put(folderID, result);
        return result;
    }

    private long internalStoreCachedDataObjects(String folderID, long oldTimestamp, long timestamp, DataObjectSet objects) throws StorageAccessException, USMStorageException {
        long resultingTimestamp = _syncStateStorage.put(folderID, timestamp, objects, oldTimestamp);
        if (resultingTimestamp != timestamp) {
            for (DataObject o : objects) {
                o.setTimestamp(resultingTimestamp);
                o.commitChanges();
            }
        }
        return resultingTimestamp;
    }

    @Override
    public DataObject readDataObject(String folderID, String objectID, String... fields) throws USMException {
        Folder folder = findFolder(folderID);
        return readDataObject(folder, objectID, generateFieldBitSet(folder.getElementsContentType(), fields));
    }

    @Override
    public DataObject readDataObject(String folderID, String objectID, BitSet fields) throws USMException {
        return readDataObject(findFolder(folderID), objectID, fields);
    }

    private Folder completeStructure(Folder f) {
        f.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(f.getElementsContentTypeID()));
        return f;
    }

    private DataObject readDataObject(Folder f, String objectID, BitSet fields) throws USMException {
        ContentType contentType = f.getElementsContentType();
        DataObject object = contentType.newDataObject(this);
        object.setID(objectID);
        object.setParentFolder(f);
        contentType.getTransferHandler().readDataObject(object, fields);
        return object;
    }

    @Override
    public Map<DataObject, USMException> updateDataObjects(DataObject... changedObjects) throws USMException {
        Map<DataObject, USMException> resultMap = new HashMap<DataObject, USMException>();

        // First, split between folder changes and element changes within folders
        List<DataObject> elementChanges = new ArrayList<DataObject>();
        List<DataObject> folderChanges = new ArrayList<DataObject>();
        for (DataObject o : changedObjects) {
            o.setFailed(true);
            if (o.getContentType().getID().equals(DefaultContentTypes.FOLDER_ID))
                folderChanges.add(o);
            else
                elementChanges.add(o);
        }
        if (!folderChanges.isEmpty()) {
            // perform all folder changes, update timestamp if possible
            long timestamp = folderChanges.get(0).getTimestamp();
            DataObjectSet newFolderHierarchy = new DataObjectSet();
            long newTimestamp = checkTimestampAndUpdate(
                newFolderHierarchy,
                folderChanges,
                timestamp,
                getCachedFolders(timestamp),
                "All Folders must have the same timestamp",
                resultMap);
            _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
            if (newTimestamp != 0) {
                long modifiedTimestamp = storeCachedFolders(timestamp, newTimestamp, newFolderHierarchy, true, 0L);
                for (DataObject folder : folderChanges)
                    folder.setTimestamp(modifiedTimestamp);
            }
        }
        // group elements by folder, operate on each folder separately
        while (!elementChanges.isEmpty()) {
            DataObject first = elementChanges.remove(elementChanges.size() - 1);
            long timestamp = first.getTimestamp();
            List<DataObject> currentFolderChanges = new ArrayList<DataObject>();
            currentFolderChanges.add(first);
            String folderId = first.getParentFolderID();
            for (int i = elementChanges.size() - 1; i >= 0; i--) {
                DataObject o = elementChanges.get(i);
                if (o.getParentFolderID().equals(folderId))
                    currentFolderChanges.add(elementChanges.remove(i));
            }
            _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, folderId));
            DataObjectSet newElements = new DataObjectSet();
            long newTimestamp = checkTimestampAndUpdate(
                newElements,
                currentFolderChanges,
                timestamp,
                getCachedFolderElements(folderId, first.getContentType(), timestamp),
                "All elements in a Folder must have the same timestamp",
                resultMap);
            if (newTimestamp != 0) {
                long modifiedTimestamp = storeCachedDataObjects(folderId, timestamp, newTimestamp, newElements, true);
                for (DataObject object : currentFolderChanges)
                    object.setTimestamp(modifiedTimestamp);
            }
        }
        return resultMap;
    }

    private long checkTimestampAndUpdate(DataObjectSet resultingServerObjects, List<DataObject> requests, long timestamp, DataObject[] oldState, String errorMessage, Map<DataObject, USMException> resultMap) {
        for (DataObject o : requests) {
            if (o.getTimestamp() != timestamp) {
                // If at least one element has a timestamp mismatch, fail on all elements in the folder
                for (DataObject o2 : requests) {
                    resultMap.put(o2, new TimestampMismatchException(
                        USMSessionManagementErrorCodes.TIMESTAMP_MISMATCH_ON_DATAOBJECT_UPDATE,
                        errorMessage));
                }
                return 0;
            }
        }
        ContentType type = requests.get(0).getContentType();
        ContentTypeTransferHandler handler = type.getTransferHandler();
        if (oldState == null) {
            for (DataObject o : requests) {
                callServerUpdatesAccordingToTimstamp(resultMap, handler, o);
            }
            for (DataObject o : requests)
                o.setFailed(false);
            return 0;
        }
        resultingServerObjects.clear();
        List<DataObject> creations = new ArrayList<DataObject>();
        List<DataObject> changes = new ArrayList<DataObject>();
        List<DataObject> deletions = new ArrayList<DataObject>();
        for (DataObject o : oldState)
            resultingServerObjects.add(o.createCopy(true));
        for (DataObject o : requests) {
            fillMapsAccordingToChangedState(resultMap, creations, changes, deletions, o);
        }

        ContentSyncerSupport.executeServerUpdates(_sessionManager, resultingServerObjects, creations, changes, deletions, resultMap);

        for (DataObject o : requests)
            o.setFailed(false);
        return ContentSyncerSupport.updateTimestampAndCommitChanges(resultingServerObjects);
    }

    private static void fillMapsAccordingToChangedState(Map<DataObject, USMException> resultMap, List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions, DataObject o) {
        switch (o.getChangeState()) {
        case CREATED:
            creations.add(o);
            return;
        case DELETED:
            deletions.add(o);
            return;
        case MODIFIED:
            changes.add(o);
            return;
        case UNMODIFIED:
            return;
        }
        resultMap.put(
            o,
            new InternalUSMException(
                USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER1,
                "Illegal ChangeState " + o.getChangeState()));
    }

    private static void callServerUpdatesAccordingToTimstamp(Map<DataObject, USMException> resultMap, ContentTypeTransferHandler handler, DataObject o) {
        switch (o.getChangeState()) {
        case CREATED:
            try {
                handler.writeNewDataObject(o);
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case DELETED:
            try {
                handler.writeDeletedDataObject(o);
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case MODIFIED:
            try {
                handler.writeUpdatedDataObject(o, o.getTimestamp());
                o.setTimestamp(0);
            } catch (USMException e) {
                resultMap.put(o, e);
            }
            return;
        case UNMODIFIED:
            return;
        }
        resultMap.put(
            o,
            new InternalUSMException(
                USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER2,
                "Illegal ChangeState " + o.getChangeState()));
    }

    private static BitSet generateFieldBitSet(ContentType contentType, String... fieldsOfInterest) {
        ContentTypeField[] fields = contentType.getFields();
        BitSet fieldSet = new BitSet();
        for (String fieldName : fieldsOfInterest) {
            generateFieldsOfInterestBitSet(contentType, fields, fieldSet, fieldName);
        }
        return fieldSet;
    }

    private static void generateFieldsOfInterestBitSet(ContentType contentType, ContentTypeField[] fields, BitSet fieldSet, String fieldName) {
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldName().equals(fieldName)) {
                fieldSet.set(i);
                return;
            }
        }
        throw new IllegalArgumentException("Field " + fieldName + " not provided by ContentType " + contentType);
    }

    @Override
    public Folder findFolder(String folderID) throws USMException {
        Folder folder = _sessionManager.getSpecialFolder(this, folderID);
        if (folder != null)
            return folder;
        folder = getCachedFolder(folderID);
        if (folder != null)
            return folder;
        FolderContentType type = _sessionManager.getFolderContentType();
        Folder result = type.newDataObject(this);
        result.setID(folderID);
        type.getTransferHandler().readDataObject(result, getFieldFilter(type));
        result.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(result.getElementsContentTypeID()));
        return result;
    }

    @Override
    public void addChangeListener(SessionChangeListener listener) {
        _changeListeners.add(listener);
    }

    @Override
    public void removeChangeListener(SessionChangeListener listener) {
        _changeListeners.remove(listener);
    }

    public void foldersChanged(long timestamp) {
        _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
        if (_changeListeners.size() > 0) {
            ObjectChanges changes = new ObjectChangesImpl(true);
            for (SessionChangeListener listener : _changeListeners) {
                try {
                    listener.changesOccurred(changes);
                } catch (Exception e) {
                    _sessionManager.getJournal().error(
                        this + " Exception while notifying listener " + listener + " of folder structure change",
                        e);
                }
            }
        }
        _lastFolderChange = timestamp;
        _sessionManager.getJournal().debug(this + " lastFolderChange " + _lastFolderChange);
        notifyOnWaitForChanges();
    }

    public void folderContentChanged(long timestamp) {
        try {
            folderContentChanged(timestamp, getAllFolderIDs(getCachedFolders()));
        } catch (StorageAccessException e) {
            _sessionManager.getJournal().error(this + " Database access error during OX event notification", e);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.OX_EVENT_DB_ERROR_1,
                this + " Database access error during OX event notification",
                e);
        } catch (USMStorageException e) {
            _sessionManager.getJournal().error(this + " SQL error during OX event notification", e);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.OX_EVENT_DB_ERROR_2,
                this + " SQL error during OX event notification",
                e);
        }
    }

    public void folderContentChanged(long timestamp, String... folderIds) {
        Log journal = _sessionManager.getJournal();
        if (journal.isDebugEnabled())
            journal.debug(this + " Folder content changed for " + Arrays.toString(folderIds) + ", timestamp: " + timestamp);
        if (_changeListeners.size() > 0) {
            ObjectChanges changes = new ObjectChangesImpl(false, folderIds);
            for (SessionChangeListener listener : _changeListeners) {
                try {
                    listener.changesOccurred(changes);
                } catch (Exception e) {
                    _sessionManager.getJournal().error(
                        this + " Exception while notifying listener " + listener + " of content change in folders " + Arrays.toString(folderIds),
                        e);
                }
            }
        }
        for (String id : folderIds) {
            if (null != id) {
                journal.debug(this + " Removing from synced folders: " + id);
                _syncedFolderContent.remove(id);
                _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, id));
                if ("1".equals(id)) { // for Remote Events when folderID == 1, it means that the default inbox has been changed
                    if (_defaultInboxId != null) {
                        if (journal.isDebugEnabled())
                            journal.debug(this + " Removing default inbox from synced folders: " + _defaultInboxId);
                        _syncedFolderContent.remove(_defaultInboxId);
                        _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, _defaultInboxId));
                    }
                }
            }
        }
        notifyOnWaitForChanges();
    }

    public void notifyOnWaitForChanges() {
        synchronized (_waitObject) {
            _waitObject.notifyAll();
        }
    }

    @Override
    public boolean needsSynchronization(String folderID) {
        return Toolkit.provided(folderID) ? !_syncedFolderContent.containsKey(folderID) : !isFolderHierarchySynchronized();
    }

    private boolean isFolderHierarchySynchronized() {
        return _lastFolderSync != 0L && _lastFolderChange != FOLDER_HIERARCHY_NEEDS_SYNCHRONIZATION && _lastFolderSync >= _lastFolderChange;
    }

    @Override
    public ObjectChanges waitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException {
        Log journal = _sessionManager.getJournal();
        if (journal.isDebugEnabled())
            journal.debug(this + " waitforChanges(" + watchFolderChanges + "," + Arrays.toString(contentChangeParentIDsToWatch) + "," + timeout + "), lastFolderChange=" + _lastFolderChange + ", lastFolderSync=" + _lastFolderSync);
        ObjectChanges result = waitForChanges(
            _sessionManager.isEmailPullEnabled(),
            watchFolderChanges,
            contentChangeParentIDsToWatch,
            timeout);
        if (journal.isDebugEnabled())
            journal.debug(this + " waitforChanges Result=" + result);
        return result;
    }

    private ObjectChanges waitForChanges(boolean emailPullEnabled, boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException {
        Log journal = _sessionManager.getJournal();
        ObjectChanges result;
        Folder[] folders = null;
        if (contentChangeParentIDsToWatch == null) {
            folders = getCachedFolders();
            contentChangeParentIDsToWatch = getAllFolderIDs(folders);
        }
        synchronized (_waitObject) {
            result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
            if (result != null) {
                if (!_sessionInitializer.allowsUnlimitedWaitForChangesOnModifiedFolders() && ++_directPingResponses > DIRECT_PING_RESPONSE_LIMIT && "EAS".equals(getProtocol())) {
                    // fix for bug 22741 - enable the ping response limit ONLY for protocols that require it (e.g. EAS)
                    if (journal.isDebugEnabled())
                        _sessionManager.getJournal().debug(
                            this + " Too many pings without sync, marked as synchronized: " + Arrays.toString(contentChangeParentIDsToWatch));
                    // May happen e.g. if ping and sync run on different nodes and the ping request do not get the information that a sync request has already run 
                    for (String id : contentChangeParentIDsToWatch) {
                        _syncedFolderContent.put(id, 0L);
                    }
                    return null;
                }
                return result;
            }
        }
        _directPingResponses = 0;
        emailPullEnabled &= contentChangeParentIDsToWatch.length > 0;
        long start = System.currentTimeMillis();
        long end = start + timeout;
        if (emailPullEnabled) {
            if (folders == null)
                folders = getCachedFolders();
            List<Folder> emailFolders = new ArrayList<Folder>();
            if (folders != null) {
                DataObjectSet folderSet = new DataObjectSet(folders);
                for (String folderId : contentChangeParentIDsToWatch) {
                    Folder f = (Folder) folderSet.get(folderId);
                    if (f != null && DefaultContentTypes.MAIL_ID.equals(f.getElementsContentTypeID()))
                        emailFolders.add(f);
                }
            }
            if (!emailFolders.isEmpty()) {
                long pullTime = _sessionManager.computePullTime(start, end);
                if (pullTime > 0) {
                    if (journal.isDebugEnabled())
                        journal.debug(this + " Waiting " + (pullTime - start) + "ms until pull timeout");
                    result = doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, pullTime);
                    if (result != null)
                        return result;
                    if (journal.isDebugEnabled())
                        _sessionManager.getJournal().debug(this + " Performing email pull");
                    result = performEmailPull(emailFolders);
                    if (result != null)
                        return result;
                }
                if (journal.isDebugEnabled())
                    journal.debug(this + " Waiting " + (end - System.currentTimeMillis()) + "ms after pull timeout");
            }
        }
        return doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, end);
    }

    private ObjectChanges performEmailPull(List<Folder> emailFolders) {
        Log journal = _sessionManager.getJournal();
        List<String> changedFolders = new ArrayList<String>();
        for (Folder f : emailFolders) {
            String folderId = f.getID();
            Long timestamp = _syncedFolderContent.get(folderId);
            if (timestamp == null)
                changedFolders.add(folderId);
            else {
                // if timestamp==0 probably sync and ping are running on different nodes or a ping and a sync are running parallel on the same node.
                // In the first case we can not assure that the time window used to determine changes since the last sync is correct.
                // Therefore we avoid the sync and belief on the normal event mechanism.
                if (timestamp != 0L) {
                    Long newestTimestamp = getNewestTimestamp(folderId);
                    if (newestTimestamp != null && newestTimestamp != 0L) {
                        try {
                            SyncResult syncResult = syncChangesWithServer(folderId, newestTimestamp, NO_LIMIT, _lastEmailFilter, false, null);
                            if (syncResult.getChanges().length > 0) {
                                changedFolders.add(folderId);
                                _syncedFolderContent.remove(folderId);
                            }
                        }
                        catch (USMException e) {
                            if (journal.isDebugEnabled())
                                journal.debug(this + "sync within email pull failed: " + e.getMessage());
                        }
                    }
                }
            }
        }
        return ObjectChangesImpl.create(false, changedFolders);
    }

    private ObjectChanges doWaitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long end) throws InterruptedException {
        ObjectChanges result = null;
        for (long waitTime = end - System.currentTimeMillis(); waitTime > 0; waitTime = end - System.currentTimeMillis()) {
            synchronized (_waitObject) {
                result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
                if (result != null || !_sessionManager.isWaitForChangesEnabled(getProtocol()))
                    return result;
                _waitObject.wait(waitTime);
            }
        }
        return result;
    }

    private ObjectChanges extractFolderChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch) {
        boolean folderStructureChanged = watchFolderChanges && !isFolderHierarchySynchronized();
        List<String> changedFolders = new ArrayList<String>();
        for (String folderId : contentChangeParentIDsToWatch) {
            if (!_syncedFolderContent.containsKey(folderId))
                changedFolders.add(folderId);
        }
        return ObjectChangesImpl.create(folderStructureChanged, changedFolders);
    }

    private static String[] getAllFolderIDs(Folder[] folders) {
        if (folders == null)
            return new String[0];
        String[] folderIds = new String[folders.length + SessionManagerImpl.SPECIAL_FOLDERS.size()];
        for (int i = 0; i < folders.length; i++)
            folderIds[i] = folders[i].getID();
        int foldersLength = folders.length;
        for (int i = 0; i < SessionManagerImpl.SPECIAL_FOLDERS.size(); i++)
            folderIds[foldersLength + i] = SessionManagerImpl.SPECIAL_FOLDERS.get(i);
        return folderIds;
    }

    @Override
    public Object getCustomProperty(String key) {
        return _customProperties.get(key);
    }

    @Override
    public Object setCustomProperty(String key, Object value) {
        synchronized (_customPropertyLimits) {
            _customPropertyLimits.remove(new CustomPropertyLimit(key));
        }
        Object oldValue = (value == null) ? _customProperties.remove(key) : _customProperties.put(key, value);
        return oldValue;
    }

    @Override
    public Object setCustomProperty(String key, Object value, long timeout) {
        synchronized (_customPropertyLimits) {
            Object oldValue = (value == null) ? _customProperties.remove(key) : _customProperties.put(key, value);
            if (oldValue != null)
                _customPropertyLimits.remove(new CustomPropertyLimit(key));
            _customPropertyLimits.add(new CustomPropertyLimit(key, timeout));
            return oldValue;
        }
    }

    public int removeOutdatedCustomProperties(long now) {
        int count = 0;
        synchronized (_customPropertyLimits) {
            for (CustomPropertyLimit limit = _customPropertyLimits.peek(); limit != null && limit.getStorageLimit() < now; limit = _customPropertyLimits.peek()) {
                if (_customProperties.remove(limit.getKey()) != null)
                    count++;
                _customPropertyLimits.poll();
            }
        }
        return count;
    }

    @Override
    public DataObjectFilter getSyncServerObjectFilter() {
        return _syncServerObjectFilter;
    }

    @Override
    public void setSyncServerObjectFilter(DataObjectFilter filter) {
        _syncServerObjectFilter = filter;
    }

    @Override
    public TimeZone getUserTimeZone() {
        return _userTimeZone;
    }

    @Override
    public long getEndDate() {
        return _endDate;
    }

    @Override
    public long getStartDate() {
        return _startDate;
    }

    @Override
    public void setEndDate(long date) {
        _endDate = date;
    }

    @Override
    public void setStartDate(long date) {
        _startDate = (date < START_DATE_LIMIT) ? START_DATE_LIMIT : date;
    }
    
    @Override
    public long getCalendarEndDate() {
        return _endDateCalendar;
    }

    @Override
    public long getCalendarStartDate() {
        return _startDateCalendar;
    }

    @Override
    public void setCalendarEndDate(long date) {
        _endDateCalendar = date;
    }

    @Override
    public void setCalendarStartDate(long date) {
        _startDateCalendar = (date < START_DATE_LIMIT) ? START_DATE_LIMIT : date;
    }

    @Override
    public int getMailLimit() {
        return _mailLimit;
    }

    @Override
    public void setMailLimit(int limit) {
        _mailLimit = limit;
    }

    @Override
    public int getObjectsLimit() {
        return _sessionManager.getMaxObjectsPerFolder();
    }

    @Override
    public int getPIMAttachmentsCountLimit() {
        return _sessionManager.getAttachmentCountLimitPerObject();
    }

    @Override
    public long getPIMAttachmentsSizeLimit() {
        return _sessionManager.getAttachmentSizeLimitPerObject();
    }

    @Override
    public String getFolderID(String shortFolderID) throws USMStorageException {
        if (shortFolderID.length() > 0) {
            switch (shortFolderID.charAt(0)) {
            case 'D':
                try {
                    return getLongID(Integer.parseInt(shortFolderID.substring(1)));
                } catch (NumberFormatException ignored) {
                    // Common Exception containing error message will be generated at end of method
                } catch (StorageAccessException e) {
                    throw new USMStorageException(e.getErrorCode(), e.getMessage());
                }
                break;
            case 'F':
                return shortFolderID.substring(1);
            case 'U':
                try {
                    return getOXObjectID(shortFolderID.substring(1));
                } catch (StorageAccessException e) {
                    throw new USMStorageException(e.getErrorCode(), e.getMessage());
                }
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Illegal short folder ID " + shortFolderID);
    }

    /**
     * Gets the OX Object id for the object based on the known UUID.
     * 
     * @param uuid
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    private String getOXObjectID(String uuid) throws StorageAccessException, USMStorageException {
        long[] allTimestamps = _syncStateStorage.getSortedTimestamps(FOLDER_HIERARCHY_ID);
        for (int i = allTimestamps.length - 1; i >= 0; i--) {
            Folder[] allFolders = getCachedFolders(allTimestamps[i]);
            for (Folder folder : allFolders) {
                if (uuid.equals(folder.getUUID().toString()))
                    return folder.getID();
            }
        }
        return null;
    }

    /**
     * Gets the OX Object id for the object based on the known UUID.
     * 
     * @param uuid
     * @param folderID
     * @param contentType
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    private String getOXObjectID(UUID uuid, String folderID, ContentType contentType) throws StorageAccessException, USMStorageException {
        long[] allTimestamps = _syncStateStorage.getSortedTimestamps(getShortFolderID(folderID));
        for (int i = allTimestamps.length - 1; i >= 0; i--) {
            DataObject[] allObjects = getCachedFolderElements(folderID, contentType, allTimestamps[i]);
            for (DataObject dataObject : allObjects) {
                if (uuid.equals(dataObject.getUUID()))
                    return dataObject.getID();
            }
        }
        return null;
    }

    /**
     * Used to get the long id for a folder if it is saved under an integer short id. This method is used for teh old mapping which saved
     * the short ids under "D"+INT in the table usmIDMapping.
     * 
     * @param shortID
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    private String getLongID(int shortID) throws USMStorageException, StorageAccessException {
        return _sessionManager.getMappingService().getLongID(getSessionId(), getContextId(), shortID);
    }

    @Override
    public String getShortFolderID(String folderID) throws StorageAccessException, USMStorageException {
        return getShortFolderID(folderID, null);
    }

    @Override
    public String getShortFolderID(String folderID, UUID uuid) throws StorageAccessException, USMStorageException {
        if (folderID.length() < getFolderIdLengthLimit())
            return "F" + folderID;
        Integer shortID = findOldShortID(folderID);
        if (shortID != null)
            return "D" + shortID;
        if (uuid != null)
            return "U" + uuid;
        return "U" + getUUID(folderID);
    }
    
    private String getUUID(String folderID) throws StorageAccessException, USMStorageException {
        Folder folder = getCachedFolder(folderID);
        UUID uuid = folder != null ? folder.getUUID() : null;
        if (uuid == null) {
            try {
                uuid = UUIDToolkit.generateUUID(
                    getContextUUID(),
                    _sessionManager.getFolderContentType().getCode(),
                    Integer.parseInt(folderID));
            } catch (NumberFormatException e) {
                uuid = UUID.randomUUID();
            }
            if (folder != null) {
                folder.setUUID(uuid);
            }
        }
        return uuid.toString();
    }

    @Override
    @Deprecated
    public long storeSyncState(long timestamp, String folderID, DataObject[] objects) throws USMException {
        return storeSyncState(0L, timestamp, folderID, objects);
    }

    @Override
    public long storeSyncState(long timestampToKeep, long timestamp, String folderID, DataObject[] objects) throws USMException {
        DataObjectSet set = checkValidObjectsForSyncState(folderID, objects);
        _sessionManager.getOXDataCache().removeCachedData(new OXDataCacheID(this, folderID));
        return storeCachedDataObjects(folderID, timestampToKeep, timestamp, set, false);
    }

    private static DataObjectSet checkValidObjectsForSyncState(String folderID, DataObject[] objects) throws USMException {
        DataObjectSet set = new DataObjectSet();
        ContentType t = null;
        for (DataObject o : objects) {
            if (t == null)
                t = o.getContentType();
            else if (!o.getContentType().getID().equals(t.getID())) {
                throw new USMException(
                    USMSessionManagementErrorCodes.ILLEGAL_OBJETCS_TO_BE_SAVED_IN_SYNC_STATE,
                    "DataObjects of different ContentTypes");
            }
            if (o.getContentType().getCode() != DefaultContentTypes.RESOURCES_CODE && o.getContentType().getCode() != DefaultContentTypes.GROUPS_CODE)
                o.setParentFolderID(folderID);
            set.add(o);
        }
        return set;
    }

    @Override
    public long storeSyncState(long timestampToKeep, long timestamp, DataObject[] objects) throws USMException {
        checkIfObjectsAreFolders(objects);
        _sessionManager.getOXDataCache().removeCachedData(_folderHierarchyCacheID);
        DataObjectSet set = new DataObjectSet(objects);
        return storeCachedFolders(timestampToKeep, timestamp, set, false, 0L);
    }

    private static void checkIfObjectsAreFolders(DataObject[] objects) throws USMException {
        for (DataObject o : objects) {
            if (o.getContentType().getCode() != DefaultContentTypes.FOLDER_CODE)
                throw new USMException(
                    USMSessionManagementErrorCodes.ILLEGAL_OBJETCS_TO_BE_SAVED_IN_SYNC_STATE,
                    "DataObjects are not folders");
        }
    }

    @Override
    public void endSyncronization() throws USMStorageException, StorageAccessException {
        _syncStateStorage.removeAllObjectsForSession();
        _persistentData.removeAllDBFieldsForSession();
        _sessionManager.getSessionStorage().removeSession(this);
    }

    public PersistentSessionData getPersistentSessionData() {
        return _persistentData;
    }

    @Override
    public UUID getContextUUID() throws StorageAccessException, USMStorageException {
        return _sessionManager.getUUIDService().getContextUUID(getContextId());
    }

    @Override
    public String getMappedObjectId(ContentType contentType, UUID uuid, String parentFolderID) throws UUIDNotFoundException {
        try {
            return getOXObjectID(uuid, parentFolderID, contentType);
        } catch (StorageAccessException e) {
            throw new UUIDNotFoundException(e.getErrorCode(), e.getMessage());
        } catch (USMStorageException e) {
            throw new UUIDNotFoundException(e.getErrorCode(), e.getMessage());
        }
    }

    @Override
    public UUID getUUID(ContentType contentType, String objectId) throws StorageAccessException, USMStorageException {
        try {
            return UUIDToolkit.generateUUID(getContextUUID(), contentType.getCode(), Integer.parseInt(objectId));
        } catch (NumberFormatException e) {
            return UUID.randomUUID();
        }
    }

    @Override
    public Folder getDummyContentTypeFolder(ContentType type) {
        return _sessionManager.getSpecialFolder(this, type.getID());
    }

    @Override
    public ContentType[] getContentTypes() {
        return _persistentData.getContentTypes();
    }

    @Override
    public void remapCachedData(String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException {
        _syncStateStorage.remapStates(oldObjectID, newObjectID);
    }

    public void setLastAccessCheck(long lastAccessCheck) {
        _lastAccessCheck = lastAccessCheck;
    }

    public long getLastAccessCheck() {
        return _lastAccessCheck;
    }

    @Override
    public JSONObject getOXUserConfiguration(String... path) throws AuthenticationFailedException, OXCommunicationException {
        return _sessionManager.getOXAjaxAccess().getConfiguration(this, path);
    }

    public boolean checkProperDBStorage() throws StorageAccessException, USMStorageException {
        return _persistentData.checkProperDBStorage();
    }

    @Override
    public void setContentTypeSorter(ContentType contentType, Comparator<? super DataObject> comparator) {
        if (comparator == null)
            _contentTypeSorters.remove(contentType);
        else
            _contentTypeSorters.put(contentType, comparator);
    }

    @Override
    public OXConnectionInformation getOXConnectionData() {
        return _completeSessionID.getOXConnectionInformation();
    }

    @Override
    public void setOXConnectionData(OXConnectionInformation data) {
        _completeSessionID.setOXConnectionInformation(data);
    }

    public OXDataCacheID getFolderHierarchyCacheID() {
        return _folderHierarchyCacheID;
    }

    @Override
    public int getFolderIdLengthLimit() {
        return _folderIdLengthLimit > 0 ? _folderIdLengthLimit : FOLDER_ID_LENGTH_LIMIT;
    }

    @Override
    public void setFolderIdLengthLimit(int limit) {
        _folderIdLengthLimit = limit;
    }

    @Override
    public String getClientIp() {
        return _clientIP;
    }

    public void setClientIp(String ip) {
        _clientIP = ip;
    }

    @Override
    public void updateClientIp(String ip) {
        if (_clientIP.equals(ip))
            return;
        _clientIP = ip;
        Log journal = _sessionManager.getJournal();
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("clientIP", _clientIP);
        try {
            JSONResult result = _sessionManager.getOXAjaxAccess().doPost("login", "changeip", this, parameters);
            if (result.getResultType() == JSONResultType.Error && journal.isDebugEnabled())
                journal.debug(this + " Changing ip failed: " + result.toString());
        } catch (AuthenticationFailedException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " Authentication failed on changing client ip.", e);
        } catch (OXCommunicationException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " OX Communication Exception on changing client ip.", e);
        }
    }

    @Override
    public Map<String, String> getXRequestHeaders() {
        return _xHeaders;
    }

    public void setXRequestHeaders(Map<String, String> headers) {
        _xHeaders = headers;
    }

    @Override
    public long getNewestTimestamp(String folderID) {
        Log journal = _sessionManager.getJournal();
        try {
            String shortID = getShortFolderID(folderID);
            return _syncStateStorage.getNewestTimestamp(shortID);
        } catch (StorageAccessException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " DatabaseAccessException on getting the newest timestamp for folder: " + folderID);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.SESSION_TIMESTAMP_DB_ERROR_1,
                this + " DatabaseAccessException on getting the newest timestamp for folder: " + folderID,
                e);
        } catch (USMStorageException e) {
            if (journal.isDebugEnabled())
                journal.debug(this + " USMSQLException on getting the newest timestamp for folder: " + folderID);
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.SESSION_TIMESTAMP_DB_ERROR_2,
                this + " USMSQLException on getting the newest timestamp for folder: " + folderID,
                e);
        }
    }

    @Override
    public void invalidateCachedData(String folderID) {
        _sessionManager.getOXDataCache().removeCachedData(
            Toolkit.provided(folderID) ? new OXDataCacheID(this, folderID) : _folderHierarchyCacheID);
        if (Toolkit.provided(folderID))
            _syncedFolderContent.remove(folderID);
        else
            markFolderHierarchyDirty();
    }

    public SyncStateStorage getDataObjectCache() {
        return _syncStateStorage;
    }

    @Override
    public Lock getOXConnectionLock() {
        return _oxLock;
    }

    public CompleteSessionID getCompleteSessionID() {
        return _completeSessionID;
    }

    @Override
    public String tryLock(String id, String acquirer) {
        return SynchronizationLock.lock(_completeSessionID.getSessionID(), id, acquirer);
    }

    @Override
    public void unlock(String id) {
        SynchronizationLock.unlock(_completeSessionID.getSessionID(), id);
    }

    public SessionInitializer getSessionInitializer() {
        return _sessionInitializer;
    }

    @Override
    public Map<String, String> getMailFolderIdSeparators() {
        return _mailFolderIdSeparators;
    }

    private void removeUnusedFoldersFromSyncStateStorage(Folder[] cachedFolders) throws StorageAccessException, USMStorageException {
        if (cachedFolders == null)
            return;
        List<String> shortIDs = new ArrayList<String>();
        List<String> folderIDs = new ArrayList<String>();
        for (DataObject o : cachedFolders) {
            String id = o.getID();
            folderIDs.add(id);
            shortIDs.add(getShortFolderID(id));
        }
        shortIDs.add(FOLDER_HIERARCHY_ID);
        for (String specialFolder : _sessionManager.getSpecialFolders())
            shortIDs.add(getShortFolderID(specialFolder));
        _syncStateStorage.retain(shortIDs.toArray(new String[shortIDs.size()]));
    }

    private Integer findOldShortID(String id) throws StorageAccessException, USMStorageException {
        return _sessionManager.getMappingService().findShortID(_uniqueSessionID, _completeSessionID.getUserID().getContextId(), id);
    }
}
