/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.database.ox.USMDatabaseOXErrorCodes;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class USMDeleteStoredProceduresUpdateTaskV2 extends USMBaseUpdateTaskV2 {

    private static final Log LOG = LogFactory.getLog(USMDeleteStoredProceduresUpdateTaskV2.class);

    public final static String[] SEQUENCE_TABLES_NAMES = { "sequence_usmSessionId", "sequence_usmIdMapping" };

	public USMDeleteStoredProceduresUpdateTaskV2(USMUpdateTaskProvider provider) {
		super(provider);
	}

	@Override
	protected void doPerform(PerformParameters params) throws OXException {
        int contextId = params.getContextId();
        for (int i = 0; i < SEQUENCE_TABLES_NAMES.length; i++) {
            Connection con = null;
            try {
                con = _provider.getDatabaseService().getForUpdateTask(contextId);
                String tablename = SEQUENCE_TABLES_NAMES[i];
                if (procedureExists(tablename, con.getMetaData()))
                    dropSequence(tablename, con, true);
            } catch (USMStorageException e) {
                throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
            } catch (SQLException e) {
                throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
            } finally {
                if (con != null)
                    _provider.getDatabaseService().backWritable(contextId, con);
            }
        }
    }

    private static void dropSequence(String sequenceName, Connection con, boolean isAutocommit) throws USMStorageException {
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                if (isAutocommit)
                    con.setAutoCommit(false);
                stmt.execute(SQLToolkit.dropSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Dropping of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                if (isAutocommit)
                    con.rollback();
                if (e.getMessage().contains("alter routine command denied to user")) {
                    LOG.info("Dropping of sequence " + sequenceName + " was not possible. Probably it did not exist anyway.");
                }
                else
                    throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                if (isAutocommit)
                    con.setAutoCommit(true);
                Toolkit.close(stmt);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }
    }

    private static boolean procedureExists(final String procedureName, final DatabaseMetaData dbmd) throws SQLException {
        ResultSet resultSet = null;
        try {
            resultSet = dbmd.getProcedures(null, null, "get_" + procedureName + "_id");
            if (resultSet == null) return false;
            return resultSet.next();
        } finally {
            Toolkit.close(resultSet);
        }
    }
    
    @Override
    public String[] getDependencies() {
		return new String[] { USMTablesUpdateTaskV2.class.getName() };
	}
}
