
package com.openexchange.usm.mapping.storage.sql.impl;

/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.util.Toolkit;

/**
 * This is a reduced version of the old Folder id short to long mapping. It will be used to read the mappings which already exist in the db
 * and are reported to clients.
 * 
 * @author <a href="mailto:ldo @microdoc.de">Ljupka Grabsch</a>
 */
public class FolderIDMappingSQL implements FolderIDMappingStorage {

    private static class SessionIdMapping {

        private final Map<String, Integer> _longToShortMap = new HashMap<String, Integer>();

        private final TIntObjectMap<String> _shortToLongMap = new TIntObjectHashMap<String>();

        public SessionIdMapping() {
            super();
        }

        public void addMapping(String longID, int shortID) {
            _longToShortMap.put(longID, shortID);
            _shortToLongMap.put(shortID, longID);
        }

        public Integer getShortID(String longID) {
            return _longToShortMap.get(longID);
        }

        public String getLongID(int shortID) {
            return _shortToLongMap.get(shortID);
        }
    }

    public static final int DB_ACCESS_FAILED = 0x610006;

    public static final int SELECT_FAILED_NUMBER1 = 0x610002;

    private static final String UNIQUE_SESSION_IDENTIFIER = " cid = ? AND usmSessionId = ?";

    private final Log _journal = LogFactory.getLog(FolderIDMappingSQL.class);

    // TODO Best solution would be to move the mapping to the USM-session itself, instead of using a separate storage per session here
    private final Map<ContextSessionID, SessionIdMapping> _sessionMappings = new ConcurrentHashMap<ContextSessionID, SessionIdMapping>();

    private final DatabaseAccess _dbAccess;

    public FolderIDMappingSQL(DatabaseAccess dbAccess) {
        _dbAccess = dbAccess;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage#getLongID(int, int, int)
     */
    @Override
    public String getLongID(int sessionID, int contextID, int shortID) throws USMStorageException, StorageAccessException {
        String longID = getSessionIdMapping(sessionID, contextID).getLongID(shortID);
        if (longID == null) {
            _journal.warn(sessionID + " No long ID for short ID " + shortID);
        }
        return longID;
    }

    private SessionIdMapping readCacheFromDB(int contextID, int sessionID) throws StorageAccessException, USMStorageException {
        SessionIdMapping mapping = new SessionIdMapping();
        EncapsulatedConnection con = _dbAccess.getReadOnly(contextID);
        PreparedStatement statement = null;
        ResultSet result = null;
        try {
            statement = con.prepareStatement("select shortID,LongID from usmIdMapping where " + UNIQUE_SESSION_IDENTIFIER);
            addSessionIdentifierParamsToStmt(statement, contextID, sessionID);
            result = statement.executeQuery();
            while (result.next()) {
                int shortID = result.getInt(1);
                String longID = result.getString(2);
                mapping.addMapping(longID, shortID);
            }
        } catch (SQLException e) {
            String errorMessage = sessionID + " SQL error reading cache from DB";
            _journal.error(errorMessage, e);
            throw new USMStorageException(SELECT_FAILED_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(result);
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return mapping;
    }

    private void addSessionIdentifierParamsToStmt(PreparedStatement statement, int contextID, int sessionID) throws SQLException {
        statement.setInt(1, contextID);
        statement.setInt(2, sessionID);
    }

    @Override
    public Integer findShortID(int sessionID, int contextID, String longID) throws StorageAccessException, USMStorageException {
        // Since v.7.4.1 USM doesn't save any more mappings in a storage system,
        // we just read the cache once:
        return getSessionIdMapping(sessionID, contextID).getShortID(longID);
    }

    private SessionIdMapping getSessionIdMapping(int sessionID, int contextID) throws StorageAccessException, USMStorageException {
        ContextSessionID id = new ContextSessionID(contextID, sessionID);
        SessionIdMapping mapping = _sessionMappings.get(id);
        if (mapping == null) {
            mapping = readCacheFromDB(contextID, sessionID);
            _sessionMappings.put(id, mapping);
        }
        return mapping;
    }
}
