/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.storage.sql.impl;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.api.session.assets.SessionInformation;
import com.openexchange.usm.api.session.assets.USMSessionID;
import com.openexchange.usm.api.session.storage.PersistentSessionDataStorage;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;
import com.openexchange.usm.util.Toolkit;

/**
 * {@link PersistentSessionDataStorageSQL}
 * 
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class PersistentSessionDataStorageSQL implements PersistentSessionDataStorage {

    private static final Log LOG = LogFactory.getLog(PersistentSessionDataStorageSQL.class);

    private static final String FIELD_PREFIX = "F_";

    private static final String LAST_ACCESS_FIELD_NAME = "USM_ACCESS";

    private final DatabaseAccess dbAccess;

    private final static String[] USM_TABLES = { "easUIDMapping", "USMDataStorage", "USMSessionFields", "usmIdMapping", "USMSession" };

    /**
     * Initializes a new {@link PersistentSessionDataStorageSQL}.
     * 
     * @param Access
     */
    public PersistentSessionDataStorageSQL(DatabaseAccess dba) {
        dbAccess = dba;
    }


    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#storeFieldValue(com.openexchange.usm.api.session.Session, java.lang.String, java.lang.String)
     */
    @Override
    public void storeFieldValue(Session session, String key, String value) throws USMStorageException, StorageAccessException {
        if(dbAccess.supportsOnDuplicateKeyUpdateStatement())
            insertOnDuplicateKeyUpdateField(session, key, value);
        else
            insertOrUpdateField(session, key, value);
    }

    private void insertOnDuplicateKeyUpdateField(Session session, String key, String value) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = dbAccess.getWritable(session.getContextId());
        PreparedStatement statement = null;
        try {
            statement = con.prepareStatement("insert into USMSessionFields (cid, usmSessionId, Field, Value) values(?,?,?,?) on duplicate key update value = ?");
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.setString(3, key);
            statement.setString(4, value);
            statement.setString(5, value);

            if (statement.executeUpdate() == 0)
                LOG.warn(session + " Setting field " + key + " in DB did not change any rows");

        } catch (SQLException e) {
            String errorMessage = session + " SQL error while updating field " + key;
            LOG.error(errorMessage, e);
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER3, errorMessage, e);
        } finally {
            if (con != null) {
                Toolkit.close(statement);
                Toolkit.close(con);
            }
        }
    }

    private void insertOrUpdateField(Session session, String key, String value) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getWritable(session.getContextId());
            con.setAutoCommit(false);
            statement = con.prepareStatement("select Field from USMSessionFields where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.setString(3, key);
            ResultSet resultSet = statement.executeQuery();
            if (resultSet.next()) {
                statement.close();
                statement = con.prepareStatement("update USMSessionFields set Value = ? where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
                statement.setString(1, value);
                statement.setInt(2, session.getContextId());
                statement.setInt(3, session.getSessionId());
                statement.setString(4, key);
            } else {
                statement.close();
                statement = con.prepareStatement("insert into USMSessionFields (cid, usmSessionId, Field, Value) values(?,?,?,?)");
                statement.setInt(1, session.getContextId());
                statement.setInt(2, session.getSessionId());
                statement.setString(3, key);
                statement.setString(4, value);
            }
            if (statement.executeUpdate() != 1)
                LOG.warn(session + " Setting field " + key + " in DB did not change any rows");
            con.commit();
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while updating field " + key;
            LOG.error(errorMessage, e);
            try {
                con.rollback();
            } catch (SQLException e1) {
                LOG.error(session + " SQL error: rollback on error failed", e1);
            }
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER3, errorMessage, e);
        } finally {
            if (con != null) {
                Toolkit.close(statement);
                try {
                    con.setAutoCommit(true);
                } catch (SQLException e) {
                    LOG.error(session + " SQL error: setAutocommit(false) failed", e);
                }
                Toolkit.close(con);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getFields(com.openexchange.usm.api.session.Session)
     */
    @Override
    public Map<String, String> getFields(Session session) throws USMStorageException, StorageAccessException {
        Map<String, String> map = null;
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getReadOnly(session.getContextId());
            map = new HashMap<String, String>();

            statement = con.prepareStatement("select Field, Value from USMSessionFields where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            ResultSet result = statement.executeQuery();

            while (result.next()) {
                String k = result.getString(1);
                String v = result.getString(2);
                map.put(k, v);
            }
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while retrieving persistent data from DB";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }

        return map;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#removeField(com.openexchange.usm.api.session.Session,
     * java.lang.String)
     */
    @Override
    public void removeField(Session session, String key) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getWritable(session.getContextId());
            statement = con.prepareStatement("delete from USMSessionFields where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and Field = ?");
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.setString(3, key);
            if (statement.executeUpdate() != 1)
                LOG.warn(session + " Deleting field " + key + " in DB did not change any rows");
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while updating field " + key;
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER2, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#removeAllFieldsForSession(com.openexchange.usm.api.session.
     * Session)
     */
    @Override
    public void removeAllFieldsForSession(Session session) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getWritable(session.getContextId());
            statement = con.prepareStatement("delete from USMSessionFields where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.executeUpdate();
            statement.close();
            statement = con.prepareStatement("delete from usmIdMapping where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.executeUpdate();
            statement.close();
            statement = con.prepareStatement("delete from USMSession where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_IDENTIFIER);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());

            if (statement.executeUpdate() != 1)
                LOG.warn(session + " Deleting session in DB did not change any rows");

        } catch (SQLException e) {
            String errorMessage = session + " SQL error while deleting fields ";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER4, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#sessionExists(com.openexchange.usm.api.session.Session)
     */
    @Override
    public boolean sessionExists(Session session) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = dbAccess.getReadOnly(session.getContextId());
        PreparedStatement statement = null;
        boolean sessionFound = false;
        try {
            statement = con.prepareStatement("select usmSessionId from USMSession where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_LONG_IDENTIFIER);

            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getUserIdentifier());
            statement.setString(3, session.getDevice());
            statement.setString(4, session.getProtocol());
            ResultSet result = statement.executeQuery();
            sessionFound = result.next();
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while retrieving session from DB";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return sessionFound;
    }

    /*
     * (non-Javadoc)
     * @see
     * com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#sessionExistsForDevice(com.openexchange.usm.api.session.Session
     * )
     */
    @Override
    public boolean sessionExistsForDevice(Session session) throws USMStorageException, StorageAccessException {
        int contextId = session.getContextId();
        EncapsulatedConnection con = dbAccess.getReadOnly(session.getContextId());
        PreparedStatement statement = null;
        boolean sessionFound = false;
        try {
            statement = con.prepareStatement("select usmSessionId from USMSession where " + PersistentSessionDataSQLStatements.DEVICE_IN_CONTEXT);
            statement.setInt(1, contextId);
            statement.setString(2, session.getDevice());
            statement.setString(3, session.getProtocol());
            ResultSet result = statement.executeQuery();
            sessionFound = result.next();
        } catch (SQLException e) {
            String errorMessage = " SQL error while retrieving session from DB";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_2, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }

        return sessionFound;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#insertSession(com.openexchange.usm.api.session.Session)
     */
    @Override
    public int insertSession(Session session) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        int sessionId;

        try {
            con = dbAccess.getWritable(session.getContextId());

            sessionId = dbAccess.getNextIdFromSequence(session.getContextId(), PersistentSessionDataSQLStatements.SEQUENCE_TABLE_NAME);

            String insert = dbAccess.supportsInsertIgnoreStatement() ? "insert ignore into USMSession (cid, id, device, protocol, usmSessionId) values(?,?,?,?,?)" : "insert into USMSession (cid, id, device, protocol, usmSessionId) values(?,?,?,?,?)";
            statement = con.prepareStatement(insert);

            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getUserIdentifier());
            statement.setString(3, session.getDevice());
            statement.setString(4, session.getProtocol());
            statement.setInt(5, sessionId);

            if (statement.executeUpdate() != 1) {
                throw new SQLException(
                    session + " Inserting session for user " + session.getUserIdentifier() + " in DB did not change any rows");
            }
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while retrieving persistent data from DB";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return sessionId;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getSessionID(com.openexchange.usm.api.session.Session)
     */
    @Override
    public Integer getSessionID(Session session) throws USMStorageException, StorageAccessException {
        Integer sessionID = null;
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getReadOnly(session.getContextId());
            statement = con.prepareStatement("select usmSessionId from USMSession where " + PersistentSessionDataSQLStatements.UNIQUE_SESSION_LONG_IDENTIFIER);

            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getUserIdentifier());
            statement.setString(3, session.getDevice());
            statement.setString(4, session.getProtocol());

            ResultSet result = statement.executeQuery();
            if (result.next()) {
                sessionID = result.getInt(1);
            }
        } catch (SQLException e) {
            String errorMessage = session + " SQL error while retrieving usmSessionId from DB";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return sessionID;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#purgeSessions(int, java.util.Set)
     */
    @Override
    public int purgeSessions(int contextID, Set<ContextSessionID> sessionIDs) throws StorageAccessException {
        if (sessionIDs.isEmpty()) {
            return 0;
        }
        int count = 0;
        EncapsulatedConnection con = null;
        PreparedStatement[] stmts = new PreparedStatement[USM_TABLES.length];
        try {
            con = dbAccess.getWritable(contextID);
            for (int i = 0; i < USM_TABLES.length; i++) {
                stmts[i] = con.prepareStatement("DELETE FROM " + USM_TABLES[i] + " WHERE cid = ? AND usmSessionId = ?");
            }
            for (ContextSessionID id : sessionIDs) {
                LOG.debug("Removing USM session data for context " + id.getContextID() + ", usmSessionId=" + id.getSessionID());
                // Directly operate on the DB, we can not create a Session object since the user name is missing (and it would also produce
                // way too much overhead)
                for (PreparedStatement s : stmts) {
                    s.setInt(1, id.getContextID());
                    s.setInt(2, id.getSessionID());
                    s.executeUpdate();
                }
                count++;
            }
        } catch (SQLException e) {
            LOG.error("SQL Exception while removing old USM sessions in same schema as context " + contextID, e);
            return count;
        } finally {
            for (PreparedStatement s : stmts) {
                Toolkit.close(s);
            }
            Toolkit.close(con);
        }
        return count;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getSessionsToRemove(int, long, java.util.Set)
     */
    @Override
    public Set<ContextSessionID> getSessionsToRemove(int contextID, long ttl, Set<Integer> validContextIDs) throws StorageAccessException {
        long now = System.currentTimeMillis();
        long interval = ttl;
        long normalLimit = now - interval;
        long specialLimit = normalLimit - interval; // For those USM sessions that have the new field not set, use the newest synckey and
                                                    // the special limit
        EncapsulatedConnection con = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            con = dbAccess.getReadOnly(contextID);
            // Step 0: Check if table USMSession exists
            if (!tableExists("USMSession", con.getMetaData())) {
                LOG.debug("Table USMSession does not exist - no USM session data must be removed for context " + contextID);
                return Collections.emptySet();
            }
            // Step 1: Get All USM sessions in that context, add all sessions not in active contexts to deletion list
            Set<ContextSessionID> sessionIDsToRemove = new HashSet<ContextSessionID>();
            stmt = con.prepareStatement("SELECT cid, usmSessionId FROM USMSession");
            rs = stmt.executeQuery();
            Set<ContextSessionID> oldSessionIDs = new HashSet<ContextSessionID>();
            while (rs.next()) {
                int contextId = rs.getInt(1);
                int sessionId = rs.getInt(2);
                ContextSessionID contextSessionID = new ContextSessionID(contextId, sessionId);
                if (validContextIDs.contains(contextId)) {
                    oldSessionIDs.add(contextSessionID);
                } else {
                    sessionIDsToRemove.add(contextSessionID);
                }
            }
            rs.close();
            rs = null;
            stmt.close();
            stmt = null;
            // Step 2: Get last access field value for all sessions, remove session from "old" session list if field is present, add to
            // deletion list if value is too old
            stmt = con.prepareStatement("SELECT cid, usmSessionId, value FROM USMSessionFields WHERE field = ?");
            stmt.setString(1, FIELD_PREFIX + LAST_ACCESS_FIELD_NAME);
            rs = stmt.executeQuery();
            while (rs.next()) {
                ContextSessionID id = new ContextSessionID(rs.getInt(1), rs.getInt(2));
                long lastAccess = Long.parseLong(rs.getString(3));
                oldSessionIDs.remove(id);
                if (lastAccess < normalLimit) {
                    sessionIDsToRemove.add(id);
                }
            }
            rs.close();
            rs = null;
            stmt.close();
            stmt = null;
            // Step 3: For all old sessions, determine newest sync key, add to deletion list if it is too old
            stmt = con.prepareStatement("SELECT MAX(SyncKey) FROM USMDataStorage WHERE cid = ? AND usmSessionId = ?");
            for (ContextSessionID sessionID : oldSessionIDs) {
                stmt.setInt(1, sessionID.getContextID());
                stmt.setInt(2, sessionID.getSessionID());
                rs = stmt.executeQuery();
                if (rs.next()) {
                    long syncKey = rs.getLong(1);
                    if (syncKey < specialLimit)
                        sessionIDsToRemove.add(sessionID);
                } else {
                    // No SyncState at all -> can be safely removed
                    sessionIDsToRemove.add(sessionID);
                }
                rs.close();
                rs = null;
            }
            stmt.close();
            stmt = null;
            return sessionIDsToRemove;
        } catch (SQLException e) {
            LOG.error("SQL Exception while determining USM sessions to remove for context " + contextID, e);
            return Collections.emptySet();
        } finally {
            Toolkit.close(rs);
            Toolkit.close(stmt);
            Toolkit.close(con);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getFieldsFromContext(int, int, int, java.lang.String,
     * java.lang.String, java.lang.String)
     */
    @Override
    public List<String> getFieldsFromSchema(int contextForDB, int cid, int id, String protocol, String device, String field, Set<Integer> validContextIDs) throws StorageAccessException, USMStorageException {
        List<String> fields = null;
        PreparedStatement statement = null;
        ResultSet result = null;
        EncapsulatedConnection con = null;
        Map<USMSessionID, SessionInformation> sessionIDs = null;
        try {
            con = dbAccess.getReadOnly(contextForDB);

            try {
                sessionIDs = retrieveUSMSessionsFromDB(cid, id, protocol, device, con, validContextIDs);
                if (sessionIDs.isEmpty()) {
                    fields = new ArrayList<String>(1);
                    fields.add(contextForDB + ": >> No matching sessions <<");
                    return fields;
                }
                if (cid == 0) {
                    statement = con.prepareStatement("SELECT cid, usmSessionId, field, value FROM USMSessionFields WHERE field like 'F%'");
                } else {
                    statement = con.prepareStatement("SELECT cid, usmSessionId, field, value FROM USMSessionFields WHERE cid = ? AND field like 'F%'");
                    statement.setInt(1, cid);
                }
                result = statement.executeQuery();
                field = isParameterSet(field) ? (FIELD_PREFIX + field) : null;
                while (result.next()) {
                    int contextid = result.getInt(1);
                    int usmSessionId = result.getInt(2);
                    SessionInformation d = sessionIDs.get(new USMSessionID(contextid, usmSessionId));
                    if (d != null) {
                        String f = result.getString(3);
                        if (field == null || field.equals(f)) {
                            d.addFieldData("    " + f.substring(FIELD_PREFIX.length()) + " = " + result.getString(4));
                        }
                    }
                }
            } finally {
                Toolkit.close(result);
                Toolkit.close(statement);
                Toolkit.close(con);
            }
            List<SessionInformation> sessionList = new ArrayList<SessionInformation>(sessionIDs.values());
            Collections.sort(sessionList);
            List<String> returnData = new ArrayList<String>();
            for (SessionInformation si : sessionList)
                si.appendFieldDataToList(returnData);
            return returnData;
        } catch (SQLException e) {
            String errorMessage = cid + ": Error while retrieving session fields.";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#updateFieldsInContext(int, int, int, java.lang.String,
     * java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public Map<USMSessionID, SessionInformation> updateFieldsInSchema(int contextForDB, int cid, int id, String protocol, String device, String field, String value, Set<Integer> validContextIDs) throws USMStorageException, StorageAccessException {
        String dbField = FIELD_PREFIX + field;
        PreparedStatement statement = null;
        EncapsulatedConnection con = null;
        PreparedStatement insertStatement = null;
        Map<USMSessionID, SessionInformation> sessionIDs = new HashMap<USMSessionID, SessionInformation>();
        try {
            con = dbAccess.getWritable(contextForDB);
            ResultSet result = null;
            try {
                sessionIDs = retrieveUSMSessionsFromDB(cid, id, protocol, device, con, validContextIDs);
                statement = con.prepareStatement("DELETE FROM USMSessionFields WHERE cid = ? AND usmSessionId = ? AND field = ?");
                if (isParameterSet(value))
                    insertStatement = con.prepareStatement("INSERT INTO USMSessionFields (cid, usmSessionId, Field, Value) VALUES(?,?,?,?)");
                for (USMSessionID usmId : sessionIDs.keySet()) {
                    statement.setInt(1, usmId.getCID());
                    statement.setInt(2, usmId.getUSMSessionID());
                    statement.setString(3, dbField);
                    statement.executeUpdate();
                    if (insertStatement != null) {
                        insertStatement.setInt(1, usmId.getCID());
                        insertStatement.setInt(2, usmId.getUSMSessionID());
                        insertStatement.setString(3, dbField);
                        insertStatement.setString(4, value);
                        insertStatement.executeUpdate();
                    }
                }
            } finally {
                if (con != null) {
                    Toolkit.close(result);
                    Toolkit.close(statement);
                    Toolkit.close(insertStatement);
                    Toolkit.close(con);
                }
            }
        } catch (SQLException e) {
            String errorMessage = cid + ": Error while updating session fields.";
            throw new USMStorageException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_ERROR_NUMBER1, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
            Toolkit.close(insertStatement);
        }

        return sessionIDs;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getAllContextIDs()
     */
    @Override
    public List<Integer> getAllContextIDs() throws StorageAccessException {
        return dbAccess.getAllContextIds();
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSessionDataStorage#getContextsInSameSchema(int)
     */
    @Override
    public int[] getContextsInSameSchema(int contextID) throws StorageAccessException {
        return dbAccess.getContextsInSameSchema(contextID);
    }

    /**
     * Retrieves USM sessions from the DB for the given parameters
     * 
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param con
     * @param validContextIDs
     * @return
     * @throws SQLException
     */
    private static Map<USMSessionID, SessionInformation> retrieveUSMSessionsFromDB(int cid, int id, String protocol, String device, EncapsulatedConnection con, Set<Integer> validContextIDs) throws SQLException {
        PreparedStatement statement = null;
        ResultSet result = null;
        try {
            StringBuilder selectSessionQuery = new StringBuilder("SELECT cid, id, protocol, device, usmSessionId FROM USMSession");
            boolean first = true;
            if (cid != 0)
                first = appendConditionToQuery(selectSessionQuery, "cid", first);
            if (id != 0)
                first = appendConditionToQuery(selectSessionQuery, "id", first);
            if (isParameterSet(device))
                first = appendConditionToQuery(selectSessionQuery, "device", first);
            if (isParameterSet(protocol))
                first = appendConditionToQuery(selectSessionQuery, "protocol", first);
            statement = con.prepareStatement(selectSessionQuery.toString());
            int paramCount = 0;
            if (cid != 0)
                statement.setInt(++paramCount, cid);
            if (id != 0)
                statement.setInt(++paramCount, id);
            if (isParameterSet(device))
                statement.setString(++paramCount, device);
            if (isParameterSet(protocol))
                statement.setString(++paramCount, protocol);
            result = statement.executeQuery();
            Map<USMSessionID, SessionInformation> sessionIDs = new HashMap<USMSessionID, SessionInformation>();
            while (result.next()) {
                int contextid = result.getInt(1);
                if (validContextIDs == null || validContextIDs.contains(contextid)) {
                    int usmSessionId = result.getInt(5);
                    sessionIDs.put(
                        new USMSessionID(contextid, usmSessionId),
                        new SessionInformation(contextid, result.getInt(2), result.getString(3), result.getString(4), usmSessionId));
                }
            }
            return sessionIDs;
        } finally {
            Toolkit.close(result);
            Toolkit.close(statement);
        }
    }

    /**
     * Applies either a WHERE or an AND condition to the given query
     * 
     * @param builder
     * @param dbColumn
     * @param first
     * @return
     */
    private static boolean appendConditionToQuery(StringBuilder builder, String dbColumn, boolean first) {
        if (first)
            builder.append(" WHERE ");
        else
            builder.append(" AND ");
        builder.append(dbColumn).append(" = ?");
        return false;
    }

    /**
     * Verifies whether the given parameter is set.
     * 
     * @param param
     * @return
     */
    private static boolean isParameterSet(String param) {
        return param != null && param.length() > 0 && !"String".equals(param);
    }

    /**
     * Check a table's existence
     * 
     * @param tableName The table name to check
     * @param dbmd The database's meta data
     * @return <code>true</code> if table exists; otherwise <code>false</code>
     * @throws SQLException If a SQL error occurs
     */
    // this code has been copied from USMBaseUpdateTask.tableExists(String, DatabaseMetaData) to avoid unwanted dependencies
    private static boolean tableExists(final String tableName, final DatabaseMetaData dbmd) throws SQLException {
        ResultSet resultSet = null;
        try {
            resultSet = dbmd.getTables(null, null, tableName, new String[] { "TABLE" });
            return resultSet.next();
        } finally {
            Toolkit.close(resultSet);
        }
    }
}
