/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.datatypes.contacts;

import java.io.Serializable;

/**
 * Complex DataType for storing Images for Contacts. By default, only
 * last_modified will be read from server (at least for now until the
 * OX server can provide the picture data without requiring extra calls)
 * and stored in the cache for synchronization.<br>
 * If a client wants to set the picture for a Contact, it sets an
 * Image object with lastModified, contentType and data set.
 * The ContactContentTypeTransferHandler takes care of the extra steps
 * required to update the picture on the OX server.<br>
 * 
 * @author afe
 *
 */
public class Image implements Serializable {

    /**
	 * 
	 */
    private static final long serialVersionUID = 1L;

    private final long _lastModified;

    private final transient String _url;

    private final transient String _contentType;

    private final transient byte[] _data;
    
    private final transient String _tempid;

    public Image() {
        _lastModified = 0L;
        _url = null;
        _contentType = null;
        _data = null;
        _tempid = null;
    }

    Image(long lastModified, String url, String contentType) {
        _lastModified = lastModified;
        _url = url;
        _contentType = contentType;
        _data = null;
        _tempid = null;
    }

    public Image(long lastModified, String contentType, byte[] data) {
        _lastModified = lastModified;
        _url = null;
        _contentType = contentType;
        _data = data;
        _tempid = null;
    }

    public Image(String tempid, String contentType) {
        _lastModified = System.currentTimeMillis();
        _url = null;
        _contentType = contentType;
        _data = null;
        _tempid = tempid;
    }

    /**
     * @return timestamp of last modification
     */
    public long getLastModified() {
        return _lastModified;
    }

    /**
	 * @return URL that can be used to retrieve image, if an image is present (normally only used by the ContactContentTypeTransferHandler), null otherwise
     */
    public String getUrl() {
        return _url;
    }

    /**
     * @return ContentType of the picture data, null if no picture is set
     */
    public String getContentType() {
        return _contentType;
    }

    /**
	 * For now this method will always return null for Images on DataObjects that are returned by
	 * the USM sync process. To retrieve the picture data, an explicit call to getPictureData()
	 * of the ContactContentTypeTransferHandler is required (since retrieval of a picture requires
	 * an extra call to the OX server).
     * 
	 * @return binary content of the picture of set (in the format specified by getContentType()), null if no picture is set or if Image object was generated by the USM sync process
     */
    public byte[] getData() {
        return _data;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (_lastModified ^ (_lastModified >>> 32));
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof Image))
            return false;
        Image other = (Image) obj;
        if (_lastModified != other._lastModified)
            return false;
        return true;
    }

    public String getTempId() {
        return _tempid;
    }
}
