/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.session.storage;

import java.util.List;
import java.util.Set;
import com.openexchange.usm.api.database.CachedDBEntry;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;


/**
 * {@link PersistentSyncStateStorage}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface PersistentSyncStateStorage {
    /**
     * Stores the given {@link DataObjectSet} to the persistent storage.
     * 
     * @param session
     * @param objectID
     * @param element
     * @param oldTimestamp
     * @param newTimestamp
     * @param timestampToKeep
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public long store(Session session, String objectID, String element, long oldTimestamp, long newTimestamp, long timestampToKeep) throws StorageAccessException, USMStorageException;
    
    /**
     * Returns the cached data for the given objectID
     * @param objectID
     * @param maxSyncStatesInDB
     * @return {@link CachedData}
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public List<CachedDBEntry> get(Session session, String objectID, int maxSyncStatesInDB) throws StorageAccessException, USMStorageException;
    
    /**
     * Removes all DB entries for the given object IDs.
     * 
     * @param session
     * @param objectIDs
     * @throws USMStorageException
     */
    public void remove(Session session, String... objectIDs) throws USMStorageException;
    
    /**
     * Returns a set of all object-ids for which data is stored and which are not in the provided set of object-ids.
     * 
     * @param session
     * @param excludedObjectIDs
     * @return a list with all objectIDs that have persistent storage but were not provided in the parameter set
     * @throws USMStorageException
     */
    public List<String> getRemainingObjectIDs(Session session, Set<String> excludedObjectIDs) throws USMStorageException;
    
    /**
     * Re-maps the already stored objectIDs.
     * 
     * @param session
     * @param oldObjectID
     * @param newObjectID
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public void remapStates(Session session, String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException;
    
    /**
     * Removes all saved states for this session.
     * 
     * @param session
     * @throws USMStorageException
     */
    public void purgeSession(Session session) throws USMStorageException;
}
