/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.tasks;

import org.apache.commons.logging.Log;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.SessionReference;
import com.openexchange.usm.session.storage.SessionStorage;

/**
 * Task that is executed regularly and removes all USM sessions from memory that have not been accessed for the configured limit.
 * {@link USMSessionCacheCleanupTask}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMSessionCacheCleanupTask implements Runnable {

    private final SessionManagerImpl _sessionManager;

    public USMSessionCacheCleanupTask(SessionManagerImpl manager) {
        _sessionManager = manager;
    }

    /**
     * Removes all Sessions from memory that have not been accessed within the configured limit for Sessions. Removes all SyncStates that
     * have not been accessed within the configured limit for SyncStates.
     */
    @Override
    public void run() {
        Log log = _sessionManager.getJournal();
        try {
            long now = System.currentTimeMillis();
            int sessionCacheTimeout = _sessionManager.getSessionCacheTimeout();
            int syncStateCacheTimeout = _sessionManager.getSyncStateCacheTimeout();
            long sessionLimit = now - sessionCacheTimeout * 1000L;
            long syncStateLimit = now - syncStateCacheTimeout * 1000L;
            if (log.isDebugEnabled())
                log.debug("Removing USM Sessions/SyncStates from memory cache that have not been accessed for " + sessionCacheTimeout + '/' + syncStateCacheTimeout + " seconds");
            SessionStorage storage = _sessionManager.getSessionStorage();
            int removedSessions = 0;
            for (SessionReference ref : storage.getSessionReferences()) {
                if (ref.getLastAccess() < sessionLimit) {
                    SessionImpl session = ref.get();
                    if (session != null) {
                        storage.removeSession(session);
                        removedSessions++;
                    }
                }
            }
            int removedSyncStates = _sessionManager.getSyncStateCacheProvider().removeUnusedSyncStates(syncStateLimit);
            int jvmRemovals = storage.removeClearedReferences();
            _sessionManager.updateCacheRemovals(removedSessions, jvmRemovals, removedSyncStates);
        } catch (Exception e) {
            log.error("Exception occurred while cleaning up memory caches", e);
        }
    }
}
