/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.mimemail;

import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimePart;
import javax.mail.internet.MimeUtility;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.mail.mime.ContentDisposition;
import com.openexchange.mail.mime.ContentType;
import com.openexchange.mail.mime.QuotedInternetAddress;
import com.openexchange.usm.connector.commands.CommandConstants;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;

/**
 * {@link AbstractMimeMailBuilder}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public abstract class AbstractMimeMailBuilder {

    protected static final char SEMICOLON = ';';

    protected static final String CONTENT_TYPE_UPPER = "Content-Type";

    protected static final String CONTENT_TYPE = "content-type";

    protected static final String CONTENT_TRANSFER_ENCODING_UPPER = "Content-Transfer-Encoding";

    protected static final String CONTENT_TRANSFER_ENCODING = "content-transfer-encoding";

    protected static final String BODY = "body";

    protected static final String SMIME_BODY_DATA = "smime_body_data";

    protected static final String HEADERS = "headers";

    protected static final String MULTIPART = "multipart/";

    protected static final String CONTENTTYPE_TEXT_PREFIX = "text/";

    protected static final String CONTENTTYPE_RFC822 = "message/rfc822";

    private static final String _DATE = "date";

    private static final String _LINE_SEPARATOR = "\r\n";

    private static final String _UTC = "utc";

    private static final String _PERSONAL = "personal";

    private static final String _ADDRESS = "address";

    private static final String _CONTENT_DISPOSITION = "content-disposition";

    private static final String _RECEIVED = "received";

    private static final String _RETURN_PATH = "return-path";

    private static final String _X_MAILER = "x-mailer";

    private static final String _X_SIEVE = "x-sieve";

    private static final String _MIME_VERSION = "mime-version";

    private static final String _CHARSET = "charset";

    private static final String _READ_DATE = "read-date";

    private static final String _PARAMS = "params";

    private static final String _TYPE = "type";

    private static final String _X_ORIGINAL_HEADERS = "x-original-headers";

    private static final Map<String, String> _SPECIAL_HEADER_NAMES = new HashMap<String, String>();

    private static final Map<String, String> _KNOWN_ADDRESS_HEADER_NAMES = new HashMap<String, String>();

    private static final Map<String, String> _KNOWN_HEADER_NAMES = new HashMap<String, String>();

    private static final Map<String, Message.RecipientType> _RECIPIENT_ADDRESS_HEADERS = new HashMap<String, Message.RecipientType>();

    protected static final String _DEFAULT_CONTENT_TYPE = CONTENTTYPE_TEXT_PREFIX + "plain; " + _CHARSET + '=' + CommandConstants.UTF_8;
    
    private static final String[] _RECEIVED_VALUE_DELIMITERS = { " from ", " by ", " via ", " for "};

    static {
        _SPECIAL_HEADER_NAMES.put(CONTENT_TYPE, CONTENT_TYPE_UPPER);
        _SPECIAL_HEADER_NAMES.put(_CONTENT_DISPOSITION, "Content-Disposition");
        _SPECIAL_HEADER_NAMES.put(CONTENT_TRANSFER_ENCODING, CONTENT_TRANSFER_ENCODING_UPPER);
        _SPECIAL_HEADER_NAMES.put(_RECEIVED, "Received");
        _SPECIAL_HEADER_NAMES.put(_RETURN_PATH, "Return-Path");
        _SPECIAL_HEADER_NAMES.put(_X_SIEVE, "X-Sieve");
        _SPECIAL_HEADER_NAMES.put(_MIME_VERSION, "MIME-Version");
        _SPECIAL_HEADER_NAMES.put(_X_MAILER, "X-Mailer");

        _KNOWN_ADDRESS_HEADER_NAMES.put("from", "From");
        _KNOWN_ADDRESS_HEADER_NAMES.put("reply-to", "Reply-To");
        _KNOWN_ADDRESS_HEADER_NAMES.put("sender", "Sender");
        _KNOWN_ADDRESS_HEADER_NAMES.put("errors-to", "Errors-To");
        _KNOWN_ADDRESS_HEADER_NAMES.put("resent-bcc", "Resent-Bcc");
        _KNOWN_ADDRESS_HEADER_NAMES.put("resent-cc", "Resent-Cc");
        _KNOWN_ADDRESS_HEADER_NAMES.put("resent-from", "Resent-From");
        _KNOWN_ADDRESS_HEADER_NAMES.put("resent-to", "Resent-To");
        _KNOWN_ADDRESS_HEADER_NAMES.put("resent-sender", "Resent-Sender");
        _KNOWN_ADDRESS_HEADER_NAMES.put("disposition-notification-to", "Disposition-Notification-To");
        
        _KNOWN_HEADER_NAMES.put("message-id", "Message-ID");
        _KNOWN_HEADER_NAMES.put("in-reply-to", "In-Reply-To");
        _KNOWN_HEADER_NAMES.put("references", "References");
        _KNOWN_HEADER_NAMES.put("subject", "Subject"); 
        _KNOWN_HEADER_NAMES.put("thread-index", "Thread-Index");
        _KNOWN_HEADER_NAMES.put("thread-topic", "Thread-Topic");
        _KNOWN_HEADER_NAMES.put("keywords", "Keywords");
        _KNOWN_HEADER_NAMES.put("importance", "Importance");
        _KNOWN_HEADER_NAMES.put("priority", "Priority");
        _KNOWN_HEADER_NAMES.put("date", "Date");
        _KNOWN_HEADER_NAMES.put("x-original-headers", "X-Original-Headers");

        _RECIPIENT_ADDRESS_HEADERS.put("to", Message.RecipientType.TO);
        _RECIPIENT_ADDRESS_HEADERS.put("cc", Message.RecipientType.CC);
        _RECIPIENT_ADDRESS_HEADERS.put("bcc", Message.RecipientType.BCC);
    }

    protected static String formatReceived(String rec) {
        if (rec == null || rec.length() < 2)
            return rec;
        StringBuilder sb = new StringBuilder(rec.length() + 10);
        int pos = 0;
        int nextPos = getNextReceivedDelimiterIndex(rec, pos + 1);
        while(nextPos > 0) {
            if(sb.length() > 0)
                sb.append(_LINE_SEPARATOR).append(' ');
            sb.append(rec, pos, nextPos);
            pos = nextPos;
            nextPos = getNextReceivedDelimiterIndex(rec, pos + 1);
        }
        if(sb.length() > 0)
            sb.append(_LINE_SEPARATOR).append(' ');
        sb.append(rec, pos, rec.length());
        return sb.toString();
    }
    
    private static int getNextReceivedDelimiterIndex(String rec, int startPos) {
        int minIndex = -1;
        for(String key : _RECEIVED_VALUE_DELIMITERS) {
            int index = rec.indexOf(key, startPos);
            if(index > 0 && (minIndex < 0 || minIndex > index))
                minIndex = index;
        }
        return minIndex;
    }

    protected static void setHeaderConditional(Part part, String header, String value) throws MessagingException {
    	if (isValueSet(value)) {
    		try {
    			part.setHeader(header, MimeUtility.encodeText(value, CommandConstants.UTF_8, null));
    		} catch (UnsupportedEncodingException e) {
    			part.setHeader(header, value);
    		}
    	} else {
    	    part.removeHeader(header);
    	}
    }
    
    protected static void appendHeaderConditional(Part part, String header, String value) throws MessagingException {
        if (isValueSet(value)) {
            try {
                part.addHeader(header, MimeUtility.encodeText(value, CommandConstants.UTF_8, null));
            } catch (UnsupportedEncodingException e) {
                part.addHeader(header, value);
            }
        } else {
            part.removeHeader(header);
        }
    }

    protected static boolean isValueSet(String value) {
        return value != null && !"null".equals(value);
    }

    protected static String getAddress(JSONObject headers, String key, String defaultVal) throws JSONException, USMJSONAPIException {
        if (headers == null || !headers.has(key))
            return defaultVal;
        Object value = headers.get(key);
        if (value == null)
            return defaultVal;
        if (value instanceof String)
            return value.toString();
        try {
            if (value instanceof JSONObject)
                return extractAddress((JSONObject) value);
            if (value instanceof JSONArray) {
                JSONArray array = (JSONArray) value;
                int length = array.length();
                if (0 >= length)
                    return defaultVal;
                StringBuilder address = new StringBuilder(128);
                for (int j = 0; j < length; j++) {
                    if (j > 0)
                        address.append(',');
                    address.append(extractAddress(array.getJSONObject(j)));
                }
                return address.toString();
            }
        } catch (JSONException e) {
            return defaultVal;
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_2,
                ResponseStatusCode.INTERNAL_ERROR,
                "unsupported encoding of text content: " + e.getMessage(),
                e);
        } catch (Exception e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_ADDRESS_EXCEPTION,
                ResponseStatusCode.INTERNAL_ERROR,
                "illegal mail address: " + e.getMessage(),
                e);
        }
        return defaultVal;
    }

    private static QuotedInternetAddress extractInternetAddress(JSONObject address) throws JSONException, UnsupportedEncodingException, AddressException {
        String addressPart = address.getString(_ADDRESS);
        String personalPart = address.optString(_PERSONAL, null);
        return new QuotedInternetAddress(addressPart, personalPart, CommandConstants.UTF_8);
    }

    private static String extractAddress(JSONObject address) throws JSONException, UnsupportedEncodingException {
        try {
            return extractInternetAddress(address).toString();
        } catch (AddressException e) {
            String addressPart = address.getString(_ADDRESS);
            String personalPart = address.optString(_PERSONAL, null);
            // In case of invalid address data, use simple composition of the raw (illegal) data
            if (personalPart == null)
                return addressPart;
            return personalPart + " <" + addressPart + ">";
        }
    }

    protected static void appendRecipientHeader(Part part, JSONObject headers, javax.mail.Message.RecipientType recipientType, String key) throws USMJSONAPIException {
        if (headers == null || !headers.has(key))
            return;
        try {
            if (!(part instanceof MimeMessage)) {
                setHeaderConditional(part, key, getAddress(headers, key, null));
                return;
            }
            MimeMessage message = (MimeMessage) part;
            Object value = headers.get(key);
            if (value instanceof String) {
                message.setRecipients(recipientType, (String) value);
            } else if (value instanceof JSONObject) {
                message.addRecipient(recipientType, extractInternetAddress((JSONObject) value));
            } else if (value instanceof JSONArray) {
                JSONArray array = (JSONArray) value;
                int length = array.length();
                for (int i = 0; i < length; i++)
                    message.addRecipient(recipientType, extractInternetAddress(array.getJSONObject(i)));
            } else if (value == null || value == JSONObject.NULL) {
               message.removeHeader(key); 
            } else {
                throw new IllegalArgumentException("header of unknown type" + value);
            }
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_4,
                ResponseStatusCode.INTERNAL_ERROR,
                "unsupported encoding of text content: " + e.getMessage(),
                e);
        } catch (Exception e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_RECIPIENT_EXCEPTION,
                ResponseStatusCode.INTERNAL_ERROR,
                "illegal mail address: " + e.getMessage(),
                e);
        }
    }

    @SuppressWarnings("unchecked")
    protected static Enumeration<Header> parseXOriginalHeaders(String xOriginalHeaders) throws USMJSONAPIException {
    	ByteArrayInputStream is = new ByteArrayInputStream(Toolkit.decodeBase64(xOriginalHeaders));
    	try {
    		InternetHeaders ih = new InternetHeaders(is);
    		return ih.getAllHeaders();
    	} catch (MessagingException e) {
    		throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR_PARSE_ORIGINAL_HEADERS,
    				ResponseStatusCode.INTERNAL_ERROR, "MessagingException by parsing x_original headers: " + e.getMessage());
    	}
    }

    protected static String extractEncodingCharset(String contentTypeOfPart) {
        int posCharset = contentTypeOfPart.indexOf(_CHARSET);
        if (posCharset < 0)
            return "UTF-8";
        int charSetBeginPos = posCharset + _CHARSET.length() + 1;
        int charSetEndPos = contentTypeOfPart.indexOf(';', charSetBeginPos);
        return ((charSetEndPos >= 0) ? contentTypeOfPart.substring(charSetBeginPos, charSetEndPos) : contentTypeOfPart.substring(charSetBeginPos)).trim();
    }

    protected static String extractPlainContentType(String contentTypeOfPart) {
        int len = contentTypeOfPart.length();
        for (int i = 0; i < len; i++) {
            char c = contentTypeOfPart.charAt(i);
            if (c == SEMICOLON || Character.isWhitespace(c))
                return contentTypeOfPart.substring(0, i);
        }
        return contentTypeOfPart;
    }

    protected final SimpleDateFormat _dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z", Locale.ENGLISH);

    protected final USMMimeMessage _message = new USMMimeMessage(javax.mail.Session.getInstance(new Properties()));

    protected String extractDateFromJSONObject(JSONObject dateObject) {
    	try {
    		if (dateObject.has(_DATE))
    			return dateObject.getString(_DATE);
    		if (dateObject.has(_UTC))
    			return _dateFormat.format(new Date(dateObject.getLong(_UTC)));
    	} catch (JSONException ignored) {
    	    // fall through
    	}
    	return null;
    }

    protected String getValue(JSONObject headers, String key, int i, String defaultVal) throws JSONException {
        if (headers == null || !headers.has(key))
            return defaultVal;
        Object value = headers.get(key);
        if (value == null)
            return defaultVal;
        if (value instanceof String)
            return (String) value;
        if (value instanceof JSONArray) {
            JSONArray array = (JSONArray) value;
            if (i >= array.length())
                return defaultVal;
            try {
                return array.getString(i);
            } catch (JSONException e) {
                return defaultVal;
            }
        }
        if (value instanceof JSONObject) {
            String v = extractDateFromJSONObject((JSONObject) value);
            return (v == null) ? value.toString() : v;
        }
        return defaultVal;
    }

    protected String getValue(MimePart part, String key, int i, String defaultVal) {
        if (part == null)
            return defaultVal;
        try {
            String[] values = part.getHeader(key);
            if (values == null || values.length == 0)
                return defaultVal;
            if (i >= values.length || i<0)
                return defaultVal;
            return values[i];
        }
        catch (MessagingException e) {
            return defaultVal;
        }
    }

    protected String buildContentDispositionAsString(Object header) throws JSONException, USMJSONAPIException {
        ContentDisposition contentDisp = new ContentDisposition();
        try {
            if (header instanceof String)
                contentDisp = new ContentDisposition((String) header);
            else if (header instanceof JSONObject) {
                String type = ((JSONObject) header).getString(_TYPE);
                contentDisp.setDisposition(type);
                JSONObject params = (((JSONObject) header).has(_PARAMS)) ? ((JSONObject) header).getJSONObject(_PARAMS) : new JSONObject();
                for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
                    String key = (String) iterator.next();
                    String value = "";
                    if (_READ_DATE.equals(key)) {
                        value = extractDateFromJSONObject(params.getJSONObject(_READ_DATE));
                    } else {
                        value = params.getString(key);
                    }
                    contentDisp.addParameter(key, value);
                }
            }
        } catch (OXException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_DISP_MAIL_EXC,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Mail Exception by add content-disposition header " + e.getMessage());
        }
        return contentDisp.toString();
    }

    protected String buildContentTypeAsString(Object header) throws JSONException, USMJSONAPIException {
        if (header == null)
            return null;
        ContentType contentType = new ContentType();
        try {
            if (header instanceof String) {
                contentType = new ContentType((String) header);
            } else if (header instanceof JSONObject) {
                String type = ((JSONObject) header).getString(_TYPE);
                contentType.setBaseType(type);
                JSONObject params = (((JSONObject) header).has(_PARAMS)) ? ((JSONObject) header).getJSONObject(_PARAMS) : new JSONObject();
                for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
                    String key = (String) iterator.next();
                    String value = "";
                    if (_READ_DATE.equals(key)) {
                        value = extractDateFromJSONObject(params.getJSONObject(_READ_DATE));
                    } else {
                        value = params.getString(key);
                    }
                    contentType.addParameter(key, value);
                }
            }
        } catch (OXException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_TYPE_MAIL_EXC,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Mail Exception by add content-type header " + e.getMessage());
        }
        return contentType.toString();
    }
    
    protected abstract void buildPart(MimePart part, JSONObject json) throws USMJSONAPIException;

    protected void buildMimeMailFromJSONObject(JSONObject data) throws USMJSONAPIException {
    	buildPart(_message, data);
    	try {
    		_message.saveChanges();
    	} catch (MessagingException e1) {
    		throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_SAVE_CHANGE_ERROR,
    				ResponseStatusCode.INTERNAL_ERROR, "error by save change of mime mail: " + e1.getMessage());
    	}
    
    }

    protected MimeMultipart buildMimePartForMultipart(JSONObject data, String subtype) throws MessagingException, JSONException, USMJSONAPIException {
        MimeMultipart multiPart = "mixed".equals(subtype) ? new MixedMimeMultipart() : new MimeMultipart(subtype);
    
        // create and add parts
        JSONArray parts = data.getJSONArray(BODY);
        if (parts != null) {
            int length = parts.length();
            for (int i = 0; i < length; i++) {
                JSONObject jsonpart = parts.getJSONObject(i);
                MimeBodyPart part = new MimeBodyPart();
                buildPart(part, jsonpart);
                multiPart.addBodyPart(part);
            }
        }
        return multiPart;
    }

    protected void appendHeadersToPart(Part part, JSONObject headers, String contentType) throws USMJSONAPIException {
    	try {
    
    		if (contentType != null)
    			part.setHeader(CONTENT_TYPE_UPPER, contentType);
    		if (headers != null && headers.has(_CONTENT_DISPOSITION)) {
    			String contDispAsString = buildContentDispositionAsString(headers.get(_CONTENT_DISPOSITION));
    			part.setDisposition(contDispAsString);
    		}
            if (headers != null && part.getHeader(CONTENT_TRANSFER_ENCODING_UPPER) == null) {
                setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null));
            }
    		//add (or replace with) the original headers
    		if (headers != null && headers.has(_X_ORIGINAL_HEADERS)) {
    			Enumeration<Header> originalHeadersMap = parseXOriginalHeaders(headers.getString(_X_ORIGINAL_HEADERS));
    			while (originalHeadersMap.hasMoreElements()) {
    				Header nextElement = originalHeadersMap.nextElement();
    				String headerName = nextElement.getName();
    				if (_RECEIVED.equalsIgnoreCase(headerName)) {
                        JSONArray receivedArray = headers.optJSONArray(_RECEIVED);
                        if (receivedArray == null)
                            receivedArray = new JSONArray();
                        receivedArray.put(nextElement.getValue().replaceAll(AbstractMimeMailBuilder._LINE_SEPARATOR, ";"));
                        headers.put(_RECEIVED, receivedArray);
                    } else {
    					if (!headers.has(headerName.toLowerCase()))
    					   appendHeaderConditional(part, headerName, nextElement.getValue());
                    }
    			}
    		}
    		//add special headers
    		String returnPath = getValue(headers, _RETURN_PATH, 0, null);
    		setHeaderConditional(part, _SPECIAL_HEADER_NAMES.get(_RETURN_PATH), returnPath);
    		
    		String xSieve = getValue(headers, _X_SIEVE, 0, null);
    		setHeaderConditional(part, _SPECIAL_HEADER_NAMES.get(_X_SIEVE), xSieve);
    
            if (headers != null && headers.has(_RECEIVED)) {
                JSONArray receivedArray = headers.optJSONArray(_RECEIVED);
                setHeaderConditional(part, _SPECIAL_HEADER_NAMES.get(_RECEIVED), AbstractMimeMailBuilder.formatReceived(receivedArray.getString(receivedArray.length() - 1)));
                for (int i = receivedArray.length() - 2  ; i >= 0 ; i--)
                    appendHeaderConditional(part, _SPECIAL_HEADER_NAMES.get(_RECEIVED), AbstractMimeMailBuilder.formatReceived(receivedArray.getString(i)));
            }
    
    		if (headers != null && headers.has(_MIME_VERSION))
    			setHeaderConditional(part, _MIME_VERSION, getValue(headers, _SPECIAL_HEADER_NAMES.get(_MIME_VERSION), 0, ""));
    
    		if (headers != null && headers.has(_X_MAILER))
    			setHeaderConditional(part, _X_MAILER, getValue(headers, _SPECIAL_HEADER_NAMES.get(_X_MAILER), 0, ""));
    
    		//add additional headers
    		if (headers != null) {
    			Iterable<String> headerNames = JSONToolkit.keys(headers);
    			for (String headerName : headerNames) {
    			    Message.RecipientType recipientType = _RECIPIENT_ADDRESS_HEADERS.get(headerName);
    			    if(recipientType != null) {
    			        appendRecipientHeader(part, headers, recipientType, headerName);
    			    } else if (_KNOWN_ADDRESS_HEADER_NAMES.containsKey(headerName.toLowerCase()) || _KNOWN_ADDRESS_HEADER_NAMES.containsValue(headerName)) {
    					setHeaderConditional(part, _KNOWN_ADDRESS_HEADER_NAMES.get(headerName.toLowerCase()), getAddress(headers, headerName, null));
    			    } else if (_KNOWN_HEADER_NAMES.containsKey(headerName.toLowerCase()) || _KNOWN_HEADER_NAMES.containsValue(headerName)) {
                        setHeaderConditional(part, _KNOWN_HEADER_NAMES.get(headerName.toLowerCase()), getValue(headers, headerName, 0, null));
    				} else if (!_SPECIAL_HEADER_NAMES.containsKey(headerName.toLowerCase()) && !_SPECIAL_HEADER_NAMES.containsValue(headerName)) {
    					setHeaderConditional(part, headerName, getValue(headers, headerName, 0, null));
    				}
    			}
    		}
    		
    	} catch (JSONException e) {
    		throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JSON_ERROR,
    				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException by add header: " + e.getMessage());
    
    	} catch (MessagingException e) {
    		throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JAVAPI_MESSAGING_ERROR,
    				ResponseStatusCode.INTERNAL_ERROR, "MessagingException by add header: " + e.getMessage());
    	}
    }
}
