/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.cache;

import java.io.Serializable;
import java.util.Set;
import com.openexchange.usm.api.session.assets.CompleteSessionID;

/**
 * {@link SyncStateCache}: A cache for the sync states stored by a single USM session. A SyncStateCacheProvider may return the same
 * SyncStateCache for more than 1 USM session. The cache implementation has then to make sure that the stored data does not interfere
 * between sessions. Every method in this interface also provides the CompleteSessionID to identify the session. It is guaranteed that the
 * CompleteSessionID is always the same as the one that generated the cache (SyncStateCacheProvider.getCache()). The USM session is uniquely
 * identified by the SessionID within the CompleteSessionID, the cache may use the additional information (UserID) as it sees fit.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public interface SyncStateCache {

    /**
     * Store the given sync state (data) for the given timestamp in the cache. A sync state is uniquely identified by the following tuple:
     * (SessionID, objectID, timestamp). Only store the data if no data already exists in the cache for the given tuple. Return true if
     * storage was successful, false otherwise.
     * 
     * @param session CompleteSessionID, same as when the SyncStateCache was retrieved from its SyncStateCacheProvider
     * @param objectID arbitrary object ID, normally not longer than 64 characters
     * @param timestamp syncid for the sync state
     * @param data sync state data
     * @return true if data was stored, false if not
     */
    boolean put(CompleteSessionID session, String objectID, long timestamp, Serializable[][] data);

    /**
     * Return a previously stored sync state from the cache. If no sync state is found for (SessionID, objectID, timestamp), return null.
     * 
     * @param session
     * @param objectID
     * @param timestamp
     * @return sync state for (SessionID, objectID, timestamp), or null if not found
     */
    Serializable[][] get(CompleteSessionID session, String objectID, long timestamp);

    /**
     * Remove the specified (obsolete) sync state from the cache (if found). It is no longer used by the sync system.
     * 
     * @param session
     * @param objectID
     * @param timestamp
     */
    void remove(CompleteSessionID session, String objectID, long timestamp);

    /**
     * Remove all sync states stored for all tuples (SessionID, objectID1),(SessionID, objectID2),... The objects have been removed (e.g.
     * folders have been deleted).
     * 
     * @param session
     * @param objectIDs
     */
    void removeSyncStates(CompleteSessionID session, String... objectIDs);

    /**
     * Remove all sync states stored except for all tuples (SessionID, objectID1),(SessionID, objectID2),... The given objects represent the
     * complete list of all synchronized folders, all other objects are no longer used.
     * 
     * @param session
     * @param objectIDs
     */
    void retainSyncStates(CompleteSessionID session, Set<String> objectIDs);

    /**
     * Change all stored sync states (i.e. for all timestamps) for the given tuple (SessionID, oldObjectID) by replacing oldObjectID with
     * newObjectID. Sometimes object IDs may change (e.g. rename or move of email folders), the stored sync states in the cache must then be
     * updated to reflect the new object ids.
     * 
     * @param session
     * @param oldObjectID
     * @param newObjectID
     */
    void remapStates(CompleteSessionID session, String oldObjectID, String newObjectID);
}
