/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.rest.services.users.mailMapping;

import java.util.HashSet;
import java.util.Set;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.PathSegment;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.context.ContextService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.ldap.User;
import com.openexchange.mailmapping.MailResolver;
import com.openexchange.mailmapping.ResolvedMail;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.user.UserService;

/**
 * {@link MailMappingRESTService}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since 7.8.0
 */
@Path("/utilities/mailResolver/v1")
public class MailMappingRESTService {

    private final ServiceLookup services;

    /**
     * Initializes a new {@link MailMappingRESTService}.
     */
    public MailMappingRESTService(ServiceLookup services) {
        super();
        this.services = services;
    }

    /**
     * GET /rest/utilities/mailResolver/resolve/[mail1];[mail2]
     * 
     * Tries to resolve the given mail addresses to context and user IDs.
     * Returns an Object with the mail addresses as keys and objects with the keys "user" (for the userId) and "context" (for the contextId) as values.
     * If a mail address is unknown, the corresponding key will be missing in the response.
     */
    @GET
    @Path("/resolve/{mail}")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public JSONObject resolve(@PathParam("mail") PathSegment mail) throws OXException {
        JSONObject response = new JSONObject();
        Set<String> mails = new HashSet<String>();
        mails.add(mail.getPath());
        mails.addAll(mail.getMatrixParameters().keySet());

        MailResolver resolver = services.getService(MailResolver.class);
        for (String m : mails) {
            ResolvedMail resolved = resolver.resolve(m);
            if (resolved == null) {
                continue;
            }
            ContextService contexts = services.getService(ContextService.class);
            Context ctx = contexts.getContext(resolved.getContextID());
            User user = services.getService(UserService.class).getUser(resolved.getUserID(), ctx);

            try {
                JSONObject userJson = new JSONObject();
                userJson.put("language", user.getPreferredLanguage());
                userJson.put("displayName", user.getDisplayName());

                JSONObject resolvedJson = new JSONObject();
                resolvedJson.put("uid", resolved.getUserID());
                resolvedJson.put("cid", resolved.getContextID());
                resolvedJson.put("user", userJson);

                response.put(m, resolvedJson);
            } catch (JSONException e) {
                throw AjaxExceptionCodes.JSON_ERROR.create(e.getMessage());
            }
        }

        return response;
    }

}
