/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.tasks;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import com.openexchange.ant.data.ProjectSetFileReader;
import com.openexchange.build.utils.Tools;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;


/**
 * {@link Scan4ChangelogStrategy}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class Scan4ChangelogStrategy extends Task {

    private File rootDir;

    public Scan4ChangelogStrategy() {
        super();
    }

    public void setRootDir(File rootDir) {
        this.rootDir = rootDir;
    }

    @Override
    public void execute() throws BuildException {
        Project project = getProject();
        String changelogStrategyDebchange = project.getProperty("changelogStrategyDebchange");
        String changelogStrategyTags = project.getProperty("changelogStrategyTags");
        if (null != changelogStrategyDebchange || null != changelogStrategyTags) {
            log("Second run, will scan nothing.", Project.MSG_INFO);
            return;
        }
        String projectSets = project.getProperty("projectSets");
        String packageName = project.getProperty("packageName");
        if (null == projectSets && null == packageName) {
            throw new BuildException("Please define either projectSets referencing packages or packageName to scan.");
        }
        if (null != packageName) {
            setStrategyProperties(project, scanPackage(packageName));
            return;
        }
        List<String> packages = new ArrayList<String>();
        for (String projectSet : Tools.split(projectSets, ',')) {
            File dir = new File(rootDir, projectSet);
            if (!dir.exists()) {
                throw new BuildException('"' + dir.getAbsolutePath() + "\" does not exist.");
            }
            if (!dir.isDirectory()) {
                throw new BuildException('"' + dir.getAbsolutePath() + "\" is not a directory.");
            }
            File[] psfFiles = dir.listFiles(new FilenameFilter() {
                public boolean accept(File dir, String name) {
                    return name.endsWith(".psf");
                }
            });
            for (File psfFile : psfFiles) {
                ProjectSetFileReader reader = new ProjectSetFileReader(psfFile.getAbsolutePath());
                packages.addAll(reader.getProjectList());
            }
        }
        int numTemplates = 0;
        int numNonTemplates = 0;
        for (String paket : packages) {
            if (scanPackage(paket)) {
                numTemplates++;
            } else {
                numNonTemplates++;
            }
        }
        // scan whatever for mixture of old changelogs and new -> error
        if (numTemplates > 0 && numNonTemplates > 0) {
            throw new BuildException("It's not allowed to mix debian/changelog templates with non-templates in one build");
        }
        setStrategyProperties(project, numTemplates > 0);
    }

    private void setStrategyProperties(Project project, boolean isTagsStrategy) {
        log("Using " + (isTagsStrategy ? "tags" : "debchange") + " strategy", Project.MSG_INFO);
        project.setInheritedProperty("changelogStrategyDebchange", Boolean.valueOf(!isTagsStrategy).toString());
        project.setInheritedProperty("changelogStrategyTags", Boolean.valueOf(isTagsStrategy).toString());
    }

    private boolean scanPackage(String packageName) {
        String line;
        boolean isTemplate = false;
        File changelogFile = new File(rootDir, packageName + "/debian/changelog");
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(changelogFile)));
            while ((line = reader.readLine()) != null) {
                if (line.contains("@GITVERSION@")
                    || line.contains("@GITTAGANNOTATION@")
                    || line.contains("@GITTAGAUTHOR@")
                    || line.contains("@GITTAGANNOTATIONTIME@")) {
                    isTemplate = true;
                    break;
                }
            }
            reader.close();
        } catch (IOException e) {
            throw new BuildException("Failed to read changelog file " + changelogFile.getAbsolutePath());
        }
        if (isTemplate) {
            log("File " + changelogFile.getAbsolutePath() + " uses tags strategy", Project.MSG_INFO);
        } else {
            log("File " + changelogFile.getAbsolutePath() + " uses debchange strategy", Project.MSG_INFO);
        }
        return isTemplate;
    }
}
