/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import static com.openexchange.ant.Tools.readFile;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import com.openexchange.ant.Tools;
import com.openexchange.ant.data.ChangeLogEntry;

/**
 * Updates a .spec file from the corresponding debian/changelog.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class UpdateSpec extends Task {

    private File changelog, spec;
    private int revision;
    private List<ChangeLogEntry> changelogs = new LinkedList<ChangeLogEntry>();

    public UpdateSpec() {
        super();
    }

    public void setChangeLog(File changelog) {
        this.changelog = changelog;
    }

    public void setSpec(File spec) {
        this.spec = spec;
    }

    @Override
    public void execute() throws BuildException {
        checks();
        parseChangeLog(Tools.readFile(changelog));
        order();
        String newContent = updateSpecFile(readFile(spec));
        writeFile(newContent);
    }

    private void checks() {
        if (null == changelog) {
            throw new BuildException("Please define path to debian/changelog file.");
        }
        if (null == spec) {
            throw new BuildException("Please define path to .spec file.");
        }
        if (!changelog.exists()) {
            throw new BuildException("debian/changelog file does not exist here: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.exists()) {
            throw new BuildException(".spec file does not exist here: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!changelog.isFile()) {
            throw new BuildException("debian/changelog file is not a file: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.isFile()) {
            throw new BuildException(".spec file is not a file: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!changelog.canRead()) {
            throw new BuildException("Can not read debian/changelog file: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.canRead()) {
            throw new BuildException("Can not read .spec file: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!spec.canWrite()) {
            throw new BuildException("Can not write .spec file: \"" + spec.getAbsolutePath() + "\".");
        }
    }

    private static final String CHANGELOG_REGEX = "^[\\w-]+\\s\\([\\d\\.]+-(\\d+)\\)\\s[\\p{Alpha};= ]+[\\p{Space}\\*]+(.*)\\p{Space}+--\\s([\\w\\s><\\.@-]+?)\\s\\s(.*)$";
    private static final Pattern PATTERN = Pattern.compile(CHANGELOG_REGEX, Pattern.MULTILINE);

    private static final DateFormat DEBIAN_DF = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z", Locale.ENGLISH);

    public void parseChangeLog(String changelogContent) {
        Matcher matcher = PATTERN.matcher(changelogContent);
        boolean isFirst = true;
        while (matcher.find()) {
            int currentRevision;
            try {
                currentRevision = Integer.parseInt(matcher.group(1));
            } catch (NumberFormatException e) {
                // TODO enabled the following line when RHEL5 was dropped
                // getProject().log(e.getMessage(), e, Project.MSG_WARN);
                getProject().log(e.getMessage(), Project.MSG_WARN);
                currentRevision = 0;
            }
            if (isFirst) {
                revision = currentRevision;
                isFirst = false;
            }
            try {
                ChangeLogEntry entry = new ChangeLogEntry(DEBIAN_DF.parse(matcher.group(4)), matcher.group(3), matcher.group(2));
                changelogs.add(entry);
            } catch (ParseException e) {
                throw new BuildException(e.getMessage(), e);
            }
        }
    }

    private void order() {
        Collections.sort(changelogs, new Comparator<ChangeLogEntry>() {
            public int compare(ChangeLogEntry o1, ChangeLogEntry o2) {
                return o2.getDate().compareTo(o1.getDate());
            }
        });
    }

    private static final DateFormat SPECFILE_DF = new SimpleDateFormat("EEE MMM dd yyyy", Locale.ENGLISH);

    private String updateSpecFile(String specfileContent) {
        String tmp = Pattern.compile("^(%define\\s+ox_release\\s+)\\d+$", Pattern.MULTILINE).matcher(specfileContent).replaceFirst("$1" + Integer.toString(revision));
        Matcher matcher = Pattern.compile("(^%changelog$).*", Pattern.MULTILINE + Pattern.DOTALL).matcher(tmp);
        final StringBuilder sb;
        if (matcher.find()) {
            tmp = matcher.replaceFirst("$1");
            sb = new StringBuilder(tmp);
            sb.append('\n');
        } else {
            sb = new StringBuilder(tmp);
            sb.append("\n%changelog\n");
        }
        for (ChangeLogEntry entry : changelogs) {
            sb.append("* ");
            sb.append(SPECFILE_DF.format(entry.getDate()));
            sb.append(' ');
            sb.append(entry.getAuthor());
            sb.append('\n');
            sb.append(entry.getComment());
            sb.append('\n');
        }
        return sb.toString();
    }

    private void writeFile(String specfileContent) {
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(spec), "UTF-8"));
            bw.write(specfileContent);
        } catch (IOException e) {
            throw new BuildException(e.getMessage(), e);
        } finally {
            if (null != bw) {
                try {
                    bw.close();
                } catch (IOException e) {
                    throw new BuildException(e.getMessage(), e);
                }
            }
        }
    }
}
