/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.util.*;

import javax.management.*;

public class USMSessionInformation extends StandardMBean implements USMSessionInformationMBean {
	private final static Map<String, String> DESCRIPTIONS = new HashMap<String, String>();
	private final static Map<String, String[]> PARAMETER_NAMES = new HashMap<String, String[]>();
	private final static Map<String, String[]> PARAMETER_DESCRIPTIONS = new HashMap<String, String[]>();

	static {
		DESCRIPTIONS
				.put(
						"WaitForChangesEmailPullDelay",
						"Delay in ms. (>= 0) or percent (< 0) after which an internal pull is performed if a client waits on email folder changes");
		DESCRIPTIONS
				.put(
						"WaitForChangesEmailMinPullDelay",
						"Minimum time in ms. before an internal pull is performed if a client waits on email folder changes. A value < 0 disables internal email pulling completely");
		DESCRIPTIONS.put("MaxSyncStatesInDB",
				"Maximum number of SyncStates kept in database for any folder of a session");
		DESCRIPTIONS
				.put("KeepLastSentSyncStateInDB",
						"If set, the last SyncState used by a client is always kept in the database, if not, only the N newest ones are kept");
		DESCRIPTIONS.put("ActiveSessionCount", "Number of currently active (i.e. stored in local memory) USM sessions");
		DESCRIPTIONS.put("listActiveSessions", "Lists all currently active USM sessions");
		DESCRIPTIONS
				.put(
						"getPropertiesOfActiveSessions",
						"Lists persistent session properties for active sessions (i.e. in memory), filtered by the given parameters. 0, null, \"\" or \"String\" can be used to ignore a specific filter");
		DESCRIPTIONS
				.put(
						"updatePropertiesOfActiveSessions",
						"Updates a persistent session property for all active sessions (i.e. in memory) indicated by the filter parameters. To clear a property, use a value of null or \"\"");
		DESCRIPTIONS
				.put(
						"getPropertiesOfAllSessions",
						"Lists persistent session properties for all sessions (i.e. in database), filtered by the given parameters. 0, null, \"\" or \"String\" can be used to ignore a specific filter");
		DESCRIPTIONS
				.put(
						"updatePropertiesOfAllSessions",
						"Updates a persistent session property for all sessions (i.e. in database) indicated by the filter parameters. To clear a property, use a value of null or \"\"");

		PARAMETER_NAMES.put("getPropertiesOfActiveSessions", new String[] {
				"cid",
				"id",
				"user",
				"protocol",
				"device",
				"field" });
		PARAMETER_DESCRIPTIONS.put("getPropertiesOfActiveSessions", new String[] {
				"Context ID, set to 0 to not filter for a specific context",
				"User ID, set to 0 to not filter for a specific user id",
				"User Login, set to empty or \"String\" to not filter for a specific user login",
				"USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
				"Device-ID, set to empty or \"String\" to not filter for a specific device id",
				"Field, set to empty or \"String\" to show all fields with non-empty values" });

		PARAMETER_NAMES.put("updatePropertiesOfActiveSessions", new String[] {
				"cid",
				"id",
				"user",
				"protocol",
				"device",
				"field",
				"value" });
		PARAMETER_DESCRIPTIONS.put("updatePropertiesOfActiveSessions", new String[] {
				"Context ID, set to 0 to not filter for a specific context",
				"User ID, set to 0 to not filter for a specific user id",
				"User Login, set to empty or \"String\" to not filter for a specific user login",
				"USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
				"Device-ID, set to empty or \"String\" to not filter for a specific device id",
				"Field, must be present, no more that 18 characters long",
				"Value, no more than 32 characters long" });

		PARAMETER_NAMES.put("getPropertiesOfAllSessions", new String[] { "cid", "id", "protocol", "device", "field" });
		PARAMETER_DESCRIPTIONS.put("getPropertiesOfAllSessions", new String[] {
				"Context ID, set to 0 to not filter for a specific context",
				"User ID, set to 0 to not filter for a specific user id",
				"USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
				"Device-ID, set to empty or \"String\" to not filter for a specific device id",
				"Field, set to empty or \"String\" to show all fields with non-empty values" });

		PARAMETER_NAMES.put("updatePropertiesOfAllSessions", new String[] {
				"cid",
				"id",
				"protocol",
				"device",
				"field",
				"value" });
		PARAMETER_DESCRIPTIONS.put("updatePropertiesOfAllSessions", new String[] {
				"Context ID, set to 0 to not filter for a specific context",
				"User ID, set to 0 to not filter for a specific user id",
				"USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
				"Device-ID, set to empty or \"String\" to not filter for a specific device id",
				"Field, must be present, no more that 18 characters long",
				"Value, no more than 32 characters long" });
	}

	private final SessionManagerImpl _sessionManager;

	public USMSessionInformation(SessionManagerImpl sessionManager) throws NotCompliantMBeanException {
		super(USMSessionInformationMBean.class);
		_sessionManager = sessionManager;
	}

	public int getActiveSessionCount() {
		return _sessionManager.getSessionStorage().getSessionCount();
	}

	public boolean getKeepLastSentSyncStateInDB() {
		return _sessionManager.isKeepLastSentSyncStateInDB();
	}

	public void setKeepLastSentSyncStateInDB(boolean set) {
		_sessionManager.setKeepLastSentSyncStateInDB(set);
	}

	public int getMaxSyncStatesInDB() {
		return _sessionManager.getMaxSyncStatesInDB();
	}

	public void setMaxSyncStatesInDB(int states) {
		_sessionManager.setMaxSyncStatesInDB(states);
	}

	public int getWaitForChangesEmailMinPullDelay() {
		return _sessionManager.getWaitForChangesEmailMinPullDelay();
	}

	public void setWaitForChangesEmailMinPullDelay(int value) {
		_sessionManager.setWaitForChangesEmailMinPullDelay(value);
	}

	public int getWaitForChangesEmailPullDelay() {
		return _sessionManager.getWaitForChangesEmailPullDelay();
	}

	public void setWaitForChangesEmailPullDelay(int value) {
		_sessionManager.setWaitForChangesEmailPullDelay(value);
	}

	public String[] listActiveSessions() {
		List<SessionImpl> list = _sessionManager.getSessionStorage().getSessionList();
		if (list.isEmpty())
			return new String[] { ">> No active sessions <<" };
		String[] result = new String[list.size()];
		for (int i = 0; i < result.length; i++)
			result[i] = list.get(i).getDescription();
		return result;
	}

	public String[] getPropertiesOfActiveSessions(int cid, int id, String user, String protocol, String device,
			String field) {
		List<String> result = new ArrayList<String>();
		for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
			if (accept(session, cid, id, user, protocol, device)) {
				if (field != null && field.length() > 0 && !field.equals("String")) {
					result.add(session.getDescription() + ": " + field + " = " + session.getPersistentField(field));
				} else {
					result.add(session.getDescription());
					for (Map.Entry<String, String> entry : session.getPersistentFields().entrySet())
						result.add("    " + entry.getKey() + " = " + entry.getValue());
				}
			}
		}
		if (result.isEmpty())
			return new String[] { ">> No matching active sessions or fields <<" };
		return result.toArray(new String[result.size()]);
	}

	public String updatePropertiesOfActiveSessions(int cid, int id, String user, String protocol, String device,
			String field, String value) {
		if (field == null || field.length() == 0 || field.equals("String"))
			throw new IllegalArgumentException("No field specified");
		if (field.length() > 18)
			throw new IllegalArgumentException("Field name too long (18 characters maximum)");
		if (value != null && value.length() > 32)
			throw new IllegalArgumentException("Field value too long (32 characters maximum)");
		int count = 0;
		for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
			if (accept(session, cid, id, user, protocol, device)) {
				try {
					session.setPersistentField(field, value);
				} catch (Exception e) {
					throw new IllegalStateException("Error while updating " + session.getDescription() + ": " + e, e);
				}
				count++;
			}
		}
		return ">> " + count + " fields updated <<";
	}

	private boolean accept(SessionImpl session, int cid, int id, String user, String protocol, String device) {
		if (cid != 0 && session.getContextId() != cid)
			return false;
		if (id != 0 && session.getUserIdentifier() != id)
			return false;
		if (user != null && user.length() > 0 && !user.equals("String") && !session.getUser().equals(user))
			return false;
		if (protocol != null && protocol.length() > 0 && !protocol.equals("String")
				&& !session.getProtocol().equals(protocol))
			return false;
		if (device != null && device.length() > 0 && !device.equals("String") && !session.getDevice().equals(device))
			return false;
		return true;
	}

	@Override
	protected String getDescription(MBeanInfo info) {
		return "Management Interface for USM Sessions";
	}

	@Override
	protected String getDescription(MBeanFeatureInfo info) {
		String v = DESCRIPTIONS.get(info.getName());
		return (v != null) ? v : super.getDescription(info);
	}

	@Override
	protected String getDescription(MBeanOperationInfo op, MBeanParameterInfo param, int sequence) {
		String[] v = PARAMETER_DESCRIPTIONS.get(op.getName());
		return (v != null) ? v[sequence] : super.getDescription(op, param, sequence);
	}

	@Override
	protected String getParameterName(MBeanOperationInfo op, MBeanParameterInfo param, int sequence) {
		String[] v = PARAMETER_NAMES.get(op.getName());
		return (v != null) ? v[sequence] : super.getDescription(op, param, sequence);
	}

	public String[] getPropertiesOfAllSessions(int cid, int id, String protocol, String device, String field) {
		return PersistentSessionData.getFieldsFromDB(_sessionManager, cid, id, protocol, device, field);
	}

	public String[] updatePropertiesOfAllSessions(int cid, int id, String protocol, String device, String field,
			String value) {
		return PersistentSessionData.updateFieldsInDB(_sessionManager, cid, id, protocol, device, field, value);
	}
}
