/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import static com.openexchange.obs.api.Tools.isProjectBuilding;
import static com.openexchange.obs.api.Tools.somePackageFailed;
import static com.openexchange.obs.api.Tools.getFirstFailedPackage;
import static com.openexchange.obs.api.Tools.getFirstFailedResult;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.ObsPackage;
import com.openexchange.obs.api.PackageStatus;
import com.openexchange.obs.api.Result;

/**
 * @author choeger
 */
public class Wait4Project extends Task {

    private String url;

    private String login;

    private String password;

    private String projectNames[];

    private String repositories[];

    private String failPackageName;

    private String failRepositoryName;

    private String failProjectName;

    private HashSet<String> packageNames;

    private String failProperty;

    private String packageNamesProperty;

    private int sleep = 1000;

    public Wait4Project() {
        super();
    }

    public final void setSleep(final int sleep) {
        this.sleep = sleep;
    }

    public void setProject(final String projectNames) {
        if (projectNames.length() != 0) {
            this.projectNames = projectNames.split(",", 0);
        }
    }

    public final void setRepositories(final String repositories) {
        if (repositories.length() != 0) {
            this.repositories = repositories.split(",", 0);
        }
    }

    public final void setPackageNames(final String packageNames) {
        if (packageNames.length() != 0) {
            this.packageNames = new HashSet<String>(Arrays.asList(packageNames.split(",", 0)));
        }
    }

    public final void setLogin(final String login) {
        this.login = login;
    }

    public final void setPassword(final String password) {
        this.password = password;
    }

    public final void setUrl(final String url) {
        this.url = url;
    }

    public void setFailPackageName(final String failPackageName) {
        this.failPackageName = failPackageName;
    }

    public void setFailRepositoryName(final String failRepositoryName) {
        this.failRepositoryName = failRepositoryName;
    }

    public void setFailProjectName(final String failProjectName) {
        this.failProjectName = failProjectName;
    }

    public void setFailProperty(final String failProperty) {
        this.failProperty = failProperty;
    }

    public void setPackageNamesProperty(final String packageNamesProperty) {
        this.packageNamesProperty = packageNamesProperty;
    }

    @Override
    public void execute() throws BuildException {
        neccessaryPropertiesCheck();
        try {
            final BuildServiceClient bsc = new BuildServiceClient(url, login, password);
            List<Result> results;
            // Wait 180 seconds to get an initial value of packages to wait for
            int counter = 0;
            while (checkPackageNamesProperty() && counter < 180 && packageNames == null) {
                Thread.sleep(sleep);
                counter++;
            }
            if (packageNames == null) {
                log("Checking build status for all packages.", Project.MSG_INFO);
            } else {
                log("Checking build status of the following packages: " + packageNames, Project.MSG_ERR);
            }
            // endless loop until build finished
            Result[] tmpResults;
            boolean didNeverBuild = true;
            do {
                Thread.sleep(sleep);
                checkPackageNamesProperty();
                results = new ArrayList<Result>();
                for (final String projectName : projectNames) {
                    HashSet<String> currentPackages = checkCurPackages(bsc, projectName);
                    final Result[] projectResults = bsc.getProjectStatus(projectName, currentPackages);
                    if (somePackageFailed(projectResults)) {
                        checkFailedResults(projectResults, projectName);
                    }
                    results.addAll(Arrays.asList(projectResults));
                }
                tmpResults = results.toArray(new Result[results.size()]);
                if (isProjectBuilding(tmpResults)) {
                    didNeverBuild = false;
                }
            } while (isProjectBuilding(tmpResults) || didNeverBuild);

        } catch (IOException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (InterruptedException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }

    private void neccessaryPropertiesCheck() throws BuildException {
        if (null == url || 0 == url.length()) {
            throw new BuildException("URL of build service is not defined.");
        }
        if (null == login || 0 == login.length()) {
            throw new BuildException("User for authenticating in build service is not defined.");
        }
        if (null == password || 0 == password.length()) {
            throw new BuildException("Password for authenticating in build service is not defined.");
        }
        if (null == projectNames || 0 == projectNames.length) {
            throw new BuildException("Project in build service is not defined.");
        }
        if (null == repositories || 0 == repositories.length) {
            throw new BuildException("Distribution in build service is not defined.");
        }
        if (sleep < 1000) {
            throw new BuildException("Sleep delay for polling build service must be greater than 1000.");
        }
        if (null == failPackageName || 0 == failPackageName.length()) {
            throw new BuildException("failPackageName is not defined.");
        }
        if (null == failRepositoryName || 0 == failRepositoryName.length()) {
            throw new BuildException("failRepositoryName is not defined.");
        }
        if (null == failProjectName || 0 == failProjectName.length()) {
            throw new BuildException("failProjectName is not defined.");
        }
        if (null == failProperty || 0 == failProperty.length()) {
            throw new BuildException("failProperty is not defined.");
        }
    }

    private HashSet<String> checkCurPackages(final BuildServiceClient bsc, final String projectName) throws BuildServiceException {
        final HashSet<String> currentPackages = new HashSet<String>();
        if (packageNames != null) {
            
            // creates intersect list of packageNames in this project and the given list of packageNames
            for (ObsPackage obsp : bsc.getPackageList(projectName)) {
                currentPackages.add(obsp.getPackageName());
            }
            currentPackages.retainAll(packageNames);
        }
        return currentPackages;
    }

    private void checkFailedResults(final Result[] tmpResults, String projectName) throws BuildException {
        final Result firstFailedResult = getFirstFailedResult(tmpResults);
        if (firstFailedResult == null) {
            throw new BuildException("One or more Results has failed, but could not determine which one.");
        }
        final PackageStatus firstFailedPackage = getFirstFailedPackage(firstFailedResult.getStatuses());
        if (firstFailedPackage == null) {
            throw new BuildException("One or Packages has failed inside result for repository " + firstFailedResult.getRepository().toString() + ", but could not determine which one.");
        }
        setFailProps(projectName, firstFailedResult, firstFailedPackage);
    }

    private void setFailProps(String projectName, Result firstFailedResult, PackageStatus firstFailedPackage) throws BuildException {
        getProject().setInheritedProperty(failProperty, "true");
        log("  Package name:    " + failPackageName + " => " + firstFailedPackage.getPackageName(), Project.MSG_ERR);
        getProject().setInheritedProperty(failPackageName, firstFailedPackage.getPackageName());
        log("  Repository name: " + failRepositoryName + " => " + firstFailedResult.getRepository().getName(),Project.MSG_ERR);
        getProject().setInheritedProperty(failRepositoryName, firstFailedResult.getRepository().getName());
        log("  Project name:    " + failProjectName + " => " + projectName, Project.MSG_ERR);
        getProject().setInheritedProperty(failProjectName, projectName);
        throw new BuildException("Build on build service failed.");
    }

    
    /**
     * Will check the current state of packageNamesProperty
     * @return <code>false</code> if packageNamesProperty is <code>null</code> and <code>true</code> if it is <code>not null</code>
     */
    private boolean checkPackageNamesProperty() {
        if (packageNamesProperty == null) {
            return false;
        }
        if (getProject().getProperty(packageNamesProperty) == null) {
            return true;
        }
        String packageNamesString = getProject().getProperty(packageNamesProperty);
        if (packageNamesString.length() != 0) {
            this.packageNames = new HashSet<String>(Arrays.asList(packageNamesString.split(",", 0)));
        }
        return true;
    }
}
