/**
 *
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 *
 */

ox.gui.initSidepanel = function (sidenode) {

    var newFolderIcon = null;

    // prepare parent node
    $("<div/>", { id: "ox-sidesplit" }).
    addClass("oxStretch").
    append($("<div/>").
            css({ position: "absolute", top: "24px", right: "-12px",
                width: "12px", height: "24px" }).addClass("sp-toggle").
                append(
                        $("<img/>").attr("src", getFullImgSrc("img/arrows/arrow_double_gray_right.gif")).
                        css({ marginTop: "8px"})
                ).click(function() {
                    ox.widgets.sidepanel.toggle();
                    delete ox.widgets.sidepanel.userToggle;
                }).hide()
    ).appendTo(sidenode[0]);

    // create new sidepanel container
    var sidepanel = ox.widgets.sidepanel = new ox.gui.Container("ox-sidepanel-container").
        setLayout(new ox.gui.BorderLayout()).
        css({ background: "white" }).
        addTo(ox.desktop).
        setParentDOMNode(sidenode[0]);

    // initial sidebar width
    sidepanel.oldWidth = 229; // 19.1em

    // function to trigger sidebar toggle
    $("#sidepanel_dragpoint_background").mousedown(function(e) {
        return false;
    });
    sidepanel.toggle = function(state) {
        // set toggle state
        if (state !== undefined) {
            if (state === this.toggled) {
                return; // nothing changed, bye
            }
            this.toggled = state;
        } else {
            this.toggled = !this.toggled;
        }

        $("div.sp-toggle").css({ display: this.toggled ? "" : "none"});
        $(".sp-toggle-img img").css({ display: this.toggled ? "none" : ""});

        // hide sidepanel drag bar
        $("#sidepanel_dragpoint").css({ display: this.toggled ? "none" : ""});

        // hide newFolderIcon when avialable and toggled (nice-to-have :-))
        if (newFolderIcon !== null) {
            newFolderIcon.setVisible(!this.toggled);
        }

        // remember old width of the sidebar before toggle
        if (this.toggled) {
            this.oldWidth = $("#sidesplit").outerWidth();
        }

        // resize split
        resizeSplit("sidesplit", this.toggled ? "3px" : this.oldWidth + "px");

        // trigger global event
        triggerEvent("OX_Sidepanel_Collapse", this.toggled);
    };

    // hide when landing page is portal
    if ((ox.api.config.get("gui.global.landing_page.module") || "portal") === "portal") {
    	sidepanel.toggle(true);
    }

    // view
    var view = ox.widgets.sidepanelViewControl = new ox.gui.ViewControl("ox-side-view").
        setLayoutParam({ position: "center" }).
        css({ background: "white" }).
        setCSSClass("oxStretch");

    // register for resize
    resizeEvents.register("Resized", function () {
        sidepanel.resize();
    });

    // view hash
    var v = ox.widgets.sidepanelViews = {};

    var meta = {
        "portal": {
            title: _("Portal")
        },
        "mail": {
            title: _("E-Mail")
        },
        "calendar": {
            title: _("Calendar"),
            module: "calendar",
            icon: getFullImgSrc("icons/16/calendar.png"),
            sections: {
                "private": _("Private calendars"),
                "shared": _("Shared calendars"),
                "public": _("Public calendars")
            }
        },
        "contacts": {
            title: _("Contacts"),
            module: "contacts",
            icon: getFullImgSrc("icons/16/contacts.png"),
            sections: {
                "private": _("Private contacts"),
                "shared": _("Shared contacts"),
                "public": _("Public contacts")
            }
        },
        "tasks": {
            title: _("Tasks"),
            module: "tasks",
            icon: getFullImgSrc("icons/16/tasks.png"),
            sections: {
                "private": _("Private tasks"),
                "shared": _("Shared tasks"),
                "public": _("Public tasks")
            }
        },
        "infostore": {
            title: _("Infostore"),
            sections: {
                "private": _("Private folders"),
                "all": _("All folders")
            }
        },
        "configuration": {
            title: _("Configuration")
        },
        "foldertree": {
            title: _("Folders")
        }
    };

    // mixin
    ox.gui.util = {};

    ox.gui.util.createFolder = function (options) {

        // options
        var opt = $.extend({
            widget: null,
            success: $.noop,
            folder: 0,
            module: ox.UIController.getModule(),
            data: {}
        }, options);

        // get free title for new folder
        ox.api.folder.getFreeTitle(opt.folder, function (title) {

            // default data
            var defaultData = {
                folder_id: opt.folder,
                id: "",
                module: opt.module,
                own_rights: 1883275524,
                permissions: [],
                standard_folder: 0,
                subscribed: true,
                subfolders: false,
                title: title,
                total: 0,
                unread: 0
            };

            if (opt.module === "mail") {
                defaultData.id = opt.folder + "/" + defaultData.title;
                defaultData.subscr_subflds = true;
                defaultData.tmpFolder = true;
            } else {
                defaultData.id = "-1";
            }

            // default folder data
            opt.data = $.extend(defaultData, opt.data);

            // mark folder as special
            opt.data.$ = { creating: true };

            // get tree & expert mode
            var tree = ox.api.config.get("modules.folder.tree", 0),
                expert = ox.UIController.showExpertTree,
                cache, cleanup, removeFromCache, cont, ok, abort,
                widget = opt.widget,
                parent, revert = false;

            // handlers
            cleanup = function () {
                // re-enable selection
                widget.selection.enable();
                // bind
                ox.api.folder.dispatcher.bind("modify", widget.onModify);
                // avoid leaks
                widget = opt.widget = opt.data = null;
                opt = null;
            };

            ok = function (title) {
                // remove from cache
                removeFromCache();
                // bind
                ox.api.folder.dispatcher.bind("modify", widget.onModify);
                // create folder
                ox.api.folder.create({
                    folder: opt.folder,
                    data: {
                        title: title,
                        module: opt.data.module
                    },
                    success: function (data) {
                        // setting a selection here will not work!
                        // tree repaint is required, so doing this later
                        cleanup();
                    },
                    error: abort
                });
            };
            
            abort = function () {
                removeFromCache();
                if (revert) {
                    parent.subfolders = false;
                    cache = ox.api.cache["folder" + tree];
                    cache.setComplete(opt.folder, false);
                }
                widget.repaint(function () {
                    widget.selection.clickQuiet(opt.folder);
                    cleanup();
                });
            };

            cont = function () {
                // widget can be simple view or tree view, so look for setOpen
                if (widget.setOpen) {
                    widget.setOpen([opt.folder]);
                }
                // repaint
                widget.repaint(function () {
                    setTimeout(function () {
                        widget.selection.clickQuiet(opt.data.id);
                        widget.startEdit(opt.data.id, ok, abort);
                    }, 10);
                });
            };

            if (widget !== null) {
                // reload/load parent's subfolders
                ox.api.folder.getSubFolders({
                    folder: opt.folder,
                    success: function () {
                        // unbind
                        ox.api.folder.dispatcher.unbind("modify", widget.onModify);
                        // get cache
                        cache = ox.api.cache["folder" + tree];
                        // parent folder now has subfolders
                        if ((parent = cache.get(opt.folder))) {
                            if (parent.subfolders === false) {
                                parent.subfolders = true;
                                cache.setComplete(opt.folder, true);
                                revert = true;
                            }
                        }

                        // get proper cache
                        if (opt.module === "mail" || opt.module === "infostore" || expert === true) {
                            // prepend to folder cache
                            cache.prepend(opt.data);
                            removeFromCache = function () {
                                cache.remove(opt.data.id);
                            };
                            cont();
                        } else {
                            // insert into flat folder cache
                            var flatCache = ox.api.cache.folderFlat.get(opt.module);
                            if (flatCache) {
                                (function () {
                                    var section, id, i, $i, item;
                                    for (id in flatCache) {
                                        section = flatCache[id];
                                        for (i = 0, $i = section.length; i < $i; i++) {
                                            item = section[i];
                                            if (item.id === opt.folder) {
                                                section.splice(i + 1, 0, opt.data);
                                                removeFromCache = function () {
                                                    var pos = $.inArray(opt.data, section);
                                                    section.splice(pos, 1);
                                                };
                                                cont();
                                                return;
                                            }
                                        }
                                    }
                                }());
                            }
                        }
                    }
                });
            }
        });
    };

    ox.gui.util.inlineEdit = function (node, value, success, abort, selector) {
        // disable selection?
        if (this.tree && this.tree.selection) {
            this.tree.selection.disable();
        } else if (this.selection) {
            this.selection.disable();
        }
        // this
        var self = this,
            innerNode,
            // input field
            input = $("<input/>", { type: "text", value: "" }).
                bind("keydown", function (e) {
                    e.stopPropagation();
                }).
                bind("selectstart", function (e) {
                    e.stopPropagation();
                }),
            // handler
            unbind = function (value) {
                $("body").unbind("mousedown", fnOk);
                input.unbind("blur keydown selectstart mousedown").remove();
                innerNode.text(value || ""); // bug #20138
                input = innerNode = null;
            },
            fnAbort = function (e) {
                unbind();
                ox.util.call(abort);
            },
            fnOk = function (e) {
                var value = input.val();
                // empty?
                if ($.trim(value) !== "") {
                    unbind(value);
                    ox.util.call(success, value);
                } else {
                    // empty value is same as abort
                    fnAbort(e);
                }
            };
        // replace node content
        input.
            css({
                width: "100%",
                border: "1px solid #555",
                borderWidth: "1px 0px 1px 0px",
                background: "#eee"
            }).
            val(value || "").
            bind("mousedown", function () {
                return false;
            }).
            bind("keydown", function (e) {
                switch (e.which) {
                case 13: // enter
                    fnOk(e);
                    return false;
                case 27: // escape
                    fnAbort(e);
                    return false;
                }
            }
        );
        // bind ok
        input.bind("blur", fnOk);
        $("body").bind("mousedown", fnOk);
        // add to node
        innerNode = selector !== undefined ? $(selector, node) : $(node);
        innerNode.empty().append(input);
        // focus input field
        input.focus();
        input.select();
    };

    var addRootFolder = configGetKey("modules.folder.tree") == 1;
    // Icon for adding new folders to the root level of the advanced tree when tree=1
    if (addRootFolder) {
        newFolderIcon = new ox.gui.Image().
        setSrc(getFullImgSrc("img/folder/newfolder.gif")).setTitle(_("New folder")).
        css({ verticalAlign: "middle", cursor: "pointer", marginLeft: "5px", marginTop: "4px" }).
        setVisible(false).
        setLayoutParam({ position: "left", width: 4 });
        if (newFolderIcon.image) {
            // adjust width and height for IE
            // todo: needs to be done in stylesheet
            jQuery(newFolderIcon.image).css({ width: "16px", height: "16px" });
        }

        // enable only when expertview is on
        ox.UIController.dispatcher.bind("modulechange expertchanged", function (e) {
            var enabled = (ox.UIController.showExpertTree === true || ox.UIController.getModule() === "mail");
            newFolderIcon.setVisible(enabled);
            newFolderIcon.layout.width = enabled ? "30px" : "4px"; // change width
            ox.widgets.sidepanel.title.paint(); // trigger paint
        });
    }

    // flat view
    (function () {

        /*
         * Context Menu
         * ------------------------------------------------------------------------------------
         */

        if (addRootFolder) {
            var rootMenu = globalContextMenus.newFolderTypeExpert = new PulldownMenu(newFolderIcon.dom.node);
            rootMenu.getContext = constant({
                data: { id: 1 },
                widget: ox.widgets.sidepanelViews.folderTree
            });
        }

        globalContextMenus.newFolderType = new PopupMenu();

        var PIM = { calendar: 1, tasks: 1, contacts: 1, system: 1 };
        var mail = { mail: 1 };
        if (addRootFolder) {
            PIM["mail"] = 1;
            mail = PIM;
        }
        var items = {
            mail:      { name: _("E-Mail"),    when: mail },
            calendar:  { name: _("Calendar"),  when: PIM },
            tasks:     { name: _("Tasks"),     when: PIM },
            contacts:  { name: _("Contacts"),  when: PIM },
            infostore: { name: _("InfoStore"), when: { infostore: 1 } }
        };

        for (var i in items) {
            // enabled or at least visible for upsell?
            if (ox.upsell.isVisible(i) && ox.upsell.isVisible(i + "-new-folder")) {

                (function (type) {

                    function createHandler(context) {
                        // Disable the default repaint and manually trigger it later.
                        // This allows to specify a callback for the repaint.
                        var widget = context.widget !== undefined ? context.widget : ox.widgets.sidepanelViews.folderTree;
                        // non-expert and module mail needs to be another widget
                        if (type === "mail" && ox.UIController.showExpertTree === false
                            && ox.api.config.get("modules.folder.tree", 0) === 1) {
                            widget = ox.widgets.sidepanelViews.mailTree;
                        }
                        // create
                        ox.gui.util.createFolder({
                            widget: widget,
                            module: type,
                            folder: context.data.id
                        });
                    }
                    var def = items[type];
                    var key = "modules." + type + ".module";
                    var item = new MenuItem(def.name, createHandler);
                    item.onShow = function () {
                        // get context
                        var ctx = this.getContext();
                        // get module
                        if (ox.UIController.showExpertTree) {
                            // show?
                            if (ctx.data && ctx.data.id === "2" && type === "mail") {
                                this.setEnabled(false); // no mail in public folders
                            } else {
                                this.setEnabled((ctx.type in def.when) && configGetKey(key)); // all
                            }
                        } else {
                            // exclude unified mail (subfolders too)
                            if (ox.api.folder.is("unifiedmail", ctx.data)) {
                                this.setEnabled(false);
                            } else {
                                // allow current type only
                                this.setEnabled(type === ox.UIController.getModule());
                            }
                        }
                    };
                    globalContextMenus.newFolderType.addItem(item);
                    if (addRootFolder) {
                        item = new MenuItem(def.name, createHandler);
                        item.onShow = function() {
                            if (ox.UIController.showExpertTree) {
                                this.setEnabled(("system" in def.when) && configGetKey(key));
                            } else {
                                this.setEnabled(type === "mail");
                            }
                        };
                        rootMenu.addItem(item);
                    }
                })(i);
            }
        }

        var menu = new ContextMenu();

        // create global refs
        globalContextMenus.simpleFolderMenu = menu;

        // new folder
        var new_folder = new MenuItem(_("New folder"), globalContextMenus.newFolderType);
        new_folder.setIcon("img/folder/newfolder.gif", "img/folder/newfolder_d.gif");
        new_folder.onShow = function () {
        	var ctx = this.getContext(),
        		canCreate = false;
        	// check for the create folder permission
        	if (ctx.data && ctx.data.id) {
            	canCreate = ox.api.folder.can("create", 
            			ox.api.folder.get({ folder: ctx.data.id }));
        	}
        	
            // check if creating folder is allowed for current module
            var module = ox.UIController.getModule();
            this.setEnabled(ox.upsell.isVisible(module + "-new-folder") 
            		&& canCreate);
        };
        menu.addItem(new_folder);

        // send as link
        var link = new MenuItem(_("Send as link"), function (context) {
            // get folder first
            ox.api.folder.get({
                folder: context.data.id,
                success: function (data) {
                    // use global fn
                    infostore_sendLink([{ data: {
                        folder_id: data.id,
                        module: data.module
                    }}]);
                }
            });
        });
        menu.addItem(link);

        // subscribe
        var subscribe = new MenuItem(_("Subscribe folder"), function (context) {
            SubscribeFolder.openWindow({ selected: ox.api.account.derive(context.data.id, true) });
        });
        menu.addItem(subscribe);

        // "Extended" (submenu)
        var extended = globalContextMenus.extended_folder_menu = new PopupMenu();
            // Import
            var import_data = new MenuItem(_("Import"), function (context) {
                ox.api.folder.get({
                    folder: context.data.id,
                    success: function (data) {
                        // set global var
                        importFolderDestination = data;
                        triggerEvent("OX_Switch_View", "configuration/import");
                    }
                });
            });
            extended.addItem(import_data);
            // Export (Submenu)
            var export_submenu = new PopupMenu();
                var export_submenu_ical = new MenuItem(_("iCalendar"),
                    function (ctx) {
                        ox.api.folder.exportContacts(ctx.data.id, "ICAL");
                    });
                export_submenu.addItem(export_submenu_ical);
                var export_submenu_vcard = new MenuItem(_("vCard"),
                    function (ctx) {
                        ox.api.folder.exportContacts(ctx.data.id, "VCARD");
                    });
                export_submenu.addItem(export_submenu_vcard);
                var export_submenu_csv = new MenuItem(_("CSV"), function (ctx) {
                    ox.api.folder.exportContacts(ctx.data.id, "CSV");
                });
                export_submenu.addItem(export_submenu_csv);
                var export_contact = new MenuItem(_("Export"), export_submenu);
                extended.addItem(export_contact);
        // add to menu
        menu.addItem( new MenuItem(_("Extended"), extended));

        // separator
        menu.addItem(new MenuSeparator());

        // delete
        var del = new MenuItem(_("Delete"), function (context) {
            removeFolder(context.data.id);
        });
        del.setIcon("img/menu/delete_folder.gif", "img/menu/delete_folder_d.gif");
        menu.addItem(del);

        // rename
        var rename = new MenuItem(_("Rename"), function (context) {
            // start edit
            context.widget.startEdit(context.data.id);
        });
        rename.setIcon("img/folder/renamefolder.gif", "img/folder/renamefolder_d.gif");
        menu.addItem(rename);

        var cut = new MenuItem(_("Cut"), function (context) {
            // cut node
            context.widget.cut(context.data.id);
        });
        cut.setIcon("img/folder/cutfolder.gif", "img/folder/cutfolder_d.gif");
        menu.addItem(cut);

        var paste = new MenuItem(_("Paste"), function (context) {
            ox.api.folder.get({
                folder: context.data.id,
                success: function(target) {
                    ox.api.folder.move(context.widget.cutNode.data, target);
                    context.widget.cut();
                }
            });
        });
        paste.setIcon("img/folder/pastefolder.gif", "img/folder/pastefolder_d.gif");
        menu.addItem(paste);

        var folder_read = new MenuItem(_("Mark folder read"), function (context) {
            // mark every mail as read
            triggerEvent("OX_Folder_Mark_Read", context.data.id);
        });
        folder_read.setIcon("icons/16/mark_as.png", "icons/16/mark_as.png");
        menu.addItem(folder_read);

        var empty = new MenuItem(_("Empty folder"), function (context) {
            // clear folder
            var text = "Are you sure you want to delete all E-Mails from this folder? The deleted E-Mails will be moved to the Trash folder."; /*i18n*/
            // permanently delete message or is folder the trash folder?
            var accounts = ox.api.account.all(), trash = false;
            for (var i=0; i < accounts.length; i++) {
                if (accounts[i].trash_fullname === context.data.id) {
                    trash = true;
                    break;
                }
            }
            if (configGetKey("mail.deletemail") || trash === true) {
                text = "Are you sure you want to permanently delete all E-Mails from this folder? The deleted E-Mails will be irrevocably lost."; /*i18n*/
            }
            accounts = trash = null;
            function cbok () {
                // select folder to clear to avoid a removed subfolder still being selected
                ox.UIController.setFolder({
                    folder: context.data.id,
                    module: "default",
                    success: function () {
                        // clear folder now
                        ox.api.folder.clear(context.data.id, function () {
                            // update folder / sub folders
                            ox.api.http.pause();
                            ox.api.folder.get({
                                folder: context.data.id,
                                cache: false
                            });
                            ox.api.folder.getSubFolders({
                                folder: context.data.id,
                                cache: false
                            });
                            ox.api.folder.get({
                                folder: configGetKey("modules.mail.defaultFolder.trash"),
                                cache: false
                            });
                            ox.api.http.resume(function () {
                                // trigger
                                triggerEvent("OX_Folder_Cleared", context.data.id);
                                // trigger modify
                                ox.api.folder.dispatcher.trigger("modify");
                            });
                        });
                    }
                });
            }
            newConfirm(_("Empty folder"),_(text),AlertPopup.YESNO,null,null,cbok,null,null);

        });
        empty.setIcon("img/folder/empty_folder.gif", "img/folder/propertiesfolder_d.gif");
        menu.addItem(empty);

        // separator
        menu.addItem(new MenuSeparator());

        var props = new MenuItem(_("Properties"), function (context) {
            ox.api.folder.viewProperties(context.data.id, false);
        });
        props.setIcon("img/folder/propertiesfolder.gif", "img/folder/propertiesfolder_d.gif");
        menu.addItem(props);

        menu.onShow = function (context) {
            // get context
            var id = context.data.id, api = ox.api.folder;
            // get folder
            api.get({
                folder: id,
                success: function (folder) {

                    // new folder
                    new_folder.setEnabled((folder.own_rights & 127) >= 4);

                    // module system and name default is the parent root mail folder
                    context.type = api.is("account", folder) ? "mail": folder.module;

                    // subscribe
                    subscribe.setEnabled(api.can("subscribe", folder));
                    subscribe.setVisible(api.is("mail", folder));

                    // send link (anything but system and mail)
                    link.setEnabled(folder.own_rights & (127 << 7) && folder.module != "mail" && ("infostore_sendLink" in window));
                    link.setVisible(!api.is("system", folder) && context.type !== "mail");

                    // delete
                    del.setEnabled(api.can("delete", folder));

                    // rename
                    rename.setEnabled(api.can("rename", folder));

                    cut.setVisible(context.widget.cut !== undefined);
                    if (cut.visible) {
                        cut.setEnabled(del.enabled &&
                            (!context.widget.cutNode ||
                                context.widget.cutNode.id != id));
                    }

                    paste.setVisible(context.widget.cut !== undefined);
                    if (paste.visible) {
                        paste.disable();
                        if (context.widget.cutNode) {
                            api.canMove(context.widget.cutNode.data, folder,
                                function() { paste.enable(); },
                                function() { paste.disable(); });
                        }
                    }

                    // empty
                    empty.setVisible(api.can("empty", folder));

                    // mark folder read
                    folder_read.setVisible(api.is("mail", folder));

                    // import
                    import_data.setEnabled(api.can("import", folder));

                    // export
                    export_contact.setEnabled(api.can("export", folder));
                    export_submenu_ical.setVisible(api.is("calendar|tasks", folder));
                    export_submenu_vcard.setVisible(api.is("contacts", folder));
                    export_submenu_csv.setVisible(api.is("contacts", folder));
                }
            });
        };

        /*
         * Tree views
         * ------------------------------------------------------------------------------------
         */
        ox.gui.TypedFolderTree = ox.gui.Tree.extend({

            construct: function (id, options) {

                // super constructor
                ox.gui.Tree.prototype.construct.call(this, id);

                // options
                this.options = $.extend({
                    type: "mail",
                    icon: "mail.png",
                    root: "1",
                    tree: ox.api.config.get("modules.folder.tree", "0"),
                    showRootNode: false,
                    loadRootNode: false
                }, options || {});

                this.css({ padding: "10px 0px 10px 0px" });

                this.firstPaint = true;

                // show root node
                this.showRootNode = this.options.showRootNode;
                this.loadRootNode = this.options.loadRootNode;
                this.loadRootNodeId = this.options.root;

                var TYPE = this.options.type;
                var self = this;

                // node parser
                this.setNodeParser(function (data) {
                    // root node fix
                    if (data) { // fix to prevent error on console TODO fix it real
                        var parent = data.id === self.options.root ? null : data.folder_id;
                    }

                    return {
                        "id": data.id, "parent": parent, "name": data.title,
                        "type": data.module, "hasChildren": data.subfolders
                    };
                });

                this.addListener("widget:select", function(e) {
                    var api = ox.api.folder;
                    // get folder data
                    api.get({
                        folder: e.data,
                        success: function (data) {
                            // can read?
                            if (api.can("read", data)) {
                                // mail?
                                if (TYPE === "mail") {
                                    if (api.is("mail", data)) {
                                        // change folder
                                        ox.UIController.setFolder({ folder: data.id, module: "mail" });
                                    } else if (api.is("messaging", data)) {
                                        // change folder AND module
                                        ox.UIController.setFolder({ folder: data.id, module: "messaging" });
                                    }
                                } else {
                                    // change folder
                                    ox.UIController.setFolder({ folder: data.id });
                                }
                            }
                        }
                    });
                });

                this.selection.onRename(function (id, item) {
                    // get folder
                    ox.api.folder.get({
                        folder: id,
                        success: function (data) {
                            // can rename?
                            if (ox.api.folder.can("rename", data)) {
                                // start edit
                                self.startEdit(id);
                            }
                        }
                    });
                });

                /**
                 * Context Menu via Left Click
                 */
                this.selection.dispatcher.bind("selected", function(e) {
                    // add context button
                    if (e.node.find(".context-button").length === 0) {
                        var button = $.button({
                            title: "\u00a0\u00a0"
                        }).bind("click", function(event) {
                            self.selection.triggerContextMenu(e.id, event);
                            return false;
                        });
                        if (IE <= 7) {
                            button.css("width", "");
                        }
                        if (IE) {
                            button = $('<form onsubmit="return false"></form>')
                                .css("margin", "0")
                                .append(button);
                        }
                        var td = $("<td/>").css({
                            width: "20px",
                            textAlign: "right",
                            paddingRight: "5px"
                        }).addClass("context-button").append(button);
                        if (IE <= 7) {
                            td.css({ "width": "20px" });
                        }
                        e.node.find("tr").append(td);
                    } else {
                        e.node.find(".context-button").show();
                    }
                });
                this.selection.dispatcher.bind("deselected", function(e) {
                    // remove context button
                    e.node.find(".context-button").hide();
                });

                this.addListener("widget:edited", function (e) {
                    // rename folder
                    ox.api.folder.rename({
                        folder: e.data.id,
                        title: e.data.value,
                        success: function (data) {
                            // get current tree id
                            var ID = self.id + "/" + ox.api.config.get("modules.folder.tree", 0);
                            // has parent node already been opened?
                            if (configGetKey(["gui", "tree", ID, "state", e.data.id]) === true) {
                                // remove previous open/close state
                                configRemoveKey(["gui", "tree", ID, "state", e.data.id]);
                                // update user config
                                configSetKey(["gui", "tree", ID, "state", data.id], true);
                                // set to open
                                self.setOpen([data.id]);
                            }
                            // continue
                            ox.util.call(e.cont, data);
                        },
                        error: function () {
                            // repaint
                            self.repaint();
                            // select
                            self.selection.select(e.data.id);
                            // continue
                            ox.util.call(e.cont);
                        }
                    });
                    return false;
                });

                var displayName = "";
                this.customize = function (node) {

                    var messagingServices = {
                        "com.openexchange.messaging.facebook": {
                            icon: "facebook.png"
                        },
                         "com.openexchange.messaging.rss": {
                            icon: "rss.png"
                        },
                        "com.openexchange.messaging.twitter": {
                            icon: "twitter.png"
                        }
                    };

                    var data = node.data, api = ox.api.folder, id;
                    var name = data.title;

                    // clear all custom styles
                    $(node.dom.title).parent().css({ fontWeight: "" }).
                        removeClass("font-color-disabled font-style-low");
                    
                    $(node.dom.icon).css({ backgroundImage: "" });

                    // add style
                    var processUnread = function (meta) {
                        if (data.unread !== null && data.unread > 0) {
                            $(node.dom.title).parent().css({ fontWeight: "bold" });
                            meta.name = meta.name;
                            meta.html = document.createTextNode(meta.name +
                                " (" + data.unread + ")");
                        }
                        return meta;
                    };

                    // top folder?
                    if ((this.options.tree === 0 && data.folder_id === this.options.root) ||
                            (ox.api.folder.is("account", data)) ) {
                        src = this.options.icon;
                        if (ox.api.folder.is("account", data)) {
                            // auto-open external accounts
                            node.selectable = false;
                            
                            // click on root of mailaccounts should always select the INBOX
                            // need to temporary overide next
                            var account = ox.api.account.get(ox.api.account.derive(data.id));
                            if (account) {
                                var prevnext = node.next;
                                node.next = function(skipChildren) {
                                    if (this.statusOpen && this.children.length > 0) {
                                        // iterate through children to search for INBOX node
                                        for (var i = 0; i < this.children.length; i++) {
                                            if (this.children[i].id === account.inbox_fullname) {
                                                return this.children[i];
                                            }
                                        }
                                    }
                                    // use default
                                    prevnext();
                                };
                            }
                        }
                    }
                    else {
                        var src = "folder_closed" /* default icon */, accountId, a = {};
                        // get account id
                        accountId = /^default(\d*)\b/.exec(data.id);
                        if ($.isArray(accountId)) {
                            // default0?
                            if (accountId[1] === "0") {
                                // use config
                                var f = config.mail.folder;
                                a.inbox_fullname = f.inbox;
                                a.trash_fullname = f.trash;
                                a.sent_fullname = f.sent;
                                a.drafts_fullname = f.drafts;
                                a.spam_fullname = f.spam;
                            } else {
                                // get account data from cache
                                a = ox.api.cache.account.get(accountId[1]);
                                id = "default" + accountId[1];
                                // unified inbox, e.g. behaves like an account but is not
                                if (!a) {
                                    a = {
                                        inbox_fullname: id + "/INBOX",
                                        trash_fullname: id + "/Trash",
                                        sent_fullname: id + "/Sent",
                                        drafts_fullname: id + "/Drafts",
                                        spam_fullname: id + "/Spam"
                                    };
                                }
                                // inbox still not set?
                                if (a.inbox_fullname === undefined) {
                                    a.inbox_fullname = id + "/INBOX";
                                }
                            }
                        }

                        // special mail folder?
                        switch (data.id) {
                        case a.inbox_fullname: src = "inbox.png"; break;
                        case a.trash_fullname: src = "garbage.png"; break;
                        case a.sent_fullname: src = "outbox.png"; break;
                        case a.drafts_fullname: src = "draft.png"; break;
                        case a.spam_fullname: src = "spam.png"; break;
                        case a.confirmed_ham_fullname: src = "ham.png"; break;
                        case a.confirmed_spam_fullname: src = "spam.png"; break;
                        default:
                            // normal mail folder
                            // can read?
                            if ((api.is("system", data) || api.can("read", data)) && node.isCut !== true) {
                                // ok
                                src = node.statusOpen && node.hasChildren ? "folder_opened.png" : "folder_closed.png";
                                $(node.dom.title.parentNode).removeClass("font-color-disabled");
                            } else {
                                // cannot read
                                src = node.statusOpen && node.hasChildren ? "folder_opened_dis.png" : "folder_closed_dis.png";
                                $(node.dom.title.parentNode).addClass("font-color-disabled");
                            }
                        }
                        // tweaking infostore icons
                        switch (data.module) {
                            case "infostore":
                                if (data.folder_id === "10"){
                                    src = "user";
                                }else{
                                    src = "infostore";
                                }
                                // can read?
                                if (!api.can("read", data)) {
                                    // don't visually change system folders
                                    if (data.folder_id === "1") {
                                        src = src + "_dis.gif"; // disable symbol
                                        $(node.dom.title.parentNode).addClass("font-color-disabled font-style-low"); // gray italic
                                    } else {
                                        src = src + ".png";
                                    }
                                    node.selectable = false;
                                } else {
                                    src = src + ".png";
                                    node.selectable = true;
                                }
                            case "messaging":
                                var service = /^(.*?):/.exec(data.id);
                                if (service && messagingServices[service[1]]){
                                   src = messagingServices[service[1]].icon;
                                }
                                break;
                        }
                        // mark published folders
                        if (ox.api.folder.is("published", data)) {
                            // add shared symbol to original
                            $(node.dom.icon).css({ backgroundImage: "url(" + ox.gui.themePath + "icons/16/" +src+ ")" });
                            src = "shared_globe.png";
                        }
                    }

                    if (data.id === "default0") {
                        // replace with user name
                        var identifier = config.identifier;
                        var user = internalCache.getUsers([identifier], function (data) {
                            displayName = data[identifier].display_name;
                        });
                        if (user && user[identifier]) {
                            name = displayName = user[identifier].display_name;
                        } else {
                            name = displayName;
                        }
                    }

                    return processUnread({
                        src: ox.gui.themePath + "icons/16/" + src,
                        name: name
                    });
                };

                this.selection.classSelected = "background-color-PMG-selection-elements";

                var fnToggle = function () {
                    // get id
                    var ID = this.id + "/" + ox.api.config.get("modules.folder.tree", 0);
                    // loop over open nodes
                    var state = {};
                    $.each(self.getOpen(), function (i, id) {
                        state[id] = true;
                    });
                    // update user config
                    configSetKey(["gui", "tree", ID, "state"], state);
                };
                this.addListener("widget:open", fnToggle);
                this.addListener("widget:close", fnToggle);

                // click handler
                this.selection.setClickHandler(function (e) {

                    // selection disabled
                    if (self.selection.statusEnabled === false) {
                        return false;
                    }

                    // context menu?
                    if (e.event.rightClick) {
                        // tracking
                        track({
                            type: "context_menu",
                            data: { id: e.id },
                            text: "Right click on folder " + e.id
                        });
                        // process
                        e.click("quiet");
                        e.context();
                    } else if (e.selectable === true && e.id !== "-1") {
                        // viewing properties?
                        if (currentpath[0] === "configuration") {
                            // change back to folder's module
                            ox.UIController.setFolder({
                                folder: e.id,
                                module: "default"
                            });
                        } else {
                            // normal click
                            // track
                            ox.api.folder.get({
                                folder: e.id,
                                success: function (data) {
                                    track({
                                        type: "click",
                                        dest: "tree",
                                        data: { id: data.id, title: data.title },
                                        text: "Select " + data.module + " folder \"" + data.title + "\" (" + e.id + ")"
                                    });
                                }
                            });
                            // process
                            if (ox.UIController.getFolder() !== e.id) {
                                e.click("force");
                            } else {
                                e.click();
                            }
                        }
                    }

                    // prevent default
                    return false;
                });

                // context menu
                this.selection.setContextMenuHandler(function (id, e) {
                    // show folder context menu
                    var menu = globalContextMenus.simpleFolderMenu;
                    menu.display(e.pageX, e.pageY, {
                        data: { id: id },
                        api: true,
                        simple: true,
                        module: TYPE,
                        widget: self
                    });
                });

                // add root node?
                if (this.loadRootNode === false) {
                    this.setRootNode({ id: this.options.root, folder_id: null, title: "Root", module: "root", subfolders: true });
                }

                this.onModify = function (data) {
                    var finalOpenNodes = [];

                    // close invalid nodes
                    if (data.why === "folder.remove" || data.why === "folder.move") {
                        // is mail?
                        if (ox.api.folder.is("mail", data.folder)) {
                            // get id
                            var key = data.oldId;
                            // close invalid nodes
                            $.each(ox.util.keys(self.openNodes).concat(self.getOpen()), function (i, id) {
                                if ((new RegExp("^" + key)).test(id)) {
                                    // get tree node
                                    var node = self.get(id);
                                    if (node) {
                                        // required to re-open later
                                        finalOpenNodes.push(id.replace(key, data.id));
                                        // close
                                        node.close();
                                    }
                                }
                            });
                        }
                        // unset current folder
                        ox.UIController.unsetFolder();

                    } else if (data.why !== "folder.refresh") {

                        // affects current mail folder?
                        var folder = ox.UIController.getFolder("mail"),
                            tree = ox.api.config.get("modules.folder.tree", 0),
                            cache = ox.api.cache["folder" + tree];
                        // is removed?
                        if (folder !== undefined && !cache.contains(folder)) {
                            // unset folder
                            ox.UIController.unsetFolder("mail");
                        }
                    }

                    if (data.why === "folder.update.local") {
                        // repaint affected node only
                        var node = self.get(data.id);
                        if (node) {
                            node.repaint();
                        }
                    } else {
                        if (data.why === "folder.move") {
                            ox.UIController.setFolder({
                                folder: data.id,
                                module: "default",
                                success: function () {
                                    // reopen node
                                    if (self.setOpen) self.setOpen(finalOpenNodes);
                                    self.repaint(function() {
                                        // clean up
                                        finalOpenNodes = null;
                                    });
                                }
                            });
                        } else {
                            // repaint full tree
                            self.repaint();
                        }
                    }
                };
            },

            loadChildren: function (id, cont) {
                // step #2: get sub folders
                var self = this;
                var type = this.options.type;
                var getFolders = function () {
                    ox.api.folder.getSubFolders({
                        folder: id,
                        tree: ox.api.config.get("modules.folder.tree", "0"),
                        success: function (data) {
                            // mail only
                            data = $.grep(data, function (elem, i) {
                                if (type === "mail") {
                                    // mail AND messaging
                                    var isMessaging = elem.module === "messaging",
                                        isMail = elem.module === "mail",
                                        isMailFolder = elem.tmpFolder === true || /^default/.test(elem.id),
                                        isSubscribed = elem.subscribed === true || elem.subscr_subflds === true,
                                        isSystem = elem.module === "system",
                                        isProperFolder = isMailFolder || isMessaging;
                                    return isProperFolder && isSubscribed && (isMail || isMessaging || isSystem);
                                } else {
                                    return (elem.subscribed === true || elem.subscr_subflds === true) && (elem.module === type || elem.module === "system");
                                }
                            });
                            cont(data);
                        },
                        error: function () {
                            cont(false);
                            // return true; // prevent global error reporting
                        }
                    });
                };
                // mail enabled?!?
                if (ox.api.config.get("modules.mail.module") === true) {
                    // step #1: get accounts (cached after first request)
                    ox.api.account.all(getFolders);
                } else {
                    getFolders();
                }
            },

            loadNode: function (id, cont) {
                // get single folder
                var self = this;
                ox.api.folder.get({
                    folder: id,
                    tree: ox.api.config.get("modules.folder.tree", "0"),
                    success: function (data) {
                        ox.util.call(cont, data);
                    }
                });
            },

            // prevent flicker
            clear: function () {
            },

            expandToFolder: function (folderID) {

                // get open nodes
                var open = this.getOpen();

                // this
                var self = this;

                var getParentIDs = function (data, callback) {
                    // loop over folder
                    var i = 1, $l = data.length, list = [], id;
                    for (; i < $l; i++) {
                        id = data[i].id;
                        // not already open?
                        if ($.inArray(id, open) === -1) {
                            list.push(id);
                        }
                    }
                    callback(list);
                };

                var expandTreeNodes = function (list) {

                    // need repaint?
                    if (list.length > 0) {
                        // yes
                        self.setOpen(list);
                        self.repaint(function () {
                            self.selection.clickQuiet(folderID);
                        });
                    } else {
                        // no, select directly
                        self.selection.clickQuiet(folderID);
                    }
                };

                // get all parents of folder via api
                ox.api.folder.getParents({
                    folder: folderID,
                    success: function (data){
                        getParentIDs(data, expandTreeNodes);
                    }
                });
            },

            paint: function (cont) {
                if (this.painter.running === false) {

                    this.painter.running = true;

                    // first paint?
                    if (this.firstPaint === true) {
                        // get folder state id
                        var ID = this.id + "/" + ox.api.config.get("modules.folder.tree", 0);
                        // get open folders
                        if (configContainsKey(["gui", "tree", ID, "state"])) {
                            this.setOpen(ox.util.keys(configGetKey(["gui", "tree", ID, "state"])));
                        } else {
                            this.setOpen(this.options.open || []);
                        }
                        // visually busy
                        $(this.dom.node).css({ background: "" }).addClass("oxBusySmall");
                        // done
                        this.firstPaint = false;
                    }

                    // unbind
                    ox.api.folder.dispatcher.unbind("modify", this.onModify);

                    // this
                    var self = this;

                    // continuation
                    var step2 = function () {
                        // get ids of open nodes (exclude external accounts)
                        var list = $.grep(ox.util.keys(self.openNodes), function (key) {
                            return /^(\d+|default0)/.test(key);
                        });
                        // pause
                        ox.api.http.pause();
                        var pssst = function () {
                            return true; // prevent global error
                        };
                        // loop
                        var i = 0, $l = list.length;
                        for (; i < $l; i++) {
                            ox.api.folder.getSubFolders({
                                folder: list[i],
                                tree: self.options.tree,
                                error: pssst
                            });
                        }
                        // resume
                        ox.api.http.resume(
                            function () {
                                // paint
                                $(self.dom.node).empty().removeClass("oxBusySmall");
                                self.painter.running = false;
                                ox.gui.Tree.prototype.paint.call(self, function () {
                                    // clear open nodes hash
                                    self.openNodes = {};
                                    // select id
                                    if (self.selection !== undefined) {
                                        self.selection.enable();
                                        self.selection.preselect(currentFolder); // before paint
                                        self.selection.clickQuiet(currentFolder); // after paint
                                    }
                                    // bind
                                    ox.api.folder.dispatcher.bind("modify", self.onModify, window);
                                    // continue
                                    ox.util.call(cont);
                                });
                            },
                            pssst
                        );
                    };

                    // consider current folder (not in nested windows)
                    var currentFolder = ox.UIController ? ox.UIController.getFolder() : undefined;
                    if (currentFolder !== undefined) {
                        // get path
                        ox.api.folder.getParents({
                            folder: currentFolder,
                            tree: this.options.tree,
                            success: function (data) {
                                // remove current folder
                                data.shift();
                                // get ids
                                var list = $.map(data, function (folder) {
                                    return folder.id;
                                });
                                // leads to root folder?
                                if ($.inArray(self.options.root, list) > -1) {
                                    // mark as open
                                    self.setOpen(list);
                                }
                                // continue
                                step2();
                            }
                        });
                    } else {
                        // continue
                        step2();
                    }
                } else {
                    this.painter.queue.push(cont);
                }
            }
        });

        /*
         * Mail folder tree
         */
        var tree = ox.api.config.get("modules.folder.tree", 0);

        v.mail = new ox.gui.Container("ox-sidepanel-mail").
        setLayout(new ox.gui.BorderLayout());

        var mail_tree = ox.widgets.sidepanelViews.mailTree = new ox.gui.TypedFolderTree("ox-sidepanel-mail-tree", {
            type: "mail",
            icon: "mail.png",
            root: "1",
            tree: tree,
            open: tree === 0 ? ["default0"] : []
        }).setLayoutParam({ position: "center" });

        mail_tree.enableDnD(ox.gui.initSidepanel.dndSource, ox.gui.initSidepanel.dndTarget);

        v.mail.add(mail_tree);

        // check for plugin
        if (ox.api.config.get("modules.mail.module") === true &&
        		ox.mail.addAccount.available()) {
            // add extension
            mail_tree.addListener("widget:painted", function () {
                var node = $(this.dom.node);
                if (node.find("a.add-account").length === 0) {
                    node.append(
                        $('<div>')
                        .css({ lineHeight: '2em', margin: '0 15px 0 15px' })
                        .append(
                            $('<a>', { href: '#' })
                            .addClass('common-link add-account')
                            .append(addTranslated(_("Add E-Mail account ...")))
                            .bind('click', function (e) {
                                e.preventDefault();
                                // enabled?
                                if (ox.api.config.get("modules.mailaccount.module", false)) {
                                    ox.mail.addAccount();
                                } else {
                                    triggerEvent("Feature_Not_Available", "configuration/mail/accounts/new", window, "foldertree");
                                }
                            })
                        )
                    );
                }
            });
        }

        /*
         * Infostore folder tree
         */
        v.infostore = new ox.gui.Container("ox-sidepanel-infostore").
        setLayout(new ox.gui.BorderLayout());

        var infostore_tree = ox.widgets.sidepanelViews.infostoreTree = new ox.gui.TypedFolderTree("ox-sidepanel-infostore-tree", {
            type: "infostore",
            icon: "infostore.png",
            root: "9",
            open: ["9", "10", ox.api.config.get("folder.infostore")]
        }).setLayoutParam({ position: "center" });
        infostore_tree.enableDnD(ox.gui.initSidepanel.dndSource, ox.gui.initSidepanel.dndTarget);

        v.infostore.add(infostore_tree);

        /*
         * Flat views
         * ------------------------------------------------------------------------------------
         */
        var flatView = ox.gui.Custom.extend({

            construct: function (paint, id, module) {

                // super constructor
                ox.gui.Custom.prototype.construct.call(this, paint, id);

                // module
                this.module = module;
                this.folders = {};

                var self = this;

                // selection
                var s = this.selection = new ox.gui.Selection();
                s.setMultiple(false);
                s.setNodeFinder(function () {
                    return $(".ox-sidepanel-flat-folder", this.container);
                });
                s.classSelected = "background-color-PMG-selection-elements";
                s.onChange(function (items) {
                    if (self.isVisible() && items.length) {
                        var id = String(items[0].id);
                        setTimeout(function () {
                            // change folder
                            ox.UIController.setFolder({ folder: id });
                        }, 10);
                    }
                });
                s.onRename(function (id, item) {
                    // get folder
                    ox.api.folder.get({
                        folder: id,
                        success: function (data) {
                            // can rename?
                            if (ox.api.folder.can("rename", data)) {
                                // start edit
                                self.startEdit(id);
                            }
                        }
                    });
                });
                s.observe(self.dom.node, "ox-sidepanel-view-wrapper");

                // click handler
                s.setClickHandler(function (e) {

                    // selection disabled
                    if (self.selection.statusEnabled === false) {
                        return false;
                    }

                    // context menu?
                    if (e.event.rightClick) {
                        // event
                        track({
                            type: "context_menu",
                            dest: "tree_flat",
                            data: { id: e.id },
                            text: "Right click on folder " + e.id + "in flat view"
                        });
                        // process
                        e.click("quiet");
                        e.context();
                    } else if (e.selectable === true) {
                        // viewing properties?
                        if (currentpath[0] === "configuration") {
                            // change back to folder's module
                            ox.UIController.setFolder({
                                folder: e.id,
                                module: "default"
                            });
                        } else if (e.id !== "-1") {
                            // normal click
                            // track
                            ox.api.folder.get({
                                folder: e.id,
                                success: function (data) {
                                    track({
                                        type: "click",
                                        dest: "tree_flat",
                                        data: { id: data.id, title: data.title },
                                        text: "Select " + data.module + " folder \"" + data.title + "\" (" + e.id + ")"
                                    });
                                }
                            });
                            // process
                            if (ox.UIController.getFolder() !== e.id) {
                                e.click("force");
                            } else {
                                e.click();
                            }
                        }
                    }

                    // prevent default
                    return false;
                });

                s.setContextMenuHandler(function (id, e) {
                    // show folder context menu
                    var menu = globalContextMenus.simpleFolderMenu;
                    menu.display(e.pageX, e.pageY, {
                        data: { id: id },
                        api: true,
                        simple: true,
                        module: module,
                        widget: self
                    });
                });

                this.addListener("widget:edited", function (e) {
                    // update folder
                    ox.api.folder.update({
                        folder: e.data.id,
                        data: { title: e.data.value },
                        success: function(data) {
                            ox.api.cache.folderFlat.remove(data.module);
                            // select
                            self.selection.select(data.id);
                            // continue
                            ox.util.call(e.cont);
                        },
                        error : function() {
                            // repaint
                            self.paint();
                            // select
                            self.selection.select(e.data.id);
                            // continue
                            ox.util.call(e.cont);
                        }
                    });
                    return false;
                });

                // track visibility
                this.addListener("widget:invisible", function () {
                    s.disable();
                });
                this.addListener("widget:visible", function () {
                    s.enable();
                });

                // set class now (IE7 issues; flickering scrollbar)
                $(this.dom.node).addClass("ox-sidepanel-view");

                this.onModify = function (data) {
                    // close invalid nodes
                    if (data.why === "folder.remove" || data.why === "folder.move") {
                        // unset current folder
                        ox.UIController.unsetFolder();
                    }

                    if (data.why === "folder.update.local") {
                        // repaint affected node only
                        var node = self.get(data.id);
                        if (node) {
                            node.repaint();
                        }
                    } else {
                        if (data.why === "folder.move") {
                            ox.UIController.setFolder({
                                folder: data.id,
                                module: "default",
                                success: function () {
                                    // reopen node
                                    if (self.setOpen) self.setOpen([ data.id ]);
                                    self.repaint();
                                }
                            });
                        } else {
                            // repaint full tree
                            self.repaint();
                        }
                    }
                };

                /**
                 * Context Menu via Left Click
                 */
                s.dispatcher.bind("selected", function (e) {
                    // add context button
                    if (e.node.find(".context-button").length === 0) {
                        var button = $.button({
                            title: "\u00a0\u00a0"
                        }).css({
                            position: "absolute",
                            right: "5px",
                            zIndex: 2
                        }).bind("click", function(event) {
                            s.triggerContextMenu(e.id,  event);
                            return false;
                        })
                        .addClass("context-button");
                        if (IE) {
                            button = $('<form onsubmit="return false"></form>')
                                .css("margin", "0")
                                .append(button);
                        }
                        e.node.prepend(button);
                    } else {
                        e.node.find(".context-button").show();
                    }
                });
                s.dispatcher.bind("deselected", function (e) {
                    // remove context button
                    e.node.find(".context-button").hide();
                });
            }
        });

        flatView.prototype.startEdit = function (id, success, abort) {
            // get
            if (this.folders[id] !== undefined) {

                var self = this;

                // unbind
                ox.api.folder.dispatcher.unbind("modify", this.onModify);

                if (success === undefined) {
                    success  = function (value) {
                        // trigger event
                        self.trigger(
                            "widget:edited",
                            {
                                node: self,
                                id: id,
                                value: value
                            },
                            function () {
                                // repaint (enables selection)
                                self.repaint();
                            }
                        );
                    };
                }

                if (abort === undefined) {
                    abort = function () {
                        // repaint (enables selection)
                        self.repaint();
                    };
                }

                this.selection.disable();

                var node = this.folders[id].node;
                ox.gui.util.inlineEdit.call(
                    this, node, node.data("title"), success, abort, "span"
                );
            }
        };

        var sections = {
            "private": {
                title: _("Private folders")
            },
            "public": {
                title: _("Public folders")
            },
            "shared": {
                title: _("Shared folders")
            }
        };

        // "Global address book"

        var paintFlatView =  function (cont) {
            var self = this;

            // unbind
            ox.api.folder.dispatcher.unbind("modify", this.onModify);

            function addDnD(folder, node, out) {
                if (self.dndSource) {
                    self.dndSource(folder, function (type, callback) {
                        out.dndSource = registerSource(node, type, callback,
                            null, null, foldertreedisable, defaultdisabledremove);
                    });
                }
                if (self.dndTarget) {
                    self.dndTarget(folder, function (callbacks) {
                        var cbs = {};
                        for (var i in callbacks) {
                            cbs[i] = (function (cb) {
                                return function (a, b, c, d, e, f) {
                                    legacyOut(a, b, c, d, null, f);
                                    cb(a, b, c, d, e, f);
                                };
                            })(callbacks[i]);
                        }
                        out.dndTarget = registerTarget(node, cbs, null, null,
                            legacyIn, legacyIn, legacyOut, true);
                    });
                }
                function legacyIn(e, type, objects, position, targetNode) {
                    foldertreeenable(e, type, objects, position, targetNode, folder);
                    var mynode = e.currentTarget || e.srcElement || false;
                    if (!mynode.className.match(/dndOver/)) {
                        mynode.className += " dndOver";
                    }
                }
                function legacyOut(e, type, objects, position, OutOpen, dropNode) {
                    if (dropNode) {
                        dropNode.className = removeClass(dropNode.className, "dndOver");
                    } else if (e) {
                        var mynode = e.currentTarget || e.srcElement || false;
                        if (mynode) {
                            mynode.className = removeClass(mynode.className, "dndOver");
                        }
                    }
                }
            }

            var paintFolders = function (folders, node, section) {

                // sort by title, default folder always first
                var id = String(ox.api.config.get("folder."+ self.module));
                // sort (skip shared folder; have special order)
                if (section !== "shared") {
                    folders.sort(function (a, b) {
                        return a.id === id ? -1 : a.title.toLowerCase() > b.title.toLowerCase() ? +1 : -1;
                    });
                }
                // add owner
                var addOwner = function (folder, node) {
                    if (ox.api.folder.is("shared", folder)) {
                        var displayName = folder["com.openexchange.folderstorage.displayName"] || "";
                        node.append("<br/>");
                        node.append(
                            $("<span/>").css({ color: "#666", fontStyle: "italic" }).
                                text("(" + displayName + ")")
                        );
                    }
                };
                // loop
                var i = 0, $l = folders.length, folder, div, count = 0;
                for (i = 0; i < $l; i++) {
                    // get
                    folder = folders[i];
                    // read permisssion?
                    if (ox.api.folder.can("read", folder)) {
                        var img = $("<img/>").attr({ src: meta[self.module].icon }).
                        css({ backgroundRepeat: "no-repeat", position: "absolute", left: "3px" });
                        if (ox.api.folder.is("published", folder)) {// published folders
                            // add shared symbol to original
                            img.css({"background-image" : "url(" + img.attr("src") + ")" });
                            img.attr("src", ox.gui.themePath + "icons/16/shared_globe.png");
                        }

                        // add nodes
                        node.append(
                            // div
                            div = $("<div/>", { oxID: folder.id }).
                                data("title", _(folder.title || "")).
                                addClass("ox-sidepanel-flat-folder").
                                css({ position: "relative" }).
                            append(img).
                            // folder name
                            append(
                                $("<span/>").text(folder.title || "")
                            )
                        );
                        // add owner?
                        addOwner(folder, div);
                        // add to index
                        var out = self.folders[folder.id] = {
                            id: folder.id,
                            node: div
                        };
                        // support DND
                        addDnD(folder, div[0], out);
                        // count
                        count++;
                    }
                }
                return count;
            };

            setTimeout(function () {

                // clear
                for (var i in self.folders) {
                    f = self.folders[i];
                    if (f.dndSource) unregisterSource(f.node[0], f.dndSource);
                    if (f.dndTarget) unregisterSource(f.node[0], f.dndTarget);
                    delete self.folders[i];
                }

                $(self.dom.node).empty();

                // add spinner
                if (!ox.browser.IE) { // IE7 flicker issue
                    $(self.dom.node).addClass("busy-spinner-white");
                }

                // get visible folders via API (flat list)
                ox.api.folder.getAllVisible(self.module, function (data) {

                    // remove spinner
                    if (!ox.browser.IE) { // IE7 flicker issue
                        $(self.dom.node).removeClass("busy-spinner-white");
                    }

                    // add sections
                    for (var id in sections) {
                        $(self.dom.node).append(
                            self.dom[id] = $("<div/>").addClass("ox-sidepanel-section").append(
                                $("<div/>").addClass("ox-sidepanel-section-title").append(
                                    addTranslated(meta[self.module].sections[id])
                                )
                            )
                        );
                        // paint folders
                        var count = paintFolders(data[id] || [], self.dom[id], id);
                        // remove section?
                        if (count === 0) {
                            self.dom[id].remove();
                        }
                    }

                    // enable selection
                    self.selection.enable();
                    self.selection.update();
                    self.selection.clickQuiet(ox.UIController.getFolder());

                    // bind to folder api
                    ox.api.folder.dispatcher.bind("modify", self.onModify, window);

                    // continuation
                    ox.util.call(cont);
                });

            }, 10);
        };

        // No source, since dragging folders without hierarchy makes no sense
        flatView.prototype.dndTarget = ox.gui.initSidepanel.dndTarget;

        // calendar
        v.calendar = new flatView(paintFlatView, "ox-sidepanel-calendar", "calendar");

        // contacts
        v.contacts = new flatView(paintFlatView, "ox-sidepanel-contacts", "contacts");

        // tasks
        v.tasks = new flatView(paintFlatView, "ox-sidepanel-tasks", "tasks");

    }());


    /*
     * Module title
     */
    var title = ox.widgets.sidepanel.title = new ox.gui.Container("ox-sidebar-title").
    setLayout(new ox.gui.BorderLayout()).
    css({ height: "26px" }).
    setLayoutParam({ position: "top", height: 27 }).
    addCSSClass("ox-sidepanel-title topheader-color").
    addTo(sidepanel).
    add(
        new ox.gui.Custom(
            function () {
                $(this.dom.node).empty().
                    append(addTranslated(this.title));
            },
            "ox-sidepanel-title"
        ).
        setLayoutParam({ position: "center" }).
        css({ marginLeft: "6px" }).
        addListener("widget:viewchanged", view, function (e) {
            // make sure folder & config are shown
            $("#folder_tree").show();
            // change title
            var match = e.data.id.match(/(\w+)$/);
            this.title = meta[match[1]].title;
            if (newFolderIcon !== null) {
                newFolderIcon.setVisible(match[1] !== "configuration");
            }
            this.paint();
        })
    ).add(
            new ox.gui.Image().
            css({ cursor: "pointer" }).
            addCSSClass("sp-toggle-img").
            setLayoutParam({ position: "right", width: 13 }).
            setSrc(getFullImgSrc("img/arrows/arrow_double_white_left.png")).
            setTitle(_("Collapse sidepanel")).
            addListener("click", function (e) {
                ox.widgets.sidepanel.toggle();
                ox.widgets.sidepanel.userToggle = true;
            })
    );

    // add new folder image only if required
    if (addRootFolder) {
        title.insert(newFolderIcon, 0);
    }

    // add view to sidepanel
    sidepanel.add(view);

    /*
     * Module views
     */
    v.portal = new ox.gui.Widget("ox-sidepanel-portal").addTo(view);
    v.mail.addTo(view);
    v.calendar.addTo(view);
    v.contacts.addTo(view);
    v.tasks.addTo(view);
    v.infostore.addTo(view);
    v.folder = new ox.gui.Container("ox-sidepanel-foldertree").addTo(view).
        setLayout(new ox.gui.BorderLayout());


    //--- quota

    var quotaContainer = { mail: [], filestore: [] }, quotaReq = [];

    // store to widgets container for public access
    if (!ox.widgets.sidepanel.quota) {
        ox.widgets.sidepanel.quota = {};
    }

    // Since all bars are needed twice - in the simple and in the expert view -
    // the bars are created by functions.

    function createMailBars(container, qContainer) {
        var bar = ox.widgets.sidepanel.quota.mail = new ox.gui.Bar({ textFormat: function(used, max) {
            return bytesToString(used) + " / " + bytesToString(max);
        }, title: function(used, max) {
            return _("E-Mail") + " " + bytesToString(used) + " / " +
                   bytesToString(max);
        }, description: _("E-Mail") }).
            setLayoutParam({ position: "bottom", height: 18 }).hide();
        container.add(bar);

        var bar2 = ox.widgets.sidepanel.quota.mail_count = new ox.gui.Bar({ textFormat: "fraction", description: _("E-Mail") }).
            setLayoutParam({ position: "bottom", height: 18 }).hide();
        container.add(bar2);

        qContainer.push({ id: "mail_storage",
            field_map: { usage:"use", quota:"quota" }, bar: bar });
        qContainer.push({ id: "mail_count",
            field_map: { usage:"countuse", quota:"countquota" }, bar: bar2 });
    }

    function createInfoStoreBars(container, qContainer) {
        var bar = ox.widgets.sidepanel.quota.infostore = new ox.gui.Bar({ textFormat: function(used, max) {
            return bytesToString(used) + " / " + bytesToString(max);
        }, title: function(used, max) {
            return _("Infostore") + " " + bytesToString(used) + " / " +
                   bytesToString(max);
        }, description: _("Infostore") }).
            setLayoutParam({ position: "bottom", height: 18 }).hide();
        container.add(bar);

        qContainer.push({ id: "filestore",
            field_map: { usage:"use", quota:"quota" }, bar: bar });
    }

    // register quota bars for available modules
    if (ox.api.config.get("modules.infostore.module")) {
        createInfoStoreBars(v.infostore, quotaContainer.filestore);
        createInfoStoreBars(v.folder, quotaContainer.filestore);
    	quotaReq.push({ module:"quota", action: "filestore" });
    }


    if (ox.api.config.get("modules.mail.module")) {
        createMailBars(v.mail, quotaContainer.mail);
        createMailBars(v.folder, quotaContainer.mail);
        quotaReq.push({ module:"quota", action: "mail" });
    }

    // fetches quota from the back-end
    function checkQuota() {
    	// helper to remove quota object with given ID from container
    	function removeFromContainer(id) {
    		for (var i in quotaContainer) {
                for (var ia in quotaContainer[i]) {
                    if (quotaContainer[i][ia].id == id) {
                    	if (quotaContainer[i][ia].bar) {
                    		var parent = quotaContainer[i][ia].bar.parent;
                    		quotaContainer[i][ia].bar.destroy();
                    		//ox.widgets.sidepanelViews.mail.validate();
                    		parent.validate();
                    	}
                        delete(quotaContainer[i][ia]);
                    }
                }
            }
    	}

    	// fire put
    	ox.api.http.PUT({
            module: "multiple",
            data: quotaReq,
            appendColumns: false,
            success: function (data) {
    			// now its gonna be tricky
    			// iterate through server results
	    		for (var i=0; i < data.length; i++) {
	    			// iterate through container
	                for (ia in quotaContainer[quotaReq[i].action]) {
	                	// getting quota object mapped to results
	                	var quota = quotaContainer[quotaReq[i].action][ia];
	                	if (!quota) continue;
	                	// no results for this request? remove it!
	                	if (data[i] == undefined || data[i].data == undefined) {
	             		   removeFromContainer(quota.id);
	                 	   continue;
	                    }
	             	   	// getting quota information (count and usage)
	                    var tQuota = data[i].data[quota.field_map.quota];
	                    var tUsage = data[i].data[quota.field_map.usage];
	                    // no results for this request? remove it!
	                    if (tQuota == undefined || tQuota == -1024 || tQuota == -1) {
	                    	removeFromContainer(quota.id);
	                        continue;
	                    }
	                    // store quota information to container
	                    quota.data = { quota: tQuota, usage: tUsage };

	                    // paint
	                    quota.bar.setMax(tQuota);
	                    quota.bar.progress(tUsage);
	                    if (quota.bar.isVisible() == false) {
	                        quota.bar.show();
	                    }
	                }
	            }
        	}
    	});
    };
    // receive information the first time
    checkQuota();
    // update quota information on refresh
    register("OX_Refresh", checkQuota);
    // register to empty folder
    register("OX_Folder_Cleared", function(folder_id) {
        // user empty a folder, this might be the either the trash
        // or user activated direct delete. if so we need to force
        // a quota update
        if (ox.api.config.get("mail.deletemail") === true
                || ox.api.config.get("mail.folder.trash") === folder_id) {
            checkQuota();
        }
    });

    //--- mini-cal

    var minical = ox.widgets.miniCalendar = new ox.gui.Widget("ox-sidepanel-view-minical");

    minical.expanded = ox.api.config.get("gui.global.minicalendar.expanded", false);

    minical.hide = function() {
        this.setVisible(false);
        this.layout.height = this.getHeight();
        this.invalidate();
        ox.widgets.sidepanel.validate();
    };

    minical.getHeight = function () {
        return this.statusVisible ? (this.expanded ? (13 * pxPerEm + 6) : (2 * pxPerEm + 4)) : 0;
    };

    minical.show = function() {
        // disabled means not visible at all
        if (this.isEnabled() === false) {
            return;
        }
        this.setVisible(true);
        this.layout.height = this.getHeight();
        this.invalidate();
        ox.widgets.sidepanel.validate();
    };

    minical.refreshState = function () {
        this.toggle(ox.api.config.get("gui.global.minicalendar.expanded", false));
    };

    minical.toggle = function (state) {
        // disabled means not visible at all
        if (this.isEnabled() === false) {
            return;
        }
        this.expanded = !this.expanded;
        if (typeof state == "boolean") {
            this.expanded = state;
        }

        // make changes permanent
        if (this.expanded === false || ox.UIController.getModule() !== "calendar") {
            ox.api.config.set("gui.global.minicalendar.expanded", this.expanded);
        }

        $("img.mini-toggle", this.dom.node)
            .attr("src", getFullImgSrc("img/" + (this.expanded ? "minus.gif" : "plus.gif")));

        this.layout.height = this.getHeight();
        this.invalidate();
        ox.widgets.sidepanel.validate();
    };

    minical.paint = function () {

        var Self = this;
        $(this.dom.node).empty().addClass("oxStretch");

        var content = $("<div/>").
            css({
                height: "11.3em",
                width: "18em",
                overflow: "hidden",
                position: "absolute"
            }).
            append($("<div/>").attr("id", "mini-content").addClass("calminimainCSS"));

        var header = $("<table/>")
            .addClass("headercontenttable border-color-design")
            .css({
                height: "2em",
                backgroundImage: "url('" + getFullImgSrc("img/toolbar/toolbar30.png") + "')"
            })
            .append(
                $("<tr/>").append(
                    $("<td/>").css({ width: "50%" }).text(" ")
                )
                .append(
                    $("<td/>").attr("align", "right").css({ cursor: "pointer", width: "10px" }).
                    append($("<img/>").attr("src", getFullImgSrc("img/arrows/arrow_darkgrey_left.gif"))
                    ).bind("click", function() {
                        oMiniCalendar.back(1);
                    })
                )
                .append(
                    $("<td/>").attr("align", "center").append(
                        $("<div/>").addClass("mini-month").attr("id", "mini-month")
                    )
                )
                .append(
                    $("<td/>").attr("align", "left").css({ cursor: "pointer", width: "10px" }).
                    append($("<img/>").attr("src", getFullImgSrc("img/arrows/arrow_darkgrey_right.gif"))
                    ).bind("click", function() {
                        oMiniCalendar.next(1);
                    })
                )
                .append(
                    $("<td/>").css({ width: "50%"}).attr("align", "left").append(
                        $("<div/>").addClass("mini-year").attr("id", "mini-year")
                    )
                ).append(
                    $("<td/>").css({ width: "20px"}).attr("align", "right").append(
                        $("<img/>").addClass("mini-toggle").
                        attr("src", getFullImgSrc("img/minus.gif")).
                        css({ cursor: "pointer" }).
                        click(function() {
                            ox.widgets.miniCalendar.userToggle = true;
                            ox.widgets.miniCalendar.toggle();
                        })
                    )
                )
            );

        $(this.dom.node).append(header);
        $(this.dom.node).append(content);

        oMiniCalendar = new cAnimMiniCalendar(
            fn_minicalendar_action, null,
            [ $("#mini-content", content)[0], $(".mini-month", header)[0], $(".mini-year", header)[0]]
        );
        b3xViewLoaded = false;
    };

    function fn_minicalendar_action() {

        oMiniCalendar.setSelected(this.id);
        activeDay=oMiniCalendar.selectedDate;
        activeMonth=oMiniCalendar.selectedMonth;
        activeYear=oMiniCalendar.selectedYear;

        // notify
        track({
            type: "click",
            dest: "mini_calender",
            data: { year: activeYear, month: activeMonth, day: activeDay },
            text: "Select date in mini calendar " + activeYear + "-" + activeMonth + "-" + activeDay
        });
        triggerEvent("OX_Mini_Calendar_Date_Picked", activeYear, activeMonth, activeDay);

        if(currentpath[0] == "calendar") {
            if (currentpath[1] != "team" && currentpath[2] == "month") {
                // this routine is not required in team view; might do too much in other views anyway...
                loadMonthView();
            }
            ox.UIController.setModule({
                module: "calendar",
                view: currentpath.join("/"),
                folder:"auto",
                force: "module"
            });
        } else if (configContainsKey("folder.calendar")) {
           ox.UIController.setModule({
               module: "calendar",
               folder:"auto",
               view:"calendar/calendar/day",
               force: "module"
           });
        }
    }

    function updateMiniCalData() {
        oMiniCalendar.update();
    }

    register("OX_Refresh_Mini_Calendar", updateMiniCalData);
    register("OX_Refresh", updateMiniCalData);

    minical
        .css({
            background: "white",
            border: "1px solid",
            borderWidth: "1px 0px 0px 0px"
        })
        .addCSSClass("border-color-design")
        .setLayoutParam({ position: "bottom", height: (13 * pxPerEm + 6) })
        .addTo(sidepanel);
};


// Two functions shared between ox.gui.FolderTree and ox.gui.TypedFolderTree
ox.gui.initSidepanel.dndSource = function(data, cont) {
    cont("folder/" + data.type, constant(data.id));
};
ox.gui.initSidepanel.dndTarget = function(subfolder, cont) {
    function folderHandler(e, t, data, p, n, d) {
        ox.api.folder.get({
            folder: data,
            success: function(data) {
                ox.api.folder.move(data, subfolder);
            }
        });
    }
    var tObj = {};
    if (   (subfolder.own_rights & 127) >= 4 // folder permissions >= create subfolders
        || subfolder.module == 'mail')
    {
        if (subfolder.type == 5) {
            if (subfolder.id == 1) {
                tObj["folder/1"] = folderHandler;
            } else if (subfolder.id in { 2: 1, 10: 1, 15: 1 }) {
                tObj["folder/2"] = folderHandler;
            } else {
                tObj["folder/" + subfolder.type] = folderHandler;
            }
        } else {
            if ( (subfolder.type == 7 || subfolder.type == 1) && configGetKey("modules.folder.tree") == 1) {
                tObj["folder/1"] = tObj["folder/7"] = folderHandler;
            } else {
                tObj["folder/" + subfolder.type] = folderHandler;
            }
        }
    }

    /*
     * check if user has writeObject permissions and if it's not a
     * shared folder of module calendar, tasks or contacts. move operations
     * into a shared folder of the named modules will be declined by the server
     */
    var tmp_module = subfolder.module;
    if (   ox.api.folder.can("write", subfolder) === true // folder permissions >= create objects
        && !(subfolder.type == 3 && tmp_module == "tasks"))
    {
        /*
         * user has writeObject permissions on this folder, register event
         * which handles move operations
         */
        tObj[subfolder.module] = function(e,dragType,data,mouseposition,targetNode,dropNode) {
            if (subfolder.type == 2) {
                for (var i in data.currentObjects) {
                    if (data.currentObjects[i].private_flag) {
                        return;
                    }
                }
            }
            data.endMoving(subfolder.id);
        };
    } else {
        /*
         * user doesn't have writeObject permissions or it's a shared folder
         * so register dummy event
         */
        tObj[subfolder.module] = $.noop;
    }

    if (subfolder.module != "contacts") {
        tObj.contacts = $.noop;
    }
    if (subfolder.module != "tasks") {
        tObj.tasks = $.noop;
    }
    if (subfolder.module != "calendar") {
        tObj.calendar = $.noop;
    }
    if (subfolder.module != "infostore") {
        tObj.infostore = $.noop;
    }
    if (subfolder.module != "mail") {
        tObj.mail = $.noop;
    }
    if (subfolder.module == "contacts") {
        tObj.mailaddress = function (e, t, objects) {
            createNewContactfromMail(objects, subfolder.id);
        };
    } else {
        tObjmailaddress = $.noop;
    }
    cont(tObj);
};
