/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Document;

/**
 * {@link Tools}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class Tools {

    private static final TransformerFactory tf = TransformerFactory.newInstance();

    private Tools() {
        super();
    }

    /**
     * Check if project build contains errors or is in progress.
     * @param statuses the package statuses.
     * @return <code>true</code> if the project has been built.
     */
    public static boolean isProjectSuccessfulBuilt(final PackageStatus[] statuses) {
        for (final PackageStatus status : statuses) {
            final Code code = status.getCode();
            switch (code) {
            case SUCCEEDED:
            case DISABLED:
            case EXCLUDED:
                break;
            default:
                return false;
            }
        }
        return true;
    }

    /**
     * Returns the complete project status as a string.
     * @param statuses the package statuses.
     * @param onlyFailed 
     * @return a string containing all statuses.
     */
    public static String getProjectStatus(final PackageStatus[] statuses, final boolean onlyFailed) {
        final StringBuilder ret = new StringBuilder();
        for (final PackageStatus status : statuses) {
            if (onlyFailed && Code.SUCCEEDED == status.getCode()) {
                continue;
            }
            ret.append(status.toString());
            ret.append('\n');
        }
        return ret.toString();
    }
    
    /**
     * Generates a comma separated list of the projects defined by status
     * 
     * @param statuses the package statuses.
     * @return a string containing all statuses.
     */
    public static String getProjectsByStatus(final PackageStatus[] statuses, final Code code) {
        final StringBuilder ret = new StringBuilder();
        for (final PackageStatus status : statuses) {
            if (code != status.getCode()) {
                continue;
            }
            ret.append(status.getPackageName());
            ret.append(',');
        }
        return ret.toString();
    }

    /**
     * Checks if the project is still building packages.
     * @param statuses the package statuses
     * @return <code>true</code> if the project is still building.
     */
    public static boolean isProjectBuilding(final PackageStatus[] statuses) {
        for (final PackageStatus status : statuses) {
            final Code code = status.getCode();
            switch (code) {
            case BUILDING:
            case SCHEDULED:
            case FINISHED:
            case BLOCKED:
            case DISPATCHING:
            case UNKNOWN:
                return true;
            default:
            }
        }
        return false;
    }

    public static boolean isProjectBuildingByStatus(final Result[] results) {
        boolean finished = true;
        for (final Result result : results) {
            finished = finished && !isProjectBuilding(result.getStatuses());
        }
        return !finished;
    }
    
    public static boolean isProjectBuilding(final Result[] results) {
        boolean finished = true;
        for (final Result result : results) {
            finished = finished && State.PUBLISHED == result.getState();
        }
        return !finished;
    }

    /**
     * Checks if the build of some package failed.
     * @param statuses the package statuses
     * @return <code>true</code> if some package built failed.
     */
    public static boolean somePackageFailed(final PackageStatus[] statuses) {
        for (final PackageStatus status : statuses) {
            if (Code.FAILED == status.getCode()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if the build of some package failed.
     * @param results the package results
     * @return <code>true</code> if some package built failed.
     */
    public static boolean somePackageFailed(final Result[] results) {
        for (final Result result : results) {
            if (somePackageFailed(result.getStatuses())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if the build of some package failed and returns the first failed.
     * @param statuses the package statuses
     * @return <code>null</code> if no failed package found else the status, which describes the failed package
     */
    public static PackageStatus getFirstFailedPackage(final PackageStatus[] statuses) {
        for (final PackageStatus status : statuses) {
            if (Code.FAILED == status.getCode()) {
                return status;
            }
        }
        return null;
    }
    
    /**
     * Checks if the build of some package failed and returns the first result containing this failed.
     * @param results the package results
     * @return <code>null</code> if no failed package found else the result, which contains the failed package
     */
    public static Result getFirstFailedResult(final Result[] results) {
        for (final Result result : results) {
            if (somePackageFailed(result.getStatuses())) {
                return result;
            }
        }
        return null;
    }
    
    public static Result extract(final Result[] results, final String name) {
        for (final Result result : results) {
            if (result.getRepository().getName().equals(name)) {
                return result;
            }
        }
        return null;
    }

    public static String transform(final Document document) throws BuildServiceException {
        final StringWriter sw = new StringWriter();
        try {
            final Transformer transformer = tf.newTransformer();
            transformer.transform(new DOMSource(document), new StreamResult(sw));
        } catch (TransformerConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (TransformerException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        return sw.toString();
    }

    public static void transfer(final InputStream in, final OutputStream out) throws IOException {
        final byte[] buffer = new byte[512];
        int length = -1;
        while ((length = in.read(buffer)) != -1) {
            out.write(buffer, 0, length);
        }
        out.flush();
    }

}
