/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import static com.openexchange.ant.Tools.readFile;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.text.ParseException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import com.openexchange.Definitions;
import com.openexchange.ant.Tools;
import com.openexchange.data.SpecChangeLogEntry;
import com.openexchange.data.ChangeLogSorterForSpec;

/**
 * Updates a .spec file from the corresponding debian/changelog.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class UpdateSpec extends Task {

    private File changelog, spec;
    private int revision;
    private List<SpecChangeLogEntry> changelogs = new LinkedList<SpecChangeLogEntry>();

    public UpdateSpec() {
        super();
    }

    public void setChangeLog(File changelog) {
        this.changelog = changelog;
    }

    public void setSpec(File spec) {
        this.spec = spec;
    }

    @Override
    public void execute() throws BuildException {
        checks();
        parseChangeLog(Tools.readFile(changelog));
        order();
        String newContent = updateSpecFile(readFile(spec));
        writeFile(newContent);
    }

    private void checks() {
        if (null == changelog) {
            throw new BuildException("Please define path to debian/changelog file.");
        }
        if (null == spec) {
            throw new BuildException("Please define path to .spec file.");
        }
        if (!changelog.exists()) {
            throw new BuildException("debian/changelog file does not exist here: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.exists()) {
            throw new BuildException(".spec file does not exist here: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!changelog.isFile()) {
            throw new BuildException("debian/changelog file is not a file: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.isFile()) {
            throw new BuildException(".spec file is not a file: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!changelog.canRead()) {
            throw new BuildException("Can not read debian/changelog file: \"" + changelog.getAbsolutePath() + "\".");
        }
        if (!spec.canRead()) {
            throw new BuildException("Can not read .spec file: \"" + spec.getAbsolutePath() + "\".");
        }
        if (!spec.canWrite()) {
            throw new BuildException("Can not write .spec file: \"" + spec.getAbsolutePath() + "\".");
        }
    }

    public void parseChangeLog(String changelogContent) {
        Matcher matcher = Definitions.DEBIAN_CHANGELOG_PATTERN.matcher(changelogContent);
        boolean isFirst = true;
        while (matcher.find()) {
            String version = matcher.group(2);
            Matcher matcher2 = Definitions.REVISION_PATTERN.matcher(version);
            if (!matcher2.matches()) {
                throw new BuildException("Can not parse version: \"" + version + "\".");
            }
            int currentRevision;
            try {
                currentRevision = Integer.parseInt(matcher2.group(1));
            } catch (NumberFormatException e) {
                throw new BuildException("Can not parse revision from debian/changelog \"" + version + "\".", e);
            }
            if (isFirst) {
                revision = currentRevision;
                isFirst = false;
            }
            try {
                SpecChangeLogEntry entry = new SpecChangeLogEntry(Definitions.DEBIAN_DF.parse(matcher.group(6)), matcher.group(5), matcher.group(4));
                changelogs.add(entry);
            } catch (ParseException e) {
                throw new BuildException(e.getMessage(), e);
            }
        }
    }

    private void order() {
        Collections.sort(changelogs, new ChangeLogSorterForSpec());
    }

    private String updateSpecFile(String specfileContent) {
        String tmp = Pattern.compile("^(%define\\s+ox_release\\s+)\\d+$", Pattern.MULTILINE).matcher(specfileContent).replaceFirst("$1" + Integer.toString(revision));
        Matcher matcher = Pattern.compile("(^%changelog$).*", Pattern.MULTILINE + Pattern.DOTALL).matcher(tmp);
        final StringBuilder sb;
        if (matcher.find()) {
            tmp = matcher.replaceFirst("$1");
            sb = new StringBuilder(tmp);
            sb.append('\n');
        } else {
            sb = new StringBuilder(tmp);
            sb.append("\n%changelog\n");
        }
        for (SpecChangeLogEntry entry : changelogs) {
            sb.append("* ");
            sb.append(Definitions.SPECFILE_DF.format(entry.getDate()));
            sb.append(' ');
            sb.append(entry.getAuthor());
            sb.append('\n');
            sb.append(entry.getComment());
            sb.append('\n');
        }
        return sb.toString();
    }

    private void writeFile(String specfileContent) {
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(spec), "UTF-8"));
            bw.write(specfileContent);
        } catch (IOException e) {
            throw new BuildException(e.getMessage(), e);
        } finally {
            if (null != bw) {
                try {
                    bw.close();
                } catch (IOException e) {
                    throw new BuildException(e.getMessage(), e);
                }
            }
        }
    }
}
