/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2011 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.user.copy.internal.reminder;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import com.openexchange.groupware.container.Appointment;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.impl.IDGenerator;
import com.openexchange.groupware.ldap.User;
import com.openexchange.groupware.reminder.ReminderException;
import com.openexchange.groupware.reminder.ReminderObject;
import com.openexchange.groupware.reminder.ReminderStorage;
import com.openexchange.groupware.tasks.Task;
import com.openexchange.user.copy.CopyUserTaskService;
import com.openexchange.user.copy.ObjectMapping;
import com.openexchange.user.copy.UserCopyException;
import com.openexchange.user.copy.UserCopyExceptionCodes;
import com.openexchange.user.copy.internal.CopyTools;
import com.openexchange.user.copy.internal.IntegerMapping;
import com.openexchange.user.copy.internal.calendar.CalendarCopyTask;
import com.openexchange.user.copy.internal.connection.ConnectionFetcherTask;
import com.openexchange.user.copy.internal.contact.ContactCopyTask;
import com.openexchange.user.copy.internal.context.ContextLoadTask;
import com.openexchange.user.copy.internal.folder.FolderCopyTask;
import com.openexchange.user.copy.internal.tasks.TaskCopyTask;
import com.openexchange.user.copy.internal.user.UserCopyTask;

/**
 * {@link ReminderCopyTask}
 * 
 * @author <a href="mailto:jan.bauerdick@open-xchange.com">Jan Bauerdick</a>
 */
public class ReminderCopyTask implements CopyUserTaskService {

    /**
     * Initializes a new {@link ReminderCopyTask}.
     */
    public ReminderCopyTask() {
        super();
    }

    public String[] getAlreadyCopied() {
        return new String[] {
            UserCopyTask.class.getName(),
            ContextLoadTask.class.getName(),
            ConnectionFetcherTask.class.getName(),
            FolderCopyTask.class.getName(),
            CalendarCopyTask.class.getName(),
            ContactCopyTask.class.getName(),
            TaskCopyTask.class.getName()
        };
    }

    public String getObjectName() {
        return com.openexchange.groupware.reminder.ReminderObject.class.getName();
    }

    public IntegerMapping copyUser(final Map<String, ObjectMapping<?>> copied) throws UserCopyException {
        final CopyTools copyTools = new CopyTools(copied);

        final Context srcCtx = copyTools.getSourceContext();
        final Context dstCtx = copyTools.getDestinationContext();

        final User srcUser = copyTools.getSourceUser();
        final User dstUser = copyTools.getDestinationUser();

        final Connection srcCon = copyTools.getSourceConnection();
        final Connection dstCon = copyTools.getDestinationConnection();

        final ObjectMapping<Integer> appointmentMapping = copyTools.checkAndExtractGenericMapping(Appointment.class.getName());
        final ObjectMapping<Integer> taskMapping = copyTools.checkAndExtractGenericMapping(Task.class.getName());

        final Map<Integer, ReminderObject> reminders = loadRemindersFromDB(srcCtx, srcCon, srcUser);
        exchangeIds(reminders, taskMapping, appointmentMapping, dstUser.getId(), dstCtx, dstCon);
        writeRemindersToDatabase(dstCon, dstCtx, srcCtx, dstUser.getId(), reminders);

        final IntegerMapping mapping = new IntegerMapping();
        for (final int reminderId : reminders.keySet()) {
            final ReminderObject reminder = reminders.get(reminderId);
            mapping.addMapping(reminderId, reminder.getObjectId());
        }

        return mapping;
    }

    public void done(final Map<String, ObjectMapping<?>> copied, final boolean failed) {

    }

    private Map<Integer, ReminderObject> loadRemindersFromDB(final Context ctx, final Connection con, final User user) throws UserCopyException {
        final Map<Integer, ReminderObject> reminders = new HashMap<Integer, ReminderObject>();
        final Date end = new Date(Long.MAX_VALUE);
        try {
            final ReminderObject[] reminderArray = ReminderStorage.getInstance().selectReminder(ctx, con, user, end);
            for (int i = 0; i < reminderArray.length; i++) {
                reminders.put(reminderArray[i].getObjectId(), reminderArray[i]);
            }
        } catch (final ReminderException e) {
            throw UserCopyExceptionCodes.UNKNOWN_PROBLEM.create(e);
        }
        return reminders;
    }

    private void writeRemindersToDatabase(final Connection dstCon, final Context dstCtx, final Context srcCtx, final int dstUserId, final Map<Integer, ReminderObject> reminders) throws UserCopyException {
        for (final int reminderId : reminders.keySet()) {
            final ReminderObject reminder = reminders.get(reminderId);
            try {
                ReminderStorage.getInstance().writeReminder(dstCon, dstCtx.getContextId(), reminder);
            } catch (final ReminderException e) {
                throw UserCopyExceptionCodes.UNKNOWN_PROBLEM.create(e);
            }
        }
    }

    private void exchangeIds(final Map<Integer, ReminderObject> reminders, final ObjectMapping<Integer> taskMapping, final ObjectMapping<Integer> appointmentMapping, final int userId, final Context ctx, final Connection con) throws UserCopyException {
        try {
            for (final int reminderId : reminders.keySet()) {
                final int newReminderId = IDGenerator.getId(ctx, com.openexchange.groupware.Types.REMINDER, con);
                final ReminderObject reminder = reminders.get(reminderId);
                reminder.setObjectId(newReminderId);
                reminder.setUser(userId);
                final int targetId = reminder.getTargetId();
                int newTargetId = targetId;
                if (reminder.getModule() == com.openexchange.groupware.Types.APPOINTMENT) {
                    for (final int copy : appointmentMapping.getSourceKeys()) {
                        if (appointmentMapping.getDestination(copy) == targetId) {
                            newTargetId = copy;
                        }
                    }
                } else if (reminder.getModule() == com.openexchange.groupware.Types.TASK) {
                    for (final int copy : taskMapping.getSourceKeys()) {
                        if (taskMapping.getDestination(copy) == targetId) {
                            newTargetId = copy;
                        }
                    }
                }
                reminder.setTargetId(newTargetId);
            }
        } catch (final SQLException e) {
            throw UserCopyExceptionCodes.SQL_PROBLEM.create(e);
        }
    }
}
