/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.authentication.kerberos.impl;

import static com.openexchange.authentication.LoginExceptionCodes.INVALID_CREDENTIALS;
import java.util.List;
import java.util.Map;
import com.openexchange.ajax.fields.Header;
import com.openexchange.authentication.Authenticated;
import com.openexchange.authentication.AuthenticationService;
import com.openexchange.authentication.AutoLoginAuthenticationService;
import com.openexchange.authentication.LoginException;
import com.openexchange.authentication.LoginExceptionCodes;
import com.openexchange.authentication.LoginInfo;
import com.openexchange.context.ContextService;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.contexts.impl.ContextException;
import com.openexchange.groupware.contexts.impl.ContextStorage;
import com.openexchange.groupware.ldap.LdapException;
import com.openexchange.groupware.ldap.UserException;
import com.openexchange.kerberos.ClientPrincipal;
import com.openexchange.kerberos.KerberosException;
import com.openexchange.kerberos.KerberosService;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.servlet.http.Authorization;
import com.openexchange.user.UserService;

/**
 * Implementes the authorization using a Kerberos backend. Two different inputs are possible:
 * - Login request contains a Negotiate authorization header
 * - Login and password are used to authorize the user against the Kerberos server.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class KerberosAuthentication implements AuthenticationService, AutoLoginAuthenticationService {

    private KerberosService kerberosService;
    private final ContextService contextService;
    private final UserService userService;

    public KerberosAuthentication(KerberosService kerberosService, ContextService contextService, UserService userService) {
        super();
        this.kerberosService = kerberosService;
        this.contextService = contextService;
        this.userService = userService;
    }

    @Override
    public Authenticated handleLoginInfo(final LoginInfo loginInfo) throws LoginException {
        final String authHeader = parseAuthheader(loginInfo);
        final ClientPrincipal principal;
        if (Authorization.checkForKerberosAuthorization(authHeader)) {
            final byte[] ticket = Base64.decode(authHeader.substring("Negotiate ".length()));
            try {
                principal = kerberosService.verifyAndDelegate(ticket);
            } catch (KerberosException e) {
                throw new LoginException(e);
            }
        } else {
            throw LoginExceptionCodes.UNKNOWN.create("Unknown authentication method.");
        }
        final String[] splitted = split(principal.getName());
        try {
            final int ctxId = contextService.getContextId(splitted[0]);
            if (ContextStorage.NOT_FOUND == ctxId) {
                throw INVALID_CREDENTIALS.create();
            }
            final Context ctx = contextService.getContext(ctxId);
            final int userId;
            try {
                userId = userService.getUserId(splitted[1], ctx);
            } catch (UserException e) {
                if (LdapException.Code.USER_NOT_FOUND.getDetailNumber() == e.getDetailNumber()) {
                    throw INVALID_CREDENTIALS.create();
                }
                throw new LoginException(e);
            }
            userService.getUser(userId, ctx);
        } catch (ContextException e) {
            throw new LoginException(e);
        } catch (UserException e) {
            throw new LoginException(e);
        }
        return new Authed(splitted[0], splitted[1], principal);
    }

    @Override
    public Authenticated handleAutoLoginInfo(LoginInfo loginInfo) throws LoginException {
        return new Redirect();
    }

    private static String parseAuthheader(final LoginInfo loginInfo) {
        String retval = null;
        Object tmp = loginInfo.getProperties().get("headers");
        if (tmp instanceof Map<?, ?>) {
            @SuppressWarnings("unchecked")
            Map<String, List<String>> headers = (Map<String, List<String>>) tmp;
            tmp = headers.get(Header.AUTH_HEADER);
            if (tmp instanceof List<?>) {
                @SuppressWarnings("unchecked")
                List<String> values = (List<String>) tmp;
                if (values.size() > 0) {
                    retval = values.get(0);
                }
            }
        }
        return retval;
    }

    /**
     * Splits user name and context.
     * @param loginInfo combined information seperated by an @ sign.
     * @return a string array with context and user name (in this order).
     */
    private static String[] split(final String loginInfo) {
        return split(loginInfo, '@');
    }

    /**
     * Splits user name and context.
     * @param loginInfo combined information seperated by an @ sign.
     * @param separator for spliting user name and context.
     * @return a string array with context and user name (in this order).
     * @throws LoginException if no seperator is found.
     */
    private static String[] split(final String loginInfo, final char separator) {
        final int pos = loginInfo.lastIndexOf(separator);
        final String[] splitted;
        if (-1 == pos) {
            splitted = new String[] { "defaultcontext", loginInfo };
        } else {
            splitted = new String[] { loginInfo.substring(pos + 1), loginInfo.substring(0, pos) };
        }
        return splitted;
    }
}
