/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.util;

/**
 * Class that represents versioning information. A version consists of multiple version
 * parts, which must all be integers >= 0. They are separated by '.'. The version may
 * be optionally appended with a description, delimited from the version information by
 * a ' '.
 * 
 * Version := \d+[\.\d+]+[\s.*]?
 * 
 * Note: For comparison between Version objects, the description is ignored, only the
 * version parts are used. The same is true for equals() and hashCode().
 * 
 * @author afe
 *
 */
public class Version implements Comparable<Version> {

	public static final Version UNKNOWN_VERSION = new Version();

	public static final int MAJOR = 0;
	public static final int MINOR = 1;
	public static final int REVISION = 2;

	/**
	 * Parses a version from a String. Any invalid Version parts are ignored by this method
	 * (replaced with 0). If the given version parameter is null, the UNKNOWN_VERSION is returned.
	 * 
	 * @param version
	 * @return
	 */
	public static Version fromString(String version) {
		return parse(version, false);
	}

	/**
	 * Parses a version from a String. If an invalid Version part is encountered by this method,
	 * this method throws an IllegalArgumentException. If version==null, a NullPointerException
	 * is thrown.
	 * 
	 * @param version
	 * @return
	 * @throws IllegalArgumentException
	 * @throws NullPointerException
	 */
	public static Version parse(String version) {
		return parse(version, true);
	}

	private static Version parse(String version, boolean throwError) {
		if (version == null && !throwError)
			return UNKNOWN_VERSION; // no version -> 0.0.0
		int descrIndex = version.indexOf(' ');
		String descr = null;
		if (descrIndex >= 0) {
			descr = version.substring(descrIndex + 1);
			version = version.substring(0, descrIndex);
		}
		String[] parts = version.split("\\.");
		if (throwError && parts.length < 2)
			throw new IllegalArgumentException("Version is missing required parts");
		int v[] = new int[parts.length];
		for (int i = 0; i < parts.length; i++) {
			try {
				v[i] = Integer.parseInt(parts[i]);
				if (v[i] < 0) {
					if (throwError)
						throw new IllegalArgumentException("Illegal negative part in version");
					v[i] = 0;
				}
			} catch (NumberFormatException e) {
				if (throwError)
					throw new IllegalArgumentException("Illegal part in version", e);
			}
		}
		return new Version(v, descr);
	}

	private final int[] _parts;
	private final String _description;

	public Version(int major, int minor, int revision) {
		this(major, minor, revision, null);
	}

	public Version(int major, int minor, int revision, String description) {
		_parts = new int[3];
		_parts[MAJOR] = major;
		_parts[MINOR] = minor;
		_parts[REVISION] = revision;
		_description = description;
	}

	public Version(int[] parts) {
		this(parts, null);
	}

	public Version(int[] parts, String description) {
		_parts = new int[parts.length];
		System.arraycopy(parts, 0, _parts, 0, parts.length);
		_description = description;
	}

	public Version() {
		_parts = new int[0];
		_description = "<Unknown>";
	}

	public int getVersionPart(int index) {
		if (index < 0)
			throw new IllegalArgumentException("Version part index must be >= 0");
		return (_parts.length > index) ? _parts[index] : 0;
	}

	/**
	 * Compares to another Version, using at most "limit" elements of the version information. This method
	 * returns 0 if these elements are equal, even if additional elements would differ. If "limit" is specified as
	 * <= 0, all elements will be compared.
	 * @param o Version to compare to
	 * @param limit limit of elements to compare (or unlimited if <= 0)
	 * @return < 0 if this is older, > 0 if this is newer, == 0 if equal
	 */
	public int compareTo(Version o, int limit) {
		if (o == null)
			throw new IllegalArgumentException("Could not compare Version with null!");

		int numberOfPartsToCompare = Math.max(_parts.length, o._parts.length);
		if (limit > 0 && limit < numberOfPartsToCompare)
			numberOfPartsToCompare = limit;

		for (int i = 0; i < numberOfPartsToCompare; i++) {
			int diff = getVersionPart(i) - o.getVersionPart(i);
			if (diff != 0)
				return diff;
		}
		return 0;
	}

	public int compareTo(Version o) {
		if (o == null)
			throw new IllegalArgumentException("Could not compare Version with null!");
		return compareTo(o, -1);
	}

	public boolean isNewer(Version o) {
		return compareTo(o) > 0;
	}

	public boolean isNewer(Version o, int limit) {
		return compareTo(o, limit) > 0;
	}

	public boolean isOlder(Version o) {
		return compareTo(o) < 0;
	}

	public boolean isOlder(Version o, int limit) {
		return compareTo(o, limit) < 0;
	}

	/**
	 * 
	 * @param o
	 * @param limit
	 * @return true if this Version matches the given Version in at least as many parts as provided by the limit
	 */
	public boolean matches(Version o, int limit) {
		return compareTo(o, limit) == 0;
	}

	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof Version))
			return false;
		return compareTo((Version) obj) == 0;
	}

	@Override
	public int hashCode() {
		int hashCode = 87;
		for (int i = 0; i < _parts.length; i++)
			hashCode = hashCode * 87 + _parts[i];
		return hashCode;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < _parts.length; i++) {
			if (sb.length() > 0)
				sb.append('.');
			sb.append(_parts[i]);
		}
		if (_description != null && _description.length() > 0)
			sb.append(' ').append(_description);
		return sb.toString();
	}

	public String getDescription() {
		return _description;
	}

	public int getMajor() {
		return getVersionPart(MAJOR);
	}

	public int getMinor() {
		return getVersionPart(MINOR);
	}

	public int getRevision() {
		return getVersionPart(REVISION);
	}
}
