/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.servlet;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;

import com.openexchange.usm.syncml.sync.SyncInformation;

/**
 * This class stores all information relevant for SyncML that should be available during a SyncML session.
 * Since some devices do not support session-cookies and therefore no HttpSessions, the SyncMLServlet
 * has its own mechanism to store temporarily the SyncMLSessionData associated with a SyncML session
 * (it will be cleared after at least 10 minutes of inactivity of the session).
 * 
 * @author afe
 *
 */
public class SyncMLSessionData {
	private long _lastAccess;
	private String _sessionID;
	private String _authentication;
	private Map<String, SyncInformation> _syncInformations;

	public SyncMLSessionData() {
		updateLastAccess();
	}

	public void updateLastAccess() {
		_lastAccess = System.currentTimeMillis();
	}

	public long getLastAccess() {
		return _lastAccess;
	}

	public String getAuthentication(String sessionID) {
		return (sessionID != null && sessionID.equals(_sessionID)) ? _authentication : null;
	}

	public void setAuthentication(String sessionID, String auth) {
		_sessionID = sessionID;
		_authentication = auth;
	}

	public List<SyncInformation> getAllSyncInformations() {
		if (_syncInformations == null)
			return Collections.emptyList();
		return new ArrayList<SyncInformation>(_syncInformations.values());
	}

	public SyncInformation getSyncInformation(String clientLocURI, String serverLocURI) {
		if (_syncInformations != null) {
			for (SyncInformation si : _syncInformations.values()) {
				if (si.getClientLocURI().equals(clientLocURI) && si.getServerLocURI().equals(serverLocURI))
					return si;
			}
		}
		return null;
	}

	public void storeSyncInformation(SyncInformation syncInformation) {
		if (_syncInformations == null)
			_syncInformations = new ConcurrentHashMap<String, SyncInformation>();
		_syncInformations.put(syncInformation.getOxFolderID(), syncInformation);
	}

	public SyncInformation removeSyncInformation(String oxFolderID) {
		return _syncInformations != null ? _syncInformations.remove(oxFolderID) : null;
	}

	@Override
	public String toString() {
		return "SyncMLSessionData(" + _lastAccess + ',' + _sessionID + ')';
	}

	public void checkForClearedSyncData(Log journal) {
		for (SyncInformation info : getAllSyncInformations()) {
			journal.info("SyncInformation not finished: " + info);
		}
	}
}
