/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.servlet;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.*;

import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.logging.Log;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlSerializer;

import com.openexchange.usm.api.contenttypes.ContentTypeManager;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.configuration.*;
import com.openexchange.usm.syncml.*;
import com.openexchange.usm.syncml.config.*;
import com.openexchange.usm.syncml.parser.XmlPullParserFactory;
import com.openexchange.usm.syncml.parser.XmlSerializerFactory;
import com.openexchange.usm.util.Toolkit;

/**
 * The SyncML Servlet handles SyncML requests from clients. 
 * It extends the HttpServlet class. 
 * 
 * @author ldo
 *
 */
public class SyncMLServlet extends HttpServlet {

	private static final String SYNC_ML_DEBUG_LOG = "SyncML_Debug.log";

	public static final int START_COMMAND_ID = 0;

	private static final boolean LOG_REQUEST_AND_OUTPUT = true;

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final transient SyncMLSessionInitializer _sessionInitializer = new SyncMLSessionInitializer(this);

	private final transient Map<String, SyncMLSessionData> _sessionData = new ConcurrentHashMap<String, SyncMLSessionData>();

	private transient SessionManager _sessionManager;

	private transient ContentTypeManager _contentTypeManager;

	private transient Log _journal;

	private transient HttpService _httpService;

	private transient String _servletAlias;

	private transient long _lastSessionDataCleanup = 0L;

	private transient int _sessionDataCleanupInterval;

	private transient boolean _formattedServerAnchors;

	private transient int _defaultMaxMessageSize;

	private transient boolean _redirectURLs;

	public void activate(SessionManager sessionManager, ContentTypeManager contentTypeManager, Log log,
			HttpService httpService, ConfigurationManager configurationManager) {

		_sessionManager = sessionManager;
		_contentTypeManager = contentTypeManager;
		_journal = log;
		_httpService = httpService;
		try {
			_servletAlias = configurationManager.getProperty(ConfigurationProperties.SYNCML_SERVLET_ALIAS_PROPERTY,
					ConfigurationProperties.SYNCML_SERVLET_ALIAS_DEFAULT, true);
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_ALIAS_FOR_SERVLET,
					ConfigurationProperties.SYNCML_SERVLET_ALIAS_PROPERTY, e1);
		}
		try {
			_sessionDataCleanupInterval = configurationManager.getProperty(
					ConfigurationProperties.SYNCML_SYNC_TIMEOUT_PROPERTY,
					ConfigurationProperties.SYNCML_SYNC_TIMEOUT_DEFAULT, false);
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_SYNC_TIMEOUT,
					ConfigurationProperties.SYNCML_SYNC_TIMEOUT_PROPERTY, e1);
		}
		try {
			_formattedServerAnchors = configurationManager.getProperty(
					ConfigurationProperties.SYNCML_FORMATTED_SERVER_ANCHORS_PROPERTY,
					ConfigurationProperties.SYNCML_FORMATTED_SERVER_ANCHORS_DEFAULT, false);
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_FORMATTED_SERVER_ANCHORS,
					ConfigurationProperties.SYNCML_FORMATTED_SERVER_ANCHORS_PROPERTY, e1);
		}
		try {
			_defaultMaxMessageSize = configurationManager.getProperty(
					ConfigurationProperties.SYNCML_DEFAULT_MAX_MESSAGE_SIZE_PROPERTY,
					ConfigurationProperties.SYNCML_DEFAULT_MAX_MESSAGE_SIZE_DEFAULT, false);
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_DEFAULT_MAX_MESSAGE_SIZE,
					ConfigurationProperties.SYNCML_DEFAULT_MAX_MESSAGE_SIZE_PROPERTY, e1);
		}
		try {
			boolean syncAllFolders = configurationManager.getProperty(
					ConfigurationProperties.SYNCML_SYNC_ALL_FOLDERS_PROPERTY,
					ConfigurationProperties.SYNCML_SYNC_ALL_FOLDERS_DEFAULT, false);
			FolderHierarchy.setImplementation(syncAllFolders ? new CompleteFolderHierarchy()
					: new SimpleFolderHierarchy());
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_SYNC_ALL_FOLDERS,
					ConfigurationProperties.SYNCML_SYNC_ALL_FOLDERS_PROPERTY, e1);
		}
		try {
			_redirectURLs = configurationManager.getProperty(ConfigurationProperties.SYNCML_REDIRECT_URLS_PROPERTY,
					ConfigurationProperties.SYNCML_REDIRECT_URLS_DEFAULT, true);
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowConfigurationError(SyncMLBundleErrorCodes.INVALID_REDIRECT_URLS,
					ConfigurationProperties.SYNCML_REDIRECT_URLS_PROPERTY, e1);
		}
		try {
			_httpService.registerServlet(_servletAlias, this, null, _httpService.createDefaultHttpContext());
			_journal.info("SyncML servlet registered under alias " + _servletAlias);
			_journal.info("Open-Xchange USM/SyncML Version " + SyncMLVersion.VERSION + " (Build "
					+ SyncMLVersion.BUILD_NUMBER + ") successfully started");
			//_debugWriter = new DebugPrintWriter(SYNC_ML_DEBUG_LOG);
			//_journal.info("SYNCML debug logging activated, using " + SYNC_ML_DEBUG_LOG);
		} catch (ServletException e) {
			logAndThrowStartupError(SyncMLBundleErrorCodes.SERVLET_REGISTRATION_FAILED, "Servlet registration failed",
					e);
		} catch (NamespaceException e) {
			logAndThrowStartupError(SyncMLBundleErrorCodes.SERVLET_ALIAS_IN_USE, "Servlet alias already in use:"
					+ _servletAlias, e);
		}
		//		} catch (FileNotFoundException e) {
		//			logAndThrowStartupError(SyncMLBundleErrorCodes.MISSING_LOG_FILE, "Syncml debug file missing"
		//					+ _servletAlias, e);
		//		}
	}

	private void logAndThrowStartupError(int errorCode, String detailMessage, Throwable cause) {
		String message = "Open-Xchange USM-SyncML couldn't be started: " + detailMessage;
		_journal.error(message, cause);
		deactivate();
		throw new USMStartupException(errorCode, message, cause);
	}

	private void logAndThrowConfigurationError(int errorCode, String property, Throwable cause) {
		logAndThrowStartupError(errorCode, "Invalid SyncML configuration value for " + property, cause);
	}

	public void deactivate() {
		if (_servletAlias != null) {
			_httpService.unregister(_servletAlias);
			_journal.info("SyncML servlet unregistered from alias " + _servletAlias);
			_servletAlias = null;
		} else {
			_journal.info("SyncML servlet unregistered");
		}
		_sessionManager = null;
		_contentTypeManager = null;
		_journal = null;
		_httpService = null;
	}

	@Override
	protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		DebugLogWriter sessionWriter = null;
		try {
			boolean wbxml = isWbxmlRequest(req);
			UUID requestUUID = null;
			if (LOG_REQUEST_AND_OUTPUT) {
				sessionWriter = new DebugLogWriter(_journal, SYNC_ML_DEBUG_LOG);
				req = new DebugServletRequest(_journal, req, wbxml, sessionWriter);
				requestUUID = ((DebugServletRequest) req).getRequestUUID();
			}
			XmlPullParser xmlparser = XmlPullParserFactory.getInstance().getParser(wbxml);
			xmlparser.setInput(req.getInputStream(), SyncMLConstants.UTF_8);
			XmlSerializer serializer = XmlSerializerFactory.getInstance().getSerializer(wbxml);
			ByteArrayOutputStream output = new ByteArrayOutputStream();
			serializer.setOutput(output, SyncMLConstants.UTF_8);
			if (_redirectURLs)
				req.getSession(); // Make sure that HttpSession is initialized for correct URL rewriting
			SyncMLRequest request = new SyncMLRequest(this, xmlparser, serializer, wbxml);
			int status = request.handleRequest(_redirectURLs ? resp : null);
			resp.setStatus(status);
			if (status == HttpServletResponse.SC_OK) {
				logResponse(status, output, sessionWriter, requestUUID, wbxml);
				resp.setContentType(req.getContentType());
				resp.setContentLength(output.size());
				resp.setCharacterEncoding(SyncMLConstants.UTF_8);
				ServletOutputStream outputStream = resp.getOutputStream();
				output.writeTo(outputStream);
				outputStream.flush();
				Toolkit.close(outputStream);
			}
		} catch (Exception error) {
			_journal.error("Unexpected internal error while processing request", error);
			resp.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
		} finally {
			if (sessionWriter != null)
				sessionWriter.flush();
		}
	}

	private boolean isWbxmlRequest(HttpServletRequest req) {
		String contentType = req.getContentType();
		boolean WBXML = contentType != null && contentType.indexOf("wbxml") >= 0;
		return WBXML;
	}

	private void logResponse(int status, ByteArrayOutputStream output, DebugWriter sessionWriter, UUID requestUUID,
			boolean isWbxml) {
		boolean printResponse = sessionWriter != null;
		boolean debugLog = _journal.isDebugEnabled();
		if (!printResponse && !debugLog)
			return;

		StringBuilder result = new StringBuilder();
		if (printResponse) {
			if (status == HttpStatus.SC_OK) {
				result.append(requestUUID).append(" --- Outgoing result: ").append(" ---\n");
				byte[] data = output.toByteArray();
				if (isWbxml) {
					try {
						result.append(new WbxmlTransformer().transformSyncMLWbxmlToXml(data));
					} catch (Exception ignored) {
					}
				} else {
					result.append(new String(data));
				}

			} else {
				result.append(requestUUID).append(" --- Outgoing error code: ").append(status).append(" ---");
			}
			String s = result.toString();
			if (sessionWriter != null)
				sessionWriter.log(s);
		}
	}

	@Override
	protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		resp.setStatus(HttpStatus.SC_METHOD_NOT_ALLOWED);
	}

	@Override
	protected void doOptions(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
		resp.setStatus(HttpStatus.SC_METHOD_NOT_ALLOWED);
	}

	public SessionManager getSessionManager() {
		return _sessionManager;
	}

	public Log getJournal() {
		return _journal;
	}

	public SyncMLSessionInitializer getSessionInitializer() {
		return _sessionInitializer;
	}

	public ContentTypeManager getContentTypeManager() {
		return _contentTypeManager;
	}

	public SyncMLSessionData getSessionData(String deviceLocURI) {
		SyncMLSessionData result = _sessionData.get(deviceLocURI);
		if (result == null) {
			result = new SyncMLSessionData();
			_sessionData.put(deviceLocURI, result);
		} else
			result.updateLastAccess();
		long now = System.currentTimeMillis();
		long storageLimit = now - _sessionDataCleanupInterval;
		if (_lastSessionDataCleanup < storageLimit) {
			removeUnusedSessionData(storageLimit);
			_lastSessionDataCleanup = now;
		}
		return result;
	}

	private void removeUnusedSessionData(long storageLimit) {
		for (Iterator<SyncMLSessionData> i = _sessionData.values().iterator(); i.hasNext();) {
			SyncMLSessionData e = i.next();
			if (e.getLastAccess() < storageLimit) {
				e.checkForClearedSyncData(_journal);
				i.remove();
			}
		}
	}

	public boolean isFormattedServerAnchorsEnabled() {
		return _formattedServerAnchors;
	}

	public int getDefaultMaxMessageSize() {
		return _defaultMaxMessageSize;
	}
}
