/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.servlet;

import static com.openexchange.usm.syncml.SyncMLConstants.FORMAT;
import static com.openexchange.usm.syncml.SyncMLConstants.TYPE;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.logging.Log;
import org.xmlpull.v1.*;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.syncml.*;
import com.openexchange.usm.syncml.commands.*;
import com.openexchange.usm.syncml.config.DeviceInfo;
import com.openexchange.usm.syncml.config.FolderHierarchy;
import com.openexchange.usm.syncml.data.SyncMLProtocolInfo;
import com.openexchange.usm.syncml.elements.*;
import com.openexchange.usm.syncml.elements.xml.SimpleXMLPart;
import com.openexchange.usm.syncml.exceptions.SyncMLException;
import com.openexchange.usm.syncml.sync.SyncInformation;
import com.openexchange.usm.syncml.util.MimeTypeUtil;
import com.openexchange.usm.syncml.util.XmlSizeSupport;

public class SyncMLRequest {
	private static final String BASIC_AUTH_DELIMITER = ":";

	private final static String DATE_FORMAT_STRING = "yyyyMMdd'T'HHmmssSSS'Z'";

	// 3 fields for a sync state information: client_anchor, server_anchor and the internal timestamp used for the server anchor
	// We store those 3 fields for 2 states, "last" and "next":
	// "last": previous values from "next" that we know the client has because he has used them or is using them now
	// "next": values that we use as a result of the current sync.
	// "last" will not change until the client uses the client anchor from "next" in which case all value from "next" are copied to "last"
	private static final String LAST_CLIENT_ANCHOR_PREFIX = "O";
	private static final String LAST_SERVER_ANCHOR_PREFIX = "L";
	private static final String LAST_SERVER_TIMESTAMP_PREFIX = "P";
	private static final String NEXT_CLIENT_ANCHOR_PREFIX = "C";
	private static final String NEXT_SERVER_ANCHOR_PREFIX = "S";
	private static final String NEXT_SERVER_TIMESTAMP_PREFIX = "T";

	private final SyncMLServlet _servlet;
	private final XmlPullParser _parser;
	private final XmlSerializer _serializer;
	private final Log _journal;
	private final boolean _wbxml;

	private SyncML _request;
	private SyncML _response;
	private Session _usmSession;
	private SyncMLSessionData _sessionData;

	private int _currentCommandID = 1;

	private final List<SyncMLCommand> _responseCommands = new ArrayList<SyncMLCommand>();

	// TODO This List can probably be removed, we most likely have to iterate over all SyncInformations currently stored (to check for partially transmitted requests)
	private final List<SyncInformation> _usedSyncInformations = new ArrayList<SyncInformation>();

	private boolean _slowSyncRequired;
	private boolean _serverDevInfRequested;
	private boolean _clientDevInfSent;

	private int _maxMessageSize;

	private Map<String, Folder> _folders = null;

	public SyncMLRequest(SyncMLServlet servlet, XmlPullParser parser, XmlSerializer serializer, boolean wbxml) {
		_servlet = servlet;
		_parser = parser;
		_serializer = serializer;
		_wbxml = wbxml;
		_journal = servlet.getJournal();
	}

	public int handleRequest(HttpServletResponse resp) {
		// 1) parse request
		try {
			_request = new SyncML(_parser);
		} catch (XmlPullParserException e) {
			return logBadRequest(SyncMLBundleErrorCodes.REQUEST_BAD_XML, ": Bad SyncML request", e);
		} catch (IOException e) {
			return logBadRequest(SyncMLBundleErrorCodes.REQUEST_IO_ERROR, ": I/O error while parsing request", e);
		}
		_sessionData = _servlet.getSessionData(_request.getSyncHdr().getSource().getLocURI());
		_response = new SyncML(_request, resp);
		// TODO Check that header fields contain expected values ?
		SyncMLStatusCode status = retrieveUSMSession();
		_maxMessageSize = extractMaxMessageSize();
		addSyncHdrStatus(status);
		if (_usmSession != null) {
			// if authenticated, execute commands and build response
			executeCommands();
		} else {
			// if authenticated, but access not allowed, build response
			failAllCommands(_request.getSyncBody().getCommands(), status);
		}
		// write response
		try {
			_response.write(_serializer, null);
			return HttpStatus.SC_OK;
		} catch (IOException e) {
			_journal.error("Error writing response", e);
			return HttpStatus.SC_INTERNAL_SERVER_ERROR;
		}
	}

	private int extractMaxMessageSize() {
		Meta meta = _request.getSyncHdr().getMeta();
		if (meta != null) {
			String val = meta.getXMLData(SyncMLConstants.MAX_MSG_SIZE);
			if (val != null) {
				try {
					int maxMsgSize = Integer.parseInt(val);
					if (maxMsgSize > 0)
						return maxMsgSize;
				} catch (NumberFormatException nfe) {
				}
				if (_journal.isDebugEnabled())
					_journal.debug(_usmSession + " Client device sent illegal MaxMsgSize of " + val);
			}
		}
		return _servlet.getDefaultMaxMessageSize();
	}

	private void executeCommands() {
		boolean hadSyncAction = false;
		for (SyncMLCommand command : _request.getSyncBody()) {
			try {
				hadSyncAction |= command.isSyncAction();
				command.execute(this);
			} catch (SyncMLException smle) {
				if (_journal.isDebugEnabled())
					_journal.debug("Error while processing " + command.getElementName(), smle);
				buildErrorStatus(command, smle.getStatusCode());
			} catch (Exception e) {
				_journal.error(_usmSession + ": Unexpected error while executing " + command.getElementName(), e);
				buildErrorStatus(command, SyncMLStatusCode.SERVER_FAILURE);
			}
		}
		if (_slowSyncRequired) {
			if (!_serverDevInfRequested)
				addDevInfPutCommand();
			if (!_clientDevInfSent)
				addCommandToResponse(DeviceInfo.getDeviceInfoRequest(_wbxml));
		}
		if (!hadSyncAction)
			checkForServerInitiatedAlert();
		for (SyncMLCommand command : _responseCommands)
			addCommandToResponse(command);
		updateSyncInfoStorage();
	}

	private void updateSyncInfoStorage() {
		int currentMessageSize = XmlSizeSupport.computeSize(_response, _wbxml, true);
		for (SyncInformation syncInfo : _sessionData.getAllSyncInformations()) {
			if (syncInfo.isNew())
				continue;
			if (syncInfo.isIncomplete())
				currentMessageSize = syncInfo.addSyncCommands(this, currentMessageSize, _maxMessageSize);
			if (syncInfo.isIncomplete()) {
				_response.getSyncBody().setFinal(false);
				break;
			}
			try {
				if (syncInfo.updateUSMStorage(_usmSession, _journal)) {
					updateAnchorData(syncInfo);
					_sessionData.removeSyncInformation(syncInfo.getOxFolderID());
				}
			} catch (USMException e) {
				_journal.error(_usmSession + ": Couldn't store persistent sync information", e);
			}
		}
	}

	private void addDevInfPutCommand() {
		try {
			Meta meta = new Meta(new SimpleXMLPart(FORMAT, null, _wbxml ? "wbxml" : "xml"), new SimpleXMLPart(TYPE,
					null, MimeTypeUtil.buildXmlEncodedMimeType(SyncMLConstants.DEVINF_TYPE, _wbxml)));
			Item item = DeviceInfo.getDeviceInfo(this, null, new Source(SyncMLConstants.DEVINF_TARGET_DEFAULT, null));
			addCommandToResponse(new Put(meta, item));
		} catch (SyncMLException e) {
			_journal.info("Automatic creation of Put on Slow-Sync failed", e);
			// Do not send Put command if creation failed for some reason
		}
	}

	private void checkForServerInitiatedAlert() {
		// TODO Check if there are any changes on the server since the last sync
		// TODO If yes, add a server-initiated Alert (problem: for which path ?)
	}

	private void buildErrorStatus(SyncMLCommand command, SyncMLStatusCode status) {
		addStatusForCommand(command, null, null, status);
		failAllCommands(command.getSubCommands(), status);
	}

	private void failAllCommands(List<SyncMLCommand> commands, SyncMLStatusCode status) {
		for (SyncMLCommand command : commands) {
			addStatusForCommand(command, null, null, status);
			failAllCommands(command.getSubCommands(), status);
		}
	}

	private SyncMLStatusCode retrieveUSMSession() {
		// 2) authenticate
		String basicAuthentication = getAuthentication();
		if (basicAuthentication == null)
			return SyncMLStatusCode.AUTHENTICATION_REQUIRED;
		String[] parts = basicAuthentication.split(BASIC_AUTH_DELIMITER, 2);
		if (parts.length != 2)
			return SyncMLStatusCode.AUTHENTICATION_REQUIRED;
		String user = parts[0];
		String password = parts[1];
		String device = _request.getSyncHdr().getSource().getLocURI();
		// 3) check if authentication is valid, retrieve USM session
		try {
			_usmSession = _servlet.getSessionManager().getSession(user, password, SyncMLConstants.PROTOCOL_NAME,
					device, _servlet.getSessionInitializer());
			return SyncMLStatusCode.AUTHENTICATED;
		} catch (AuthenticationFailedException e) {
			if (_journal.isDebugEnabled())
				_journal.debug(user + ": " + e.getMessage());
			return SyncMLStatusCode.UNAUTHORIZED;
		} catch (USMAccessDeniedException e) {
			if (_journal.isDebugEnabled())
				_journal.debug(user + ": " + e.getMessage());
			return SyncMLStatusCode.AUTHENTICATION_REQUIRED;
		} catch (USMException e) {
			_journal.error("Error retrieving USM session for user " + user + ", device " + device, e);
			return SyncMLStatusCode.SERVER_FAILURE; // LATER: Provide other status codes based on exception ?
		}
	}

	private void addSyncHdrStatus(SyncMLStatusCode status) {
		Chal chal = (status == SyncMLStatusCode.OK || status == SyncMLStatusCode.AUTHENTICATED) ? null : new Chal();
		Data data = new Data(String.valueOf(status.getCode()));
		addCommandToResponse(new Status(_request.getSyncHdr(), null, chal, data));
	}

	private String getAuthentication() {
		String sessionID = _request.getSyncHdr().getSessionID();
		Cred cred = _request.getSyncHdr().getCred();
		// 2.a) If authentication not present, get authentication from HttpSession
		if (cred == null)
			return _sessionData.getAuthentication(sessionID);
		// 2.b) try to get authentication from SyncHdr
		String auth = cred.getBasicAuthentication(_journal);
		_sessionData.setAuthentication(sessionID, auth);
		return auth;
	}

	private int logBadRequest(int code, String message, Throwable t) {
		if (_journal.isDebugEnabled())
			_journal.debug(Integer.toHexString(code) + message, t);
		return HttpStatus.SC_BAD_REQUEST;
	}

	public void addCommandToResponse(SyncMLCommand command) {
		SyncBody syncBody = _response.getSyncBody();
		syncBody.addCommand(command);
		setCommandIDs(command);
	}

	public void addCommand(SyncMLCommand command) {
		_responseCommands.add(command);
	}

	private void setCommandIDs(SyncMLCommand command) {
		command.setCmdID(String.valueOf(_currentCommandID++));
		for (SyncMLCommand subCommand : command.getSubCommands())
			setCommandIDs(subCommand);
	}

	public void addStatusForCommand(SyncMLCommand command, String targetRef, String sourceRef, SyncMLStatusCode status) {
		addStatusForCommand(command, targetRef, sourceRef, new Data(String.valueOf(status.getCode())));
	}

	public void addStatusForCommand(SyncMLCommand command, String targetRef, String sourceRef, SyncMLStatusCode status,
			Item... items) {
		addStatusForCommand(command, targetRef, sourceRef, new Data(String.valueOf(status.getCode())), items);
	}

	public void addStatusForCommand(SyncMLCommand command, String targetRef, String sourceRef, Data data) {
		if (!command.isNoResp() && !_request.getSyncHdr().isNoResp())
			addCommandToResponse(new Status(_request.getSyncHdr().getMsgID(), command, targetRef, sourceRef, null,
					null, data));
	}

	public void addStatusForCommand(SyncMLCommand command, String targetRef, String sourceRef, Data data, Item... items) {
		if (!command.isNoResp() && !_request.getSyncHdr().isNoResp())
			addCommandToResponse(new Status(_request.getSyncHdr().getMsgID(), command, targetRef, sourceRef, null,
					null, data, items));
	}

	public Log getJournal() {
		return _journal;
	}

	public Session getUSMSession() {
		return _usmSession;
	}

	public int getMsgId() {
		return _request.getSyncHdr().getMsgID();
	}

	public Map<String, Folder> getFolders() throws USMException {
		if (_folders == null)
			_folders = FolderHierarchy.getInstance().getFolders(_usmSession);
		return _folders;
	}

	public ContentType getContentType(String contentType) {
		return _servlet.getContentTypeManager().getContentType(contentType);
	}

	public boolean isWbxml() {
		return _wbxml;
	}

	public Folder getFolder(String locURI) throws USMException {
		return FolderHierarchy.getInstance().getFolder(getFolders(), locURI);
	}

	public boolean isClientAnchorValid(String path, String clientAnchor, Folder folder) throws DatabaseAccessException,
			USMSQLException {
		String lastClientAnchorKey = getPersistentKey(LAST_CLIENT_ANCHOR_PREFIX, path);
		String nextClientAnchorKey = getPersistentKey(NEXT_CLIENT_ANCHOR_PREFIX, path);
		if (_usmSession.getPersistentField(nextClientAnchorKey).equals(clientAnchor)) {
			String nextServerAnchorKey = getPersistentKey(NEXT_SERVER_ANCHOR_PREFIX, path);
			String nextServerTimestampKey = getPersistentKey(NEXT_SERVER_TIMESTAMP_PREFIX, path);
			_usmSession.setPersistentField(lastClientAnchorKey, _usmSession.getPersistentField(nextClientAnchorKey));
			_usmSession.setPersistentField(getPersistentKey(LAST_SERVER_ANCHOR_PREFIX, path), _usmSession
					.getPersistentField(nextServerAnchorKey));
			_usmSession.setPersistentField(getPersistentKey(LAST_SERVER_TIMESTAMP_PREFIX, path), _usmSession
					.getPersistentField(nextServerTimestampKey));
			_usmSession.setPersistentField(nextClientAnchorKey, null);
			_usmSession.setPersistentField(nextServerAnchorKey, null);
			_usmSession.setPersistentField(nextServerTimestampKey, null);
		}
		if (!_usmSession.getPersistentField(lastClientAnchorKey).equals(clientAnchor))
			return false;
		if (getLastServerAnchor(path) == null)
			return false;
		long lastSyncTimestamp = getLastSyncTimestamp(path);
		if (lastSyncTimestamp == 0L)
			return false;
		return _usmSession.getCachedFolderElements(folder.getID(), folder.getElementsContentType(), lastSyncTimestamp) != null;
	}

	public String getLastServerAnchor(String path) {
		String result = _usmSession.getPersistentField(getPersistentKey(LAST_SERVER_ANCHOR_PREFIX, path));
		return result.length() > 0 ? result : null;
	}

	private void updateAnchorData(SyncInformation syncInfo) throws DatabaseAccessException, USMSQLException {
		String path = syncInfo.getClientLocURI();
		_usmSession.setPersistentField(getPersistentKey(NEXT_CLIENT_ANCHOR_PREFIX, path), syncInfo
				.getNextClientAnchor());
		_usmSession.setPersistentField(getPersistentKey(NEXT_SERVER_ANCHOR_PREFIX, path), syncInfo
				.getNextServerAnchor());
		_usmSession.setPersistentField(getPersistentKey(NEXT_SERVER_TIMESTAMP_PREFIX, path), Long.toString(syncInfo
				.getNextServerTimestamp()));
		if (_journal.isDebugEnabled()) {
			_journal.debug("For " + syncInfo + ":\n\t"
					+ _usmSession.getPersistentField(getPersistentKey(LAST_SERVER_TIMESTAMP_PREFIX, path)) + "->"
					+ _usmSession.getPersistentField(getPersistentKey(NEXT_SERVER_TIMESTAMP_PREFIX, path)) + "\n\t"
					+ _usmSession.getPersistentField(getPersistentKey(LAST_CLIENT_ANCHOR_PREFIX, path)) + "->"
					+ _usmSession.getPersistentField(getPersistentKey(NEXT_CLIENT_ANCHOR_PREFIX, path)) + "\n\t"
					+ _usmSession.getPersistentField(getPersistentKey(LAST_SERVER_ANCHOR_PREFIX, path)) + "->"
					+ _usmSession.getPersistentField(getPersistentKey(NEXT_SERVER_ANCHOR_PREFIX, path)));
		}
		// LATER Remove this after initial testing
		checkSynchronizedObjects(syncInfo.getOxFolderID(), syncInfo.getNextServerTimestamp());
	}

	private void checkSynchronizedObjects(String folderID, long timestamp) {
		try {
			Folder folder = _usmSession.findFolder(folderID);
			for (DataObject o : _usmSession.getCachedFolderElements(folderID, folder.getElementsContentType(),
					timestamp)) {
				if (!(o.getProtocolInformation() instanceof SyncMLProtocolInfo))
					throw new IllegalStateException("No SyncMLProtocolInfo for " + o);
			}
		} catch (Exception e) {
			_journal.error("Error while checking integrity", e);
		}
	}

	public long getLastSyncTimestamp(String path) {
		String result = _usmSession.getPersistentField(getPersistentKey(LAST_SERVER_TIMESTAMP_PREFIX, path));
		if (result.length() > 0) {
			try {
				return Long.parseLong(result);
			} catch (NumberFormatException ignored) {
			}
		}
		return 0L;
	}

	public String getNextServerAnchor(long lastSyncTimestamp) {
		long nextSyncTimestamp = System.currentTimeMillis();
		if (nextSyncTimestamp <= lastSyncTimestamp)
			nextSyncTimestamp = lastSyncTimestamp + 1;
		return _servlet.isFormattedServerAnchorsEnabled() ? new SimpleDateFormat(DATE_FORMAT_STRING).format(new Date(
				nextSyncTimestamp)) : String.valueOf(nextSyncTimestamp);
	}

	private String getPersistentKey(String prefix, String path) {
		if (path.length() < 18)
			return prefix + path;
		return prefix + path.substring(0, 17);
	}

	public void setSlowSyncRequired() {
		_slowSyncRequired = true;
	}

	public void setServerDevInfRequested() {
		_serverDevInfRequested = true;
	}

	public void setClientDevInfSent() {
		_clientDevInfSent = true;
	}

	public void storeSyncInformation(SyncInformation syncInformation) {
		_sessionData.storeSyncInformation(syncInformation);
	}

	public SyncInformation getSyncInformation(String clientLocURI, String serverLocURI) {
		SyncInformation syncInfo = _sessionData.getSyncInformation(clientLocURI, serverLocURI);
		if (syncInfo != null && !_usedSyncInformations.contains(syncInfo))
			_usedSyncInformations.add(syncInfo);
		return syncInfo;
	}
}
