/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.mapping.vcard;

import java.util.*;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.ProtocolInformation;
import com.openexchange.usm.syncml.SyncMLBundleErrorCodes;
import com.openexchange.usm.syncml.SyncMLStatusCode;
import com.openexchange.usm.syncml.data.SyncMLProtocolInfo;
import com.openexchange.usm.syncml.elements.xml.ComposedXMLPart;
import com.openexchange.usm.syncml.mapping.BaseContentTypeMapping;
import com.openexchange.usm.syncml.mapping.MappingFailedException;

public class VCard_2_1_Mapping extends BaseContentTypeMapping {

	private static final String[] INTERNET = { "INTERNET" };
	private static final String[] PAGER = { "PAGER" };
	private static final String[] ISDN = { "ISDN" };
	private static final String[] CAR = { "CAR" };
	private static final String[] FAX = { "FAX" };
	private static final String[] HOME_FAX = { "HOME", "FAX" };
	private static final String[] CELL = { "CELL" };
	private static final String[] HOME = { "HOME" };
	private static final String[] WORK = { "WORK" };
	private static final String[] VOICE = { "VOICE" };

	private final static FieldMapping[] FIELD_MAPPINGS = {
			new SimpleFieldMapping("FN", null, null, "display_name"),
			new NameMapping("N", "last_name", "first_name", "second_name", "title", "suffix"),
			new OrgMapping("ORG", "company", "branches", "department"),
			new AdrMapping("ADR", WORK, null, "street_business", "city_business", "state_business",
					"postal_code_business", "country_business"),
			new AdrMapping("ADR", null, HOME, "street_home", "city_home", "state_home", "postal_code_home",
					"country_home"),
			new SimpleFieldMapping("TEL", CAR, VOICE, "telephone_car"),
			new SimpleFieldMapping("TEL", ISDN, VOICE, "telephone_isdn"),
			new SimpleFieldMapping("TEL", PAGER, null, "telephone_pager"),
			new SimpleFieldMapping("TEL", CELL, VOICE, "cellular_telephone1", "cellular_telephone2"),
			new SimpleFieldMapping("TEL", HOME_FAX, null, "fax_home"),
			new SimpleFieldMapping("TEL", FAX, null, "fax_business"),
			new SimpleFieldMapping("TEL", HOME, VOICE, "telephone_home1", "telephone_home2"),
			new SimpleFieldMapping("TEL", null, WORK, "telephone_business1", "telephone_business2"),
			new SimpleFieldMapping("EMAIL", null, INTERNET, "email1", "email2", "email3"),
			new SimpleFieldMapping("URL", null, null, "url"),
			new BirthdayMapping("BDAY", "birthday") };

	public String getMimeType() {
		return "text/x-vcard";
	}

	public String getVersion() {
		return "2.1";
	}

	public String getContentType() {
		return DefaultContentTypes.CONTACTS_ID;
	}

	@Override
	public void addFieldDescriptionList(ComposedXMLPart ctCap) {
		super.addFieldDescriptionList(ctCap);
		Map<String, Set<String>> supportedProps = new HashMap<String, Set<String>>();
		for (FieldMapping m : FIELD_MAPPINGS) {
			Set<String> params = supportedProps.get(m.getVCardProperty());
			if (params == null) {
				params = new HashSet<String>();
				supportedProps.put(m.getVCardProperty(), params);
			}
			for (String v : m.getVCardOptions())
				params.add(v);
			for (String v : m.getSendOptions())
				params.add(v);
		}
		addPropValue(ctCap, "BEGIN", "VCARD");
		addPropValue(ctCap, "END", "VCARD");
		addPropValue(ctCap, "VERSION", "2.1");
		for (Map.Entry<String, Set<String>> e : supportedProps.entrySet()) {
			Set<String> params = e.getValue();
			addProperty(ctCap, e.getKey(), params.toArray(new String[params.size()]));
		}
	}

	public BitSet determineSynchronizedFields(ContentType contentType, Map<String, String[]> properties) {
		BitSet result = new BitSet();
		Map<String, Integer> contentTypeFieldMap = buildContentTypeFieldMap(contentType);
		for (FieldMapping fm : FIELD_MAPPINGS) {
			if (fm.isFieldDefined(properties)) {
				for (String oxField : fm._oxFields)
					result.set(contentTypeFieldMap.get(oxField));
			}
		}
		return result;
	}

	public BitSet getSynchronizationFields(ContentType contentType) {
		BitSet result = new BitSet();
		Map<String, Integer> contentTypeFieldMap = buildContentTypeFieldMap(contentType);
		for (FieldMapping fm : FIELD_MAPPINGS) {
			for (String oxField : fm._oxFields)
				result.set(contentTypeFieldMap.get(oxField));
		}
		return result;
	}

	public void mapToDataObject(String vcard, DataObject object) throws MappingFailedException {
		if (!object.getContentType().getID().equals(getContentType()))
			throw new MappingFailedException(SyncMLBundleErrorCodes.VCARD_CONTENT_TYPE_MISMATCH,
					SyncMLStatusCode.SERVER_FAILURE, "Internal error: ContentType mismatch: "
							+ object.getContentType().getID() + " != " + getContentType());
		List<VCardEntry> entries = getVCardEntries(vcard);
		Object[] fieldMappingData = new Object[FIELD_MAPPINGS.length];
		for (VCardEntry entry : entries) {
			for (int i = 0; i < FIELD_MAPPINGS.length; i++) {
				FieldMapping mapping = FIELD_MAPPINGS[i];
				if (mapping.matches(entry)) {
					try {
						fieldMappingData[i] = mapping.storeInDataObject(entry, object, fieldMappingData[i]);
					} catch (IllegalArgumentException e) {
						throw new MappingFailedException(SyncMLBundleErrorCodes.VCARD_FIELD_PARSING_FAILED,
								SyncMLStatusCode.BAD_REQUEST, "Couldn't store " + entry + " in DataObject", e);
					}
					break;
				}
			}
		}
	}

	public String mapToText(DataObject object) throws MappingFailedException {
		List<VCardEntry> vcard = new ArrayList<VCardEntry>();
		vcard.add(new VCardEntry(VCardConstants.BEGIN, VCardConstants.VCARD));
		vcard.add(new VCardEntry(VCardConstants.VERSION, getVersion()));
		ProtocolInformation protocolInfo = object.getProtocolInformation();
		if (protocolInfo instanceof SyncMLProtocolInfo)
			vcard.add(new VCardEntry("X-IRMC-LUID", ((SyncMLProtocolInfo) protocolInfo).getDeviceObjectID()));
		for (FieldMapping mapping : FIELD_MAPPINGS)
			mapping.addToVCard(object, vcard);
		vcard.add(new VCardEntry(VCardConstants.END, VCardConstants.VCARD));
		StringBuilder sb = new StringBuilder();
		for (VCardEntry entry : vcard)
			sb.append(entry).append('\r').append('\n');
		return sb.toString();
	}

	private List<VCardEntry> getVCardEntries(String vcard) {
		List<VCardEntry> result = new ArrayList<VCardEntry>();
		int startOfLine = 0;
		StringBuilder partialLine = new StringBuilder();
		boolean shouldAddLine = false;
		for (;;) {
			int endOfLine = vcard.indexOf('\n', startOfLine) - 1;
			if (endOfLine < 0) {
				shouldAddLine = addToVCardEntryList(result, shouldAddLine, partialLine.append(
						vcard.substring(startOfLine)).toString());
				return result;
			}
			int startOfNextLine = endOfLine + 2;
			if (endOfLine > startOfLine && vcard.charAt(endOfLine) == '\r')
				endOfLine--;
			int endOfData = endOfLine;
			while (endOfLine > startOfLine && Character.isWhitespace(vcard.charAt(endOfLine)))
				endOfLine--;
			if (!appendLine(endOfLine, startOfLine, vcard, partialLine)) {
				shouldAddLine = addToVCardEntryList(result, shouldAddLine, partialLine.append(
						vcard.substring(startOfLine, endOfData + 1)).toString());
				partialLine = new StringBuilder();
			}
			startOfLine = startOfNextLine;
		}
	}

	private boolean appendLine(int endOfLine, int startOfLine, String vcard, StringBuilder partialLine) {
		if (endOfLine <= startOfLine || vcard.charAt(endOfLine) != '=')
			return false;
		String line = vcard.substring(startOfLine, endOfLine);
		if ((partialLine.toString() + line).toUpperCase().indexOf("ENCODING=B") >= 0)
			return false;
		partialLine.append(line);
		return true;
	}

	private boolean addToVCardEntryList(List<VCardEntry> result, boolean shouldAddLine, String line) {
		if (line == null || line.trim().length() == 0)
			return shouldAddLine;
		if (shouldAddLine && result.size() > 0 && line.indexOf(':') < 0) {
			VCardEntry last = result.get(result.size() - 1);
			last.addToLastDataEntry(line);
			return true;
		}
		VCardEntry entry = new VCardEntry(line);
		if (!shouldAddLine)
			return entry.isVCardBegin();
		if (entry.isVCardEnd())
			return false;
		result.add(entry);
		return true;
	}
}
