/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.commands;

import static com.openexchange.usm.syncml.SyncMLConstants.*;

import java.io.IOException;
import java.util.*;
import java.util.Map;
import java.util.Map.Entry;

import org.xmlpull.v1.*;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.syncml.SyncMLConstants;
import com.openexchange.usm.syncml.SyncMLStatusCode;
import com.openexchange.usm.syncml.config.DeviceInfo;
import com.openexchange.usm.syncml.elements.*;
import com.openexchange.usm.syncml.elements.xml.ComposedXMLPart;
import com.openexchange.usm.syncml.elements.xml.XMLPart;
import com.openexchange.usm.syncml.exceptions.SyncMLException;
import com.openexchange.usm.syncml.mapping.ContentTypeMapping;
import com.openexchange.usm.syncml.mapping.ContentTypeMappings;
import com.openexchange.usm.syncml.parser.XmlPullParserUtil;
import com.openexchange.usm.syncml.servlet.SyncMLRequest;

public class Put extends BaseCommand {
	public static final String ELEMENT_NAME = "Put";

	private final boolean _noResp;
	private final String _lang;
	private final Cred _cred;
	private final Meta _meta;
	private final List<Item> _items = new ArrayList<Item>();

	public Put(XmlPullParser parser, Meta meta, String cmdID) throws XmlPullParserException, IOException {
		setCmdID(cmdID);
		_noResp = readNoResp(parser);
		_lang = readOptionalLang(parser);
		_cred = readOptionalCred(parser, meta);
		_meta = readOptionalMeta(parser, meta);
		readItems(parser, _meta, _items, 1);
		readElementEnd(parser);
	}

	public Put(Meta meta, Item item) {
		_noResp = SyncMLConstants.NO_RESP_DEFAULT;
		_lang = null;
		_cred = null;
		_meta = meta;
		_items.add(item);
	}

	public String getElementName() {
		return ELEMENT_NAME;
	}

	@Override
	protected void writeContent(XmlSerializer serializer, Meta meta) throws IOException {
		super.writeContent(serializer, meta);
		writeNoResp(serializer, _noResp);
		XmlPullParserUtil.writeOptionalText(serializer, SimpleElements.LANG, _lang);
		if (_cred != null)
			_cred.write(serializer, meta);
		if (_meta != null) {
			_meta.write(serializer, meta);
			meta = _meta;
		}
		writeItems(serializer, meta, _items);
	}

	public boolean isNoResp() {
		return _noResp;
	}

	public void execute(SyncMLRequest syncMLRequest) throws SyncMLException {
		SyncMLStatusCode status = SyncMLStatusCode.OPTIONAL_FEATURE_NOT_SUPPORTED;
		Source s = null;
		for (Item i : _items) {
			s = i.getSource();
			if (DeviceInfo.isDeviceInfoRequest(s, i.getMeta())) {
				status = updateSynchronizationFields(syncMLRequest, i.getData());
			}
		}
		syncMLRequest.addStatusForCommand(this, null, (s == null) ? null : s.getLocURI(), status);
	}

	private SyncMLStatusCode updateSynchronizationFields(SyncMLRequest syncMLRequest, Data data) {
		if (data == null)
			return SyncMLStatusCode.INCOMPLETE_COMMAND;
		if (data.isStringContent())
			return (data.getStringContent().length() == 0) ? SyncMLStatusCode.OK : SyncMLStatusCode.BAD_REQUEST;
		XMLPart part = data.getXmlContent();
		if (!DEV_INFO.equals(part.getTag()))
			return SyncMLStatusCode.INCOMPLETE_COMMAND;
		part = part.getSubPart(CT_CAP);
		if (!(part instanceof ComposedXMLPart))
			return SyncMLStatusCode.INCOMPLETE_COMMAND;
		java.util.Map<ContentType, BitSet> capabilitiesMap = new HashMap<ContentType, BitSet>();
		java.util.Map<String, String[]> properties = new HashMap<String, String[]>();
		List<String> parameters = new ArrayList<String>();
		String currentProperty = null;
		ContentType contentType = null;
		ContentTypeMapping mapping = null;
		for (XMLPart p : ((ComposedXMLPart) part).getSubParts()) {
			if (CT_TYPE.equals(p.getTag())) {
				updateContentTypeFields(capabilitiesMap, contentType, mapping, properties);
				mapping = ContentTypeMappings.getMapping(p.getValue());
				if (mapping != null)
					contentType = syncMLRequest.getContentType(mapping.getContentType());
				if (contentType == null)
					mapping = null;
				properties.clear();
				currentProperty = null;
			} else if (mapping != null) {
				if (PROP_NAME.equals(p.getTag())) {
					if (currentProperty != null)
						properties.put(currentProperty, parameters.toArray(new String[parameters.size()]));
					currentProperty = p.getValue();
					parameters.clear();
				} else if (PARAM_NAME.equals(p.getTag())) {
					parameters.add(p.getValue());
				}
			}
		}
		updateContentTypeFields(capabilitiesMap, contentType, mapping, properties);
		Session session = syncMLRequest.getUSMSession();
		try {
			for (Entry<ContentType, BitSet> entry : capabilitiesMap.entrySet())
				session.setFieldFilter(entry.getKey(), entry.getValue());
			syncMLRequest.setClientDevInfSent();
			return SyncMLStatusCode.OK;
		} catch (USMException e) {
			syncMLRequest.getJournal().error("Failed to set the fields to synchronize", e);
			return SyncMLStatusCode.SERVER_FAILURE;
		}
	}

	private void updateContentTypeFields(Map<ContentType, BitSet> capabilitiesMap, ContentType contentType,
			ContentTypeMapping mapping, Map<String, String[]> properties) {
		if (mapping == null)
			return;
		BitSet fields = mapping.determineSynchronizedFields(contentType, properties);
		BitSet oldFields = capabilitiesMap.get(contentType);
		if (oldFields != null)
			oldFields.or(fields);
		else
			capabilitiesMap.put(contentType, fields);
	}
}
