/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
/**
 * 
 */
package com.openexchange.usm.session.sync.test;

import junit.framework.TestCase;

import org.apache.commons.logging.Log;

import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.configuration.ConfigurationProperties;
import com.openexchange.usm.journal.SimJournal;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.sync.*;

/**
 * @author ldo
 *
 */
public class SlowContentSyncerTest extends TestCase {
	private final static int MANY_DATAOBJECTS_COUNT = 100;

	DataObject[] clientData = {};
	DataObject[] dataOnServer = {};
	SimContentSyncerStorage syncerStorage = new SimContentSyncerStorage();
	SimJournal journal = new SimJournal();
	Log log = journal.getLog(SlowContentSyncerTest.class);
	SimSession session = new SimSession();
	SimSyncContentType contentType = new SimSyncContentType();
	SimSyncContentTypeTransferHandler handler = (SimSyncContentTypeTransferHandler) contentType.getTransferHandler();
	SlowContentSyncer slowContentSyncer = new SlowContentSyncer(new SessionManagerImpl(),
			ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT);

	public void testNoObjectOnClientNoObjectOnServerERROR() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.ERROR);
	}

	public void testNoObjectOnClientNoObjectOnServerCLIENT() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testNoObjectOnClientNoObjectOnServerSERVER() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.USE_SERVER);
	}

	public void testNoObjectOnClientNoObjectOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testNoObjectOnClientNoObjectOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testNoObjectOnClientNoObjectOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientNoObjectOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void noObjectOnClientNoObjectOnServer(ConflictResolution resolution) throws USMException {
		clientData = new DataObject[0];
		dataOnServer = new DataObject[0];
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(0, result.getChanges().length);
		assertEquals(0, syncerStorage.getServerDataToStore().length);
	}

	public void testObjectOnClientNoObjectOnServerERROR() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.ERROR);
	}

	public void testObjectOnClientNoObjectOnServerCLIENT() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testObjectOnClientNoObjectOnServerSERVER() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.USE_SERVER);
	}

	public void testObjectOnClientNoObjectOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testObjectOnClientNoObjectOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testObjectOnClientNoObjectOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void objectOnClientNoObjectOnServer(ConflictResolution resolution) throws USMException {
		clientData = new DataObject[1];
		clientData[0] = contentType.newDataObject(session);
		dataOnServer = new DataObject[0];
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		handler._writeNewToCall = true;
		handler._objectToSend = clientData[0];
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(1);
		assertEquals(0, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(clientData[0], syncerStorage.getServerDataToStore()[0]);
	}

	public void testNoObjectOnClientObjectOnServerERROR() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.ERROR);
	}

	public void testNoObjectOnClientObjectOnServerCLIENT() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testNoObjectOnClientObjectOnServerSERVER() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.USE_SERVER);
	}

	public void testNoObjectOnClientObjectOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testNoObjectOnClientObjectOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testNoObjectOnClientObjectOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		noObjectOnClientObjectOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void noObjectOnClientObjectOnServer(ConflictResolution resolution) throws USMException {
		clientData = new DataObject[0];
		dataOnServer = new DataObject[1];
		dataOnServer[0] = contentType.newDataObject(session);
		dataOnServer[0].commitChanges();
		DataObject expectedClientResult = dataOnServer[0].createCopy(false);
		expectedClientResult.setChangeState(ChangeState.CREATED);
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		contentType.setMatchRating(1);
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(1, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(dataOnServer[0], syncerStorage.getServerDataToStore()[0]);
		assertTrue(contentType.getMatchRating(result.getChanges()[0], dataOnServer[0]) >= 0);
		assertEquals(expectedClientResult, result.getChanges()[0]);
	}

	public void testDifferentObjectsOnClientAndOnServerERROR() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.ERROR);
	}

	public void testDifferentObjectsOnClientAndOnServerCLIENT() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testDifferentObjectsOnClientAndOnServerSERVER() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.USE_SERVER);
	}

	public void testDifferentObjectsOnClientAndOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testDifferentObjectsOnClientAndOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testDifferentObjectsOnClientAndOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		differentObjectsOnClientAndOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void differentObjectsOnClientAndOnServer(ConflictResolution resolution) throws USMException {
		clientData = new DataObject[1];
		clientData[0] = contentType.newDataObject(session);
		clientData[0].setID("clientID");
		clientData[0].setFieldContent("Field2", "Client");
		dataOnServer = new DataObject[1];
		dataOnServer[0] = contentType.newDataObject(session);
		dataOnServer[0].setID("serverID");
		dataOnServer[0].setFieldContent("Field2", "Server");
		dataOnServer[0].commitChanges();
		DataObject expectedClientResult = dataOnServer[0].createCopy(false);
		expectedClientResult.setChangeState(ChangeState.CREATED);
		contentType.setMatchRating(-1);
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		handler._writeNewToCall = true;
		handler._objectToSend = clientData[0];
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(1);
		assertEquals(1, result.getChanges().length);
		assertEquals(2, syncerStorage.getServerDataToStore().length);
		checkObjectInArray(dataOnServer[0], syncerStorage.getServerDataToStore());
		assertEquals(expectedClientResult, result.getChanges()[0]);
	}

	private void checkObjectInArray(DataObject object, DataObject[] data) {
		for (DataObject e : data) {
			if (object.equals(e))
				return;
		}
		fail("Required object not present or not with correct contents in result");
	}

	public void testIdenticalObjectsOnClientAndOnServerERROR() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.ERROR);
	}

	public void testIdenticalObjectsOnClientAndOnServerCLIENT() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testIdenticalObjectsOnClientAndOnServerSERVER() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.USE_SERVER);
	}

	public void testIdenticalObjectsOnClientAndOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testIdenticalObjectsOnClientAndOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testIdenticalObjectsOnClientAndOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsOnClientAndOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void identicalObjectsOnClientAndOnServer(ConflictResolution resolution) throws USMException {
		clientData = new DataObject[1];
		clientData[0] = contentType.newDataObject(session);
		clientData[0].setID("ID");
		dataOnServer = new DataObject[1];
		dataOnServer[0] = clientData[0].createCopy(false);
		dataOnServer[0].commitChanges();
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(0, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(dataOnServer[0], syncerStorage.getServerDataToStore()[0]);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerERROR() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.ERROR);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerCLIENT() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.USE_CLIENT);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerSERVER() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.USE_SERVER);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testIdenticalObjectsWithDifferentIDsOnClientAndOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void identicalObjectsWithDifferentIDsOnClientAndOnServer(ConflictResolution resolution) throws USMException {
		dataOnServer = new DataObject[1];
		dataOnServer[0] = contentType.newDataObject(session);
		dataOnServer[0].setID("serverID");
		clientData = new DataObject[1];
		clientData[0] = dataOnServer[0].createCopy(false);
		clientData[0].setID(null);
		dataOnServer[0].commitChanges();
		DataObject expectedClientChange = clientData[0].createCopy(false);
		expectedClientChange.commitChanges();
		expectedClientChange.setID("serverID");
		contentType.setMatchRating(1);
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(1, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(dataOnServer[0], syncerStorage.getServerDataToStore()[0]);
		assertEquals(expectedClientChange, result.getChanges()[0]);
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerERROR() throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.ERROR);
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerERROR_DELETE_OVER_CHANGE()
			throws USMException {
		objectOnClientNoObjectOnServer(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void objectsWithSameIDAndDifferentFieldsOnClientAndOnServerERROR(ConflictResolution resolution)
			throws USMException {
		initConflictingObjectsWithSameID();
		try {
			slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
			fail("Conflicting changes exception not thrown!");
		} catch (SynchronizationConflictException e) {

		}
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerCLIENT() throws USMException {
		objectsWithSameIDAndDifferentFieldsOnClientAndOnServerCLIENT(ConflictResolution.USE_CLIENT);
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerCLIENT_DELETE_OVER_CHANGE()
			throws USMException {
		objectsWithSameIDAndDifferentFieldsOnClientAndOnServerCLIENT(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void objectsWithSameIDAndDifferentFieldsOnClientAndOnServerCLIENT(ConflictResolution resolution)
			throws USMException {
		initConflictingObjectsWithSameID();
		handler.resetServerCalls();
		handler._writeUpdateToCall = true;
		handler._objectToSend = DataObjectUtil.copyAndModify(dataOnServer[0], clientData[0], false);
		//		handler._objectToSend.setChangeState(ChangeState.MODIFIED);
		//		handler._objectToSend.commitChanges();
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, resolution, log, 0, clientData);
		handler.checkCalls(1);
		assertEquals(0, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(clientData[0], syncerStorage.getServerDataToStore()[0]);
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerSERVER() throws USMException {
		objectsWithSameIDAndDifferentFieldsOnClientAndOnServerSERVER(ConflictResolution.USE_SERVER);
	}

	public void testObjectsWithSameIDAndDifferentFieldsOnClientAndOnServerSERVER_DELETE_OVER_CHANGE()
			throws USMException {
		objectsWithSameIDAndDifferentFieldsOnClientAndOnServerSERVER(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	public void objectsWithSameIDAndDifferentFieldsOnClientAndOnServerSERVER(ConflictResolution conflictResolution)
			throws USMException {
		initConflictingObjectsWithSameID();
		DataObject expectedClientResult = DataObjectUtil.copyAndModify(clientData[0], dataOnServer[0], false);
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, conflictResolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(1, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(dataOnServer[0], syncerStorage.getServerDataToStore()[0]);
		assertEquals(expectedClientResult, result.getChanges()[0]);
	}

	private void initConflictingObjectsWithSameID() {
		clientData = new DataObject[1];
		clientData[0] = contentType.newDataObject(session);
		clientData[0].setID("ID");
		dataOnServer = new DataObject[1];
		dataOnServer[0] = clientData[0].createCopy(false);
		dataOnServer[0].setFieldContent("Field3", true);
		dataOnServer[0].commitChanges();
		clientData[0].setFieldContent("Field3", false);
		contentType.setMatchRating(-1);
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerERROR() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerERROR(ConflictResolution.ERROR);
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerERROR_DELETE_OVER_CHANGE() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerERROR(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void conflictingDifferentObjectsOnClientAndOnServerERROR(ConflictResolution conflictResolution)
			throws USMException {
		initConflictingObjectWithDifferentIDs();
		try {
			slowContentSyncer.syncWithServer(syncerStorage, conflictResolution, log, 0, clientData);
			fail("Conflicting changes exception not thrown!");
		} catch (SynchronizationConflictException e) {

		}
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerCLIENT() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerCLIENT(ConflictResolution.USE_CLIENT);
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerCLIENT_DELETE_OVER_CHANGE() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerCLIENT(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void conflictingDifferentObjectsOnClientAndOnServerCLIENT(ConflictResolution conflictResolution)
			throws USMException {
		initConflictingObjectWithDifferentIDs();
		handler._writeUpdateToCall = true;
		DataObject serverChange = DataObjectUtil.copyAndModify(dataOnServer[0], clientData[0], false);
		serverChange.setID(dataOnServer[0].getID());
		DataObject expectedClientChange = clientData[0].createCopy(false);
		expectedClientChange.commitChanges();
		expectedClientChange.setID(dataOnServer[0].getID());
		DataObject expectedStorageResult = expectedClientChange.createCopy(false);
		expectedStorageResult.commitChanges();
		handler._objectToSend = serverChange;
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, conflictResolution, log, 0, clientData);
		handler.checkCalls(1);
		assertEquals(1, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(expectedStorageResult, syncerStorage.getServerDataToStore()[0]);
		assertEquals(expectedClientChange, result.getChanges()[0]);
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerSERVER() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerSERVER(ConflictResolution.USE_SERVER);
	}

	public void testConflictingDifferentObjectsOnClientAndOnServerSERVER_DELETE_OVER_CHANGE() throws USMException {
		conflictingDifferentObjectsOnClientAndOnServerSERVER(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	private void conflictingDifferentObjectsOnClientAndOnServerSERVER(ConflictResolution conflictResolution)
			throws USMException {
		initConflictingObjectWithDifferentIDs();
		DataObject expectedClientResult = DataObjectUtil.copyAndModify(clientData[0], dataOnServer[0], false);
		SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, conflictResolution, log, 0, clientData);
		handler.checkCalls(0);
		assertEquals(1, result.getChanges().length);
		assertEquals(1, syncerStorage.getServerDataToStore().length);
		assertEquals(dataOnServer[0], syncerStorage.getServerDataToStore()[0]);
		assertEquals(expectedClientResult, result.getChanges()[0]);
	}

	private void initConflictingObjectWithDifferentIDs() {
		clientData = new DataObject[1];
		clientData[0] = contentType.newDataObject(session);
		clientData[0].setID("clientID");
		clientData[0].setFieldContent("Field3", false);
		dataOnServer = new DataObject[1];
		dataOnServer[0] = contentType.newDataObject(session);
		dataOnServer[0].setID("serverID");
		dataOnServer[0].setFieldContent("Field3", true);
		dataOnServer[0].commitChanges();
		contentType.setMatchRating(1);
		syncerStorage.setCurrentServerData(dataOnServer);
		handler.resetServerCalls();
	}

	public void testLimitedSlowSyncWithManyChanges() throws USMException, InterruptedException {
		contentType.setMatchRating(0);
		clientData = new DataObject[0];
		dataOnServer = new DataObject[MANY_DATAOBJECTS_COUNT];
		for (int i = 0; i < dataOnServer.length; i++) {
			DataObject o = contentType.newDataObject(session);
			o.setID("serverID" + i);
			o.setFieldContent("Field3", true);
			o.commitChanges();
			dataOnServer[i] = o;
		}
		syncerStorage.setCurrentServerData(dataOnServer);
		for (int i = 0; i < MANY_DATAOBJECTS_COUNT; i++) {
			SyncResult result = slowContentSyncer.syncWithServer(syncerStorage, ConflictResolution.USE_SERVER, log, 1,
					clientData);
			handler.checkCalls(0);
			assertEquals(i < MANY_DATAOBJECTS_COUNT - 1, result.isIncomplete());
			assertEquals(1, result.getChanges().length);
			DataObject[] newServerData = syncerStorage.getServerDataToStore();
			assertEquals(i + 1, newServerData.length);
			DataObject clientResult = result.getChanges()[0];
			boolean found = false;
			for (int j = 0; j < dataOnServer.length; j++) {
				if (dataOnServer[j].getID().equals(clientResult.getID())) {
					DataObject expectedClientResult = dataOnServer[j].createCopy(false);
					expectedClientResult.setChangeState(ChangeState.CREATED);
					assertEquals(expectedClientResult, clientResult);
					found = true;
					break;
				}
			}
			assertTrue(i + ": Missing synced element " + clientResult, found);
			DataObject newObject = clientResult.createCopy(false);
			newObject.commitChanges();
			findAndCheckForEquality(i, newObject, newServerData);
			for (int j = 0; j < clientData.length; j++)
				findAndCheckForEquality(i, clientData[j], newServerData);
			clientData = newServerData;
		}
	}

	private void findAndCheckForEquality(int loop, DataObject object, DataObject[] data) {
		for (int i = 0; i < data.length; i++) {
			if (object.getID().equals(data[i].getID())) {
				assertEquals(object, data[i]);
				return;
			}
		}
		fail(loop + ": Missing element " + object);
	}
}
