/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.sync;

import java.util.*;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.session.dataobject.*;
import com.openexchange.usm.session.impl.SessionManagerImpl;

public class ContentSyncerSupport {
	private static final String ID = "id";

	private static final boolean READ_SERVER_CHANGES_ON_CREATION = true;

	private static interface UpdateHandler {
		void update(DataObject update, DataObject serverObject, boolean isFolderUpdate) throws USMException;
	}

	public static SyncResult storeDataObjectsAndBuildClientResult(boolean incomplete,
			ContentSyncerStorage serverInterface, DataObjectSet serverDataObjects, List<DataObject> clientResults,
			Map<DataObject, USMException> errorMap, DataObject... extraDataObjects) throws DatabaseAccessException,
			USMSQLException {
		long mt = computeTimestamp(serverDataObjects);
		createMissingUUIDs(serverDataObjects);
		mt = serverInterface.storeServerData(mt, serverDataObjects, incomplete);
		setTimestampAndCommitChanges(mt, serverDataObjects);
		DataObject[] dataToReturn = serverDataObjects.toArray();
		for (DataObject f : clientResults)
			f.setTimestamp(mt);
		for (DataObject f : extraDataObjects)
			f.setTimestamp(mt);
		return new SyncResult(incomplete, mt, dataToReturn, errorMap, DataObjectUtil.toArray(clientResults));
	}

	private static void createMissingUUIDs(Collection<DataObject> serverDataObjects) throws DatabaseAccessException,
			USMSQLException {
		for (DataObject o : serverDataObjects) {
			if (o.getUUID() == null) {
				throw new IllegalStateException("DataObject without UUID");
			}
		}
	}

	public static long updateTimestampAndCommitChanges(Collection<DataObject> serverDataObjects)
			throws DatabaseAccessException, USMSQLException {
		long timestamp = computeTimestamp(serverDataObjects);
		createMissingUUIDs(serverDataObjects);
		setTimestampAndCommitChanges(timestamp, serverDataObjects);
		return timestamp;
	}

	public static void setTimestampAndCommitChanges(long t, Collection<DataObject> serverDataObjects) {
		for (DataObject f : serverDataObjects) {
			f.setTimestamp(t);
			f.commitChanges();
		}
	}

	private static long computeTimestamp(Collection<DataObject> serverDataObjects) {
		long mt = 0;
		for (DataObject f : serverDataObjects)
			mt = Math.max(mt, f.getTimestamp());
		// If we have no objects and therefore no timestamp, use current system time as timestamp
		if (mt == 0)
			mt = System.currentTimeMillis();
		return mt;
	}

	public static void executeServerUpdates(SessionManagerImpl sessionManager, DataObjectSet serverObjects,
			List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions,
			Map<DataObject, USMException> resultMap) {
		try {
			executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap,
					false, null);
		} catch (ConflictingChangeException e) {
			throw new IllegalStateException("Unexpected exception on update objects", e);
		}
	}

	public static void updateOnServer(SessionManagerImpl sessionManager, DataObjectSet serverObjects,
			List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions,
			Map<DataObject, USMException> resultMap, List<DataObject> serverChanges) throws ConflictingChangeException {
		executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap, true,
				serverChanges);
	}

	private static void executeServerUpdatesInternal(final SessionManagerImpl sessionManager,
			final DataObjectSet serverObjects, List<DataObject> creations, List<DataObject> changes,
			List<DataObject> deletions, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions,
			final List<DataObject> serverChanges) throws ConflictingChangeException {
		final Map<String, String> folderIdMapping = new HashMap<String, String>();
		performUpdates(serverObjects, changes, resultMap, throwConflictExceptions, new UpdateHandler() {
			public void update(DataObject change, DataObject serverObject, boolean isFolder) throws USMException {
				if (folderIdMapping.containsKey(change.getOriginalParentFolderID())
						|| folderIdMapping.containsKey(change.getOriginalID())) {
					DataObject serverObjectCopy = serverObject.createCopy(true);
					if (folderIdMapping.containsKey(change.getOriginalParentFolderID()))
						serverObjectCopy.setParentFolderID(folderIdMapping.get(change.getOriginalParentFolderID()));
					if (folderIdMapping.containsKey(change.getOriginalID()))
						serverObjectCopy.setID(folderIdMapping.get(change.getOriginalID()));
					serverObjectCopy.commitChanges();
					DataObjectUtil.copyModifications(serverObjectCopy, change);
					serverObjectCopy.getContentType().getTransferHandler().writeUpdatedDataObject(serverObjectCopy,
							serverObjectCopy.getTimestamp());
				} else {
					change.getContentType().getTransferHandler().writeUpdatedDataObject(change, change.getTimestamp());
				}
				if (serverObject != null) {
					DataObjectUtil.copyModifications(serverObject, change);
					serverObject.commitChanges();
				}
				if (isFolder && change.isFieldModified(change.getFieldIndex(ID))) {
					//this means that we have renamed a mail folder and the id is changed 
					folderIdMapping.put(change.getOriginalID(), serverObject.getID()); //old to new
					Session session = serverObject.getSession();
					session.remapCachedData(session.getShortFolderID(change.getOriginalID()), session
							.getShortFolderID(serverObject.getID()));
					for (DataObject dataObject : serverObjects) {
						if (dataObject.getID().startsWith(change.getOriginalID() + "/")) {
							String newParentFolderID = serverObject.getID();
							int beginIndex = change.getOriginalID().length() + 1;
							if (!serverObject.getID().equals(dataObject.getParentFolderID()))
								newParentFolderID += "/" + dataObject.getParentFolderID().substring(beginIndex);
							String newID = serverObject.getID() + "/" + dataObject.getID().substring(beginIndex);
							folderIdMapping.put(dataObject.getOriginalID(), newID); //old to new
							dataObject.setParentFolderID(newParentFolderID);
							dataObject.setID(newID);
							session = dataObject.getSession();
							session.remapCachedData(session.getShortFolderID(dataObject.getOriginalID()), session
									.getShortFolderID(newID));
							dataObject.commitChanges();
						}
					}
				}
			}
		});
		performUpdates(null, creations, resultMap, throwConflictExceptions, new UpdateHandler() {
			private DataObjectSet _newestCachedData = null;

			public void update(DataObject creation, DataObject serverObject, boolean isFolder) throws USMException {
				Folder f = creation.getParentFolder();
				if (f != null)
					creation.setParentFolder(f);
				ContentType contentType = creation.getContentType();
				contentType.getTransferHandler().writeNewDataObject(creation);
				DataObject creationCopy = creation.createCopy(true);
				if (READ_SERVER_CHANGES_ON_CREATION) {
					DataObject creationCopy2 = creation.createCopy(true);
					creationCopy2.commitChanges();
					contentType.getTransferHandler().readDataObject(creationCopy2,
							creationCopy2.getSession().getFieldFilter(contentType));
					if (creationCopy2.isModified() && serverChanges != null)
						serverChanges.add(creationCopy2);
				}
				serverObjects.add(creationCopy);
				if (creation.getUUID() == null) {
					_newestCachedData = sessionManager.insertStoredUUID(creation, null, _newestCachedData);
					if (creation.getUUID() == null)
						creation.setUUID(UUID.randomUUID());
				} else {
					sessionManager.storeUUID(creation);
				}
			}

		});
		performUpdates(serverObjects, deletions, resultMap, throwConflictExceptions, new UpdateHandler() {
			public void update(DataObject deletion, DataObject serverObject, boolean isFolder) throws USMException {
				boolean executeDeletion = true;
				if (isFolder && serverObject != null) {
					Folder f = serverObject.getParentFolder();
					executeDeletion = f == null || serverObjects.contains(f);
				}
				if (executeDeletion)
					deletion.getContentType().getTransferHandler().writeDeletedDataObject(deletion);
				if (serverObject != null)
					serverObjects.remove(serverObject);
			}
		});
	}

	private static void performUpdates(DataObjectSet serverObjects, List<DataObject> updates,
			Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler)
			throws ConflictingChangeException {
		if (updates == null || updates.isEmpty())
			return;
		if (performFolderUpdates(serverObjects, updates, resultMap, throwConflictExceptions, handler))
			return;
		for (DataObject update : updates)
			performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, update, false);
	}

	private static boolean performFolderUpdates(DataObjectSet serverObjects, List<DataObject> updates,
			Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler)
			throws ConflictingChangeException {
		if (updates.get(0).getContentType().getCode() != DefaultContentTypes.FOLDER_CODE)
			return false;
		int size = updates.size();
		Folder[] folders = new Folder[size];
		for (int i = 0; i < size; i++) {
			DataObject o = updates.get(i);
			if (!(o instanceof Folder))
				return false;
			folders[i] = (Folder) o;
		}
		Comparator<Folder> comparator = new FolderHierarchyComparator(folders);
		Arrays.sort(folders, comparator);
		for (Folder f : folders)
			performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, f, true);
		return true;
	}

	private static void performOneUpdate(DataObjectSet serverObjects, Map<DataObject, USMException> resultMap,
			boolean throwConflictExceptions, UpdateHandler handler, DataObject update, boolean isFolderUpdate)
			throws ConflictingChangeException {
		try {
			DataObject serverObject = null;
			if (serverObjects != null) {
				serverObject = serverObjects.get(update.getID());
				if (serverObject != null)
					update.setTimestamp(serverObject.getTimestamp());
			}
			handler.update(update, serverObject, isFolderUpdate);
		} catch (ConflictingChangeException e) {
			if (throwConflictExceptions)
				throw e;
			resultMap.put(update, e);
		} catch (USMException e) {
			resultMap.put(update, e);
		}
	}
}
